//-----------------------------------------------------------------------------
// F930_FlashPrimitives.c
//-----------------------------------------------------------------------------
// Copyright 2012 Silicon Laboratories, Inc.
//
// This program contains several useful utilities for writing and updating
// FLASH memory.
//
// Target:         C8051F930
// Tool chain:     Raisonance / Keil / SDCC
// Command Line:   None
//
// Release 1.0 / 15 JUN 2012 (ST)
//    -Initial Revision
//


//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include <si_toolchain.h>
#include <c8051F930_defs.h>
#include "F930_FlashPrimitives.h"

//-----------------------------------------------------------------------------
// Structures, Unions, Enumerations, and Type Definitions
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Global Constants
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Function Prototypes
//-----------------------------------------------------------------------------

// FLASH read/write/erase routines
uint8_t   FLASH_ByteWrite (FLADDR addr, uint8_t byte, bit SFLE);
uint8_t   FLASH_ByteRead  (FLADDR addr, bit SFLE);
uint8_t   FLASH_PageErase (FLADDR addr, bit SFLE);

//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// FLASH Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// FLASH_ByteWrite
//-----------------------------------------------------------------------------
//
// Return Value :
//   0 if Write unsuccessful, 1 if Write successful
//
// Parameters   :
//   1) FLADDR addr - address of the byte to write to
//                    valid range is 0x0000 to 0xFBFF for 64K Flash devices
//                    valid range is 0x0000 to 0x7FFF for 32K Flash devices
//   2) uint8_t byte - byte to write to Flash.
//
// This routine writes <byte> to the linear FLASH address <addr>.
//
// This routine conforms to the recommendations in the C8051F93x/92x data sheet
// 
// If the MCU is operating from the internal voltage regulator, the VDD
// monitor should be set threshold and enabled as a reset source only when
// writing or erasing Flash. Otherwise, it should be set to the low threshold.
//
// If the MCU is operating from an external voltage regulator powering VDD
// directly, the VDD monitor can be set to the high threshold permanently.
//-----------------------------------------------------------------------------
uint8_t FLASH_ByteWrite (FLADDR addr, uint8_t byte, bit SFLE)
{
   int8_t SFRPAGE_SAVE = SFRPAGE;          // Preserve SFRPAGE
   bit EA_SAVE = EA;                   // Preserve EA
   int8_t xdata * data pwrite;             // FLASH write pointer
   uint8_t i;

   EA = 0;                             // Disable interrupts

   SFRPAGE = LEGACY_PAGE;

   RSTSRC = 0x00;                      // 1. Disable VDD monitor as a reset source

   VDM0CN = 0xA0;                      // 2. Enable VDD monitor and high threshold

   for (i = 0; i < 255; i++) {}        // 3. Wait for VDD monitor to stabilize

   if (!(VDM0CN & 0x40))               // 4. If the VDD voltage is not high
      return 0;                        //    enough don't write to Flash

   RSTSRC = 0x02;                      // 5. Safe to enable VDD Monitor as a 
                                       //    reset source

   pwrite = (uint8_t xdata *) addr;
                                       // 6. Enable Flash Writes

   FLKEY  = 0xA5;                      // Key Sequence 1
   FLKEY  = 0xF1;                      // Key Sequence 2
   PSCTL |= 0x01;                      // PSWE = 1

   if (SFLE) {
      PSCTL |= 0x04;                   // Set SFLE
   }

   VDM0CN = 0xA0;                      // 7. Enable VDD monitor and high threshold

   RSTSRC = 0x02;                      // 8. Enable VDD monitor as a reset source

   *pwrite = byte;                     // 9. Write the byte

   RSTSRC = 0x00;                      // 10. Disable the VDD monitor as reset 
                                       //     source
   VDM0CN = 0x80;                      // 11. Change VDD Monitor to low threshold
   RSTSRC = 0x02;                      // 12. Re-enable the VDD monitor as a 
                                       //     reset source
   if (SFLE) {
      PSCTL &= ~0x04;                  // Clear SFLE
   }
   PSCTL &= ~0x01;                     // PSWE = 0

   SFRPAGE = SFRPAGE_SAVE;             // Restore SFRPAGE
   EA      = EA_SAVE;                  // Restore interrupts

   return 1;
}

//-----------------------------------------------------------------------------
// FLASH_ByteRead
//-----------------------------------------------------------------------------
//
// This routine reads a <byte> from the linear FLASH address <addr>.
//
//-----------------------------------------------------------------------------
uint8_t FLASH_ByteRead (FLADDR addr, bit SFLE)
{
   int8_t SFRPAGE_SAVE = SFRPAGE;          // Preserve SFRPAGE
   bit EA_SAVE = EA;                   // Preserve EA
   int8_t code * data pread;               // FLASH read pointer
   uint8_t byte;

   EA = 0;                             // Disable interrupts

   pread = (uint8_t code *) addr;

   SFRPAGE = LEGACY_PAGE;

   if (SFLE) {
      PSCTL |= 0x04;                   // Set SFLE
   }

   byte = *pread;                      // Read the byte

   if (SFLE) {
      PSCTL &= ~0x04;                  // Clear SFLE
   }

   SFRPAGE = SFRPAGE_SAVE;             // Restore SFRPAGE
   EA = EA_SAVE;                       // Restore interrupts

   return byte;
}

//-----------------------------------------------------------------------------
// FLASH_PageErase
//-----------------------------------------------------------------------------
//
// This routine erases the FLASH page containing the linear FLASH address
// <addr>.
//
// This routine conforms to the recommendations in the C8051F93x/92x data sheet
// 
// If the MCU is operating from the internal voltage regulator, the VDD
// monitor should be set threshold and enabled as a reset source only when
// writing or erasing Flash. Otherwise, it should be set to the low threshold.
//
// If the MCU is operating from an external voltage regulator powering VDD
// directly, the VDD monitor can be set to the high threshold permanently.
//-----------------------------------------------------------------------------
uint8_t FLASH_PageErase (FLADDR addr, bit SFLE)
{
   int8_t SFRPAGE_SAVE = SFRPAGE;          // Preserve SFRPAGE
   bit EA_SAVE = EA;                   // Preserve EA
   int8_t xdata * data pwrite;             // FLASH write pointer
   uint8_t i;

   EA = 0;                             // Disable interrupts

   SFRPAGE = LEGACY_PAGE;

   RSTSRC = 0x00;                      // 1. Disable VDD monitor as a reset source

   VDM0CN = 0xA0;                      // 2. Enable VDD monitor and high threshold

   for (i = 0; i < 255; i++) {}        // 3. Wait for VDD monitor to stabilize

   if (!(VDM0CN & 0x40))               // 4. If the VDD voltage is not high enough
      return 0;                        //    don't attempt to write to Flash

   RSTSRC = 0x02;                      // 5. Safe to enable VDD Monitor as a reset 
                                       //    source

   pwrite = (uint8_t xdata *) addr;
                                       // 6. Enable Flash Writes

   FLKEY  = 0xA5;                      // Key Sequence 1
   FLKEY  = 0xF1;                      // Key Sequence 2
   PSCTL |= 0x03;                      // PSWE = 1; PSEE = 1

   if (SFLE) {
      PSCTL |= 0x04;                   // Set SFLE
   }

   VDM0CN = 0xA0;                      // 7. Enable VDD monitor and high threshold

   RSTSRC = 0x02;                      // 8. Enable VDD monitor as a reset source

   *pwrite = 0;                        // 9. Initiate page erase

   RSTSRC = 0x00;                      // 10. Disable the VDD monitor as a reset
                                       //     source
   VDM0CN = 0x80;                      // 11. Change VDD Monitor to low threshold
   RSTSRC = 0x02;                      // 12. Re-enable the VDD monitor as a reset 

   if (SFLE) {
      PSCTL &= ~0x04;                  // Clear SFLE
   }

   PSCTL &= ~0x03;                     // PSWE = 0; PSEE = 0

   SFRPAGE = SFRPAGE_SAVE;             // Restore SFRPAGE
   EA = EA_SAVE;                       // Restore interrupts

   return 1;
}


//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------