// USBAudioDevice.cpp: implementation of the CUSBAudioDevice class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "USBAudioDevice.h"
#include "math.h"


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CUSBAudioDevice::CUSBAudioDevice()
{
	//Initialize the critical section variable used for exclusivity
	InitializeCriticalSection(&gWaveCriticalSection);

	//Make the handles NULL to begin with
	m_USBAudioAudioHandle = NULL;
	m_USBAudioDataHandle = NULL;
	m_SoundCardHandle = NULL;

	//Set the input buffer pointers to NULL, and size to 0
	m_pEndpoint0ReportBuffer = NULL;
	m_pEndpoint1ReportBuffer = NULL;
	m_pEndpoint2ReportBuffer = NULL;
	m_Endpoint0ReportBufferSize = 0;
	m_Endpoint1ReportBufferSize = 0;
	m_Endpoint2ReportBufferSize = 0;

	//The device is not streaming or tuning initially
	m_Streaming = false;
	m_Tuning = false;

//	m_RDSCleared = false;

	//The current block starts at zero, and all blocks are initially free
	m_CurrentBlock = 0;
	m_FreeBlock = 0;
	gWaveFreeBlockCount = BLOCK_COUNT;

	//Allocate memory for the blocks of audio data to stream
	m_OutputHeader = AllocateBlocks(BLOCK_SIZE, BLOCK_COUNT);

	m_WaveformBuffer = (char*)malloc(BLOCK_SIZE);
	m_InputHeader.lpData = m_WaveformBuffer;
	m_InputHeader.dwBufferLength = BLOCK_SIZE;
	m_InputHeader.dwFlags = 0;

}

CUSBAudioDevice::~CUSBAudioDevice()
{
	//Free all allocated memory when destroyed
	FreeBlocks(m_OutputHeader);
	free(m_WaveformBuffer);
}

BYTE CUSBAudioDevice::OpenUSBAudio(AudioData* audioData)
{
	BYTE status = STATUS_ERROR;

	//Check that audiodata is not NULL
	if (audioData)
	{
		//Open the USB audio input device
		if (OpenUSBAudioAudio())
		{
			//Open the sound card
			if (OpenSoundCard())
			{
				status = STATUS_OK;
			}
			else
			{
				status = STATUS_OUTPUTAUDIO_ERROR;
			}
		}
		else
		{
			status = STATUS_USBAudioAUDIO_ERROR;
		}	
	}

	return status;
}

bool CUSBAudioDevice::CloseUSBAudio()
{
	bool status = false;
	
	//Close all pipes
	CloseUSBAudioAudio();
	CloseSoundCard();
	CloseUSBAudioData();
	
	status = true;

	return status;
}
bool CUSBAudioDevice::OpenUSBAudioAudio()
{
	bool status = false;

	WAVEINCAPS		waveInputCapabilities;
	WAVEFORMATEX	USBAudioWaveFormat;

	//Setup the wave format based on our defined audio data
	USBAudioWaveFormat.wFormatTag = WAVE_FORMAT_PCM;
	USBAudioWaveFormat.nSamplesPerSec = SAMPLES_PER_SECOND;
	USBAudioWaveFormat.wBitsPerSample = BITS_PER_SAMPLE;
	USBAudioWaveFormat.nChannels = CHANNELS;

	//Obtain the number of input devices on the system
	DWORD numWaveInputDevices = waveInGetNumDevs();

	//Scan through each input device to see if we can find our device
	for (DWORD i = 0; i < numWaveInputDevices; i++)
	{
		//Get the device capabilities of the currently indexed device
		if (waveInGetDevCaps(i, &waveInputCapabilities, sizeof(waveInputCapabilities)) == MMSYSERR_NOERROR)
		{
			//Compare the name of the device to see if it is the correct device
			if (!strcmp(waveInputCapabilities.szPname, "TONE GENERATOR") || !strcmp(waveInputCapabilities.szPname, "USB Audio Device"))
			{
				//When the device is found, open a handle to the device
				if (waveInOpen(&m_USBAudioAudioHandle, i, &USBAudioWaveFormat, 0, 0, CALLBACK_NULL) == MMSYSERR_NOERROR)
				{
					//Set i to the last item to kill the loop once the device is found
					i = numWaveInputDevices;

					status = true;
				}
			}
		}
	}

	return status;
}

bool CUSBAudioDevice::OpenSoundCard()
{
	bool status = false;

	WAVEFORMATEX	soundCardWaveFormat;

	//Setup the wave format based on our defined audio data
	soundCardWaveFormat.wFormatTag = WAVE_FORMAT_PCM;
	soundCardWaveFormat.nSamplesPerSec = SAMPLES_PER_SECOND;
	soundCardWaveFormat.wBitsPerSample = BITS_PER_SAMPLE;
	soundCardWaveFormat.nChannels = CHANNELS;

	//Open a handle to the default wave output device (sound card)
	if (waveOutOpen(&m_SoundCardHandle, WAVE_MAPPER, &soundCardWaveFormat, (DWORD_PTR)waveOutProc, (DWORD_PTR)&gWaveFreeBlockCount, CALLBACK_FUNCTION | WAVE_ALLOWSYNC) == MMSYSERR_NOERROR)
	{
		status = true;
	}

	return status;
}

void CUSBAudioDevice::InitializeStream()
{
	waveInStart(m_USBAudioAudioHandle);

	//Fill the audio buffer to initialize the stream
	while (gWaveFreeBlockCount > BUFFER_PADDING)
	{
		StreamAudioIn();
		Sleep(50);
	}

	//Set streaming to true
	m_Streaming = true;
}

void CUSBAudioDevice::StreamAudio()
{
	if (!m_Streaming)
	{
		In_Count = 0;
		Out_Count = 0;
		InitializeStream();
	}

		//Check that the handles arent NULL
		if ((m_USBAudioAudioHandle) && (m_SoundCardHandle))
		{
			//If there are any free blocks, then stream audio in
			if (gWaveFreeBlockCount)
			{
				if (StreamAudioIn())
				{
					In_Count++;
					TRACE(_T("StreamIn Success\n"));
				}
			}
			//If there are any blocks ready for output, then stream audio out
			if (gWaveFreeBlockCount < BLOCK_COUNT)
				if (StreamAudioOut())
				{
					Out_Count++;
					TRACE(_T("StreamOut Success\n"));
				}
		}
	
}

bool CUSBAudioDevice::IsStreaming()
{
	return m_Streaming;
}

bool CUSBAudioDevice::StreamAudioIn()
{
	bool status = false;

	//Prepare the header for streaming in
	if (waveInPrepareHeader(m_USBAudioAudioHandle, &m_OutputHeader[m_FreeBlock], sizeof(m_InputHeader)) == MMSYSERR_NOERROR)
	{
		//Get the buffer of audio in the input header
		if (waveInAddBuffer(m_USBAudioAudioHandle, &m_OutputHeader[m_FreeBlock], sizeof(m_InputHeader)) == MMSYSERR_NOERROR)
		{				
			//Enter the critical section to decrement the free block count
			EnterCriticalSection(&gWaveCriticalSection);
			gWaveFreeBlockCount--;
			LeaveCriticalSection(&gWaveCriticalSection);

			//Increment the free block index, and scale it
			m_FreeBlock++;
			m_FreeBlock %= BLOCK_COUNT;

			status = true;
		}
	}
	else
		TRACE(_T("Could not prepare header\n"));

	return status;
}

static void CALLBACK waveOutProc(HWAVEOUT hwo, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2)
{
	if (uMsg == WOM_DONE)
	{
		EnterCriticalSection(&gWaveCriticalSection);
		int* freeBlockCounter = (int*)dwInstance;
		(*freeBlockCounter)++;
		LeaveCriticalSection(&gWaveCriticalSection);
	}
}

bool CUSBAudioDevice::StreamAudioOut()
{
	bool status = false;

	//Prepare the header for streaming out
	if (waveOutPrepareHeader(m_SoundCardHandle, &m_OutputHeader[m_CurrentBlock], sizeof(WAVEHDR)) == MMSYSERR_NOERROR)
	{
		//Write the sound to the sound card
		if (waveOutWrite(m_SoundCardHandle, &m_OutputHeader[m_CurrentBlock], sizeof(WAVEHDR)) == MMSYSERR_NOERROR)
		{
			//waveOutProc callback function will get called, and free block counter gets incremented

			//Increment the index of the current block to be played
			m_CurrentBlock++;
			m_CurrentBlock %= BLOCK_COUNT;

			status = true;
		}
	}

	return status;
}

UINT CUSBAudioDevice::GetCurrentBlock(void)
{
	return m_CurrentBlock;
}
UINT CUSBAudioDevice::GetFreeBlock(void)
{
	return m_FreeBlock;
}

long CUSBAudioDevice::GetIn_Count(void)
{
	return In_Count;
}
long CUSBAudioDevice::GetOut_Count(void)
{
	return Out_Count;
}

BYTE CUSBAudioDevice::GetWaveOutVolume()
{
	DWORD level = VOLUME_MIN;

	//This gets the current wave output volume
	waveOutGetVolume(m_SoundCardHandle, &level);

	//This determines the level of one channel
	level = level & 0xFFFF;
	
	//If the level is above 0, then calculate it's percentage (0-100%)
	if (level)	
		level = (DWORD)((double)(level / (double)0xFFFF) * 100.0);

	//Return the percentage level of volume
	return (BYTE)(level & 0xFF);
}

bool CUSBAudioDevice::SetWaveOutVolume(BYTE level)
{
	bool status = false;
	DWORD setLevel = 0x00000000;

	//Don't set the volume to anything greater than the max level
	if (level > VOLUME_MAX)
		level = VOLUME_MAX;
	
	//Calculate a value based on the percentage input of one channel
	if (level)
		setLevel = (DWORD)(((double)level / 100.0) * (double)0xFFFF);

	//Set the volume for L and R channels
	setLevel = (setLevel << 16) | setLevel;

	//Set the volume
	if (waveOutSetVolume(m_SoundCardHandle, setLevel) == MMSYSERR_NOERROR)
		status = true;

	return status;
}

bool CUSBAudioDevice::CloseUSBAudioAudio()
{
	bool status = false;

	//Stop the input from the device
	waveInStop(m_USBAudioAudioHandle);

	//Reset the device
	waveInReset(m_USBAudioAudioHandle);

	//Close the device
	waveInClose(m_USBAudioAudioHandle);

	m_USBAudioAudioHandle = NULL;
	m_CurrentBlock = 0;
	m_FreeBlock = 0;
	gWaveFreeBlockCount = BLOCK_COUNT;
	m_Streaming = false;
	status = true;
			
	return status;
}

bool CUSBAudioDevice::CloseSoundCard()
{
	bool status = false;

	//Reset the device
	waveOutReset(m_SoundCardHandle);
	
	//Close the device
	waveOutClose(m_SoundCardHandle);

	m_SoundCardHandle = NULL;
	status = true;
				
	return status;
}

WAVEHDR* CUSBAudioDevice::AllocateBlocks(int size, int count)
{
	char* buffer;
	WAVEHDR* blocks;
	DWORD totalBufferSize = (size + sizeof(WAVEHDR)) * count;

	//Allocate zero initialized memory the size of our total buffer
	if (buffer = (char*)HeapAlloc(GetProcessHeap(), HEAP_ZERO_MEMORY, totalBufferSize))
	{
		blocks = (WAVEHDR*)buffer;
		buffer += sizeof(WAVEHDR) * count;

		//Fill the headers out based on our allocated space
		for (int i = 0; i < count; i++)
		{
			blocks[i].dwBufferLength = size;
			blocks[i].lpData = buffer;
			buffer += size;
		}
	}

	return blocks;
}

void CUSBAudioDevice::FreeBlocks(WAVEHDR* blockArray)
{
	//Free the heap memory from the pointer provided
	HeapFree(GetProcessHeap(), 0, blockArray);
}

bool CUSBAudioDevice::OpenUSBAudioData()
{
	bool status = false;

	HANDLE		hHidDeviceHandle = NULL;
	GUID		hidGuid;
	HDEVINFO	hHidDeviceInfo = NULL;

	//Obtain the HID GUID
	HidD_GetHidGuid(&hidGuid);

	//Use the HID GUID to get a handle to a list of all HID devices connected
	hHidDeviceInfo = SetupDiGetClassDevs(&hidGuid, NULL, NULL, DIGCF_PRESENT | DIGCF_DEVICEINTERFACE);

	if (hHidDeviceInfo != INVALID_HANDLE_VALUE)
	{
		SP_DEVICE_INTERFACE_DATA hidDeviceInterfaceData;
		hidDeviceInterfaceData.cbSize = sizeof(hidDeviceInterfaceData);

		DWORD i = 0;
		BOOL hidResult = 1;

		//Loop through devices until the hidResult fails, the max USB devices are reached, or status is true
		while ((hidResult) && (i < MAX_USB_DEVICES) && (!status))
		{
			//Query the device using the index to get the interface data
			hidResult = SetupDiEnumDeviceInterfaces(hHidDeviceInfo, 0, &hidGuid, i, &hidDeviceInterfaceData);

			//If a successful query was made, use it to get the detailed data of the device
			if (hidResult)
			{
				BOOL detailResult;
				DWORD length, required;
				PSP_DEVICE_INTERFACE_DETAIL_DATA hidDeviceInterfaceDetailData;

				//Obtain the length of the detailed data structure, then allocate space and retrieve it
				SetupDiGetDeviceInterfaceDetail(hHidDeviceInfo, &hidDeviceInterfaceData, NULL, 0, &length, NULL);
				hidDeviceInterfaceDetailData = (PSP_DEVICE_INTERFACE_DETAIL_DATA)malloc(length);
				hidDeviceInterfaceDetailData->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);
				detailResult = SetupDiGetDeviceInterfaceDetail(hHidDeviceInfo, &hidDeviceInterfaceData, hidDeviceInterfaceDetailData, length, &required, NULL);
				
				//If another successful query to the device detail was made, open a handle to
				//determine if the VID and PID are a match as well
				if (detailResult)
				{
					//Open the device
					hHidDeviceHandle = CreateFile(hidDeviceInterfaceDetailData->DevicePath, GENERIC_READ | GENERIC_WRITE, NULL, NULL, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
					
					if (hHidDeviceHandle != INVALID_HANDLE_VALUE)
					{
						HIDD_ATTRIBUTES	hidDeviceAttributes;

						//If it is a valid open, then get the attributes of the HID device
						if (HidD_GetAttributes(hHidDeviceHandle, &hidDeviceAttributes))
						{
							//Check that the VID and PID match
							if ((hidDeviceAttributes.VendorID == USBAudio_VID) && (hidDeviceAttributes.ProductID == USBAudio_PID))
							{
								m_USBAudioDataHandle = hHidDeviceHandle;

								PHIDP_PREPARSED_DATA preparsedData;

								//Get the preparsed data structure
								if (HidD_GetPreparsedData(hHidDeviceHandle, &preparsedData))
								{
									HIDP_CAPS capabilities;

									//Used the preparsed data structure to get the device capabilities
									if (HidP_GetCaps(preparsedData, &capabilities))
									{
										//Check that the feature report length is more than 2
										if (capabilities.FeatureReportByteLength > (USBAudio_REGISTER_NUM * USBAudio_REGISTER_SIZE))
										{
											//Allocate the right amount of space for the control feature reports (1-17)
											m_Endpoint0ReportBufferSize = capabilities.FeatureReportByteLength;
											m_pEndpoint0ReportBuffer = (BYTE*)malloc(m_Endpoint0ReportBufferSize);
											memset(m_pEndpoint0ReportBuffer, 0, m_Endpoint0ReportBufferSize);

											//Allocate the right amout of space for endpoint 1 feature report (18)
											m_Endpoint1ReportBufferSize = RDS_REPORT_SIZE;
											m_pEndpoint1ReportBuffer = (BYTE*)malloc(m_Endpoint1ReportBufferSize);
											memset(m_pEndpoint1ReportBuffer, 0, m_Endpoint1ReportBufferSize);

											//m_Endpoint2ReportBufferSize
											m_Endpoint2ReportBufferSize = 18;
											m_pEndpoint2ReportBuffer = (BYTE*)malloc(m_Endpoint2ReportBufferSize);
											memset(m_pEndpoint2ReportBuffer, 0, m_Endpoint2ReportBufferSize);
											/*
											if (ReadAllRegisters(m_Register))
											{
												ChangeLED(CONNECT_STATE);
												status = true;
											}
											*/
										}
									}

									HidD_FreePreparsedData(preparsedData);
								}
							}
							else
							{
								//If they dont match, close the handle and continue the search
								CloseHandle(hHidDeviceHandle);
							}
						}
					}
				}

				//Deallocate space for the detailed data structure
				free(hidDeviceInterfaceDetailData);
			}

			//Increment i for the next device
			i++;
		}
	}

	return status;
}


bool CUSBAudioDevice::CloseUSBAudioData()
{
	bool status = false;

	if (m_USBAudioDataHandle)
//		ChangeLED(DISCONNECT_STATE);

	//Free the endpoint buffers
	if (m_pEndpoint0ReportBuffer)
		free(m_pEndpoint0ReportBuffer);
	if (m_pEndpoint1ReportBuffer)
		free(m_pEndpoint1ReportBuffer);
	if (m_pEndpoint2ReportBuffer)
		free(m_pEndpoint2ReportBuffer);

	//Set the endpoint buffer sizes back to zero
	m_Endpoint0ReportBufferSize = 0;
	m_Endpoint1ReportBufferSize = 0;
	m_Endpoint2ReportBufferSize = 0;

	m_pEndpoint0ReportBuffer = NULL;
	m_pEndpoint1ReportBuffer = NULL;
	m_pEndpoint2ReportBuffer = NULL;

	//Close the USB Audio Device handle and make it NULL
	CloseHandle(m_USBAudioDataHandle);
	m_USBAudioDataHandle = NULL;

	status = true;

	return status;
}

bool CUSBAudioDevice::StopStream(bool stop)
{
	m_Tuning = stop;

	return true;
}
