/*
** ============================================================================
**
** FILE
**  main.c
**
** DESCRIPTION
**  This is the main file of the project.	    
**  HW platform SDB with C8051F93x Silabs MCU 
**
** CREATED
**  Silicon Laboratories Hungary Ltd
**
** COPYRIGHT
**  Copyright 2009 Silicon Laboratories, Inc.  
**	http://www.silabs.com
**
** ============================================================================
*/

								/* ======================================================== *
								 *									INCLUDE										*
							 	 * ======================================================== */


#include "C8051F930_defs.h"
#include "compiler_defs.h"
#include "string.h"

							/* ================================================================ *
							 * C8051F930 Pin definitions for Software Development Board         *
							 * 					(using compiler_def.h macros) 						  *
							 *	================================================================ */	

SBIT(NSS, SFR_P1, 3);
SBIT(NIRQ, SFR_P0, 6);
SBIT(PB1, SFR_P0, 0);
SBIT(PB2, SFR_P0, 1);
SBIT(PB3, SFR_P2, 0);
SBIT(PB4, SFR_P2, 1);
SBIT(LED1, SFR_P1, 4);
SBIT(LED2, SFR_P1, 5);
SBIT(LED3, SFR_P1, 6);
SBIT(LED4, SFR_P1, 7);

//One out of these definitions has to be uncommented which tells to the compiler what kind 
//of Testcard is plugged into the Software Development board
#define SEPARATE_RX_TX
//#define ANTENNA_DIVERSITY
//#define ONE_SMA_WITH_RF_SWITCH


								/* ======================================================== *
								 *							Function PROTOTYPES							* 
								 * ======================================================== */
//MCU initialization
void MCU_Init(void);
//SPI functions						
void SpiWriteRegister (U8, U8);
U8 SpiReadRegister (U8);


void main(void)
{
	U8 ItStatus1,ItStatus2;
	U16 delay;
	U8 length,temp8;
	U8 payload[10];

	//Initialize the MCU: 
	//	- set IO ports for the Software Development board
	//	- set MCU clock source
	//	- initialize the SPI port 
	//	- turn off LEDs
	MCU_Init();
							/* ======================================================== *
							 *						Initialize the Si4432 ISM chip				* 
							 * ======================================================== */
	
	//read interrupt status registers to clear the interrupt flags and release NIRQ pin
	ItStatus1 = SpiReadRegister(0x03);													//read the Interrupt Status1 register
	ItStatus2 = SpiReadRegister(0x04);													//read the Interrupt Status2 register
	
	//SW reset   
   SpiWriteRegister(0x07, 0x80);															//write 0x80 to the Operating & Function Control1 register 
	
	//wait for POR interrupt from the radio (while the nIRQ pin is high)
	while ( NIRQ == 1); 
	//read interrupt status registers to clear the interrupt flags and release NIRQ pin
	ItStatus1 = SpiReadRegister(0x03);													//read the Interrupt Status1 register
	ItStatus2 = SpiReadRegister(0x04);													//read the Interrupt Status2 register
 
	//wait for chip ready interrupt from the radio (while the nIRQ pin is high)
	while ( NIRQ == 1); 
	//read interrupt status registers to clear the interrupt flags and release NIRQ pin
	ItStatus1 = SpiReadRegister(0x03);													//read the Interrupt Status1 register
	ItStatus2 = SpiReadRegister(0x04);													//read the Interrupt Status2 register
					
							/*set the physical signal parameters*/
	//set the center frequency to 915 MHz
	SpiWriteRegister(0x75, 0x75);															//write 0x75 to the Frequency Band Select register             
  	SpiWriteRegister(0x76, 0xBB);															//write 0xBB to the Nominal Carrier Frequency1 register
  	SpiWriteRegister(0x77, 0x80); 														//write 0x80 to the Nominal Carrier Frequency0 register
	
							/*set the modem parameters according to the exel calculator(parameters: 9.6 kbps, deviation: 45 kHz, channel filter BW: 112.1 kHz*/
	SpiWriteRegister(0x1C, 0x05);															//write 0x05 to the IF Filter Bandwidth register		
	SpiWriteRegister(0x20, 0xA1);															//write 0xA1 to the Clock Recovery Oversampling Ratio register		
	SpiWriteRegister(0x21, 0x20);															//write 0x20 to the Clock Recovery Offset 2 register		
	SpiWriteRegister(0x22, 0x4E);															//write 0x4E to the Clock Recovery Offset 1 register		
	SpiWriteRegister(0x23, 0xA5);															//write 0xA5 to the Clock Recovery Offset 0 register		
	SpiWriteRegister(0x24, 0x00);															//write 0x00 to the Clock Recovery Timing Loop Gain 1 register		
	SpiWriteRegister(0x25, 0x13);															//write 0x13 to the Clock Recovery Timing Loop Gain 0 register		
	SpiWriteRegister(0x1D, 0x40);															//write 0x40 to the AFC Loop Gearshift Override register		
	SpiWriteRegister(0x72, 0x1F);															//write 0x1F to the Frequency Deviation register		
										

							/*Configure the receive packet handler*/
	//Disable header bytes; set variable packet length (the length of the payload is defined by the
	//received packet length field of the packet); set the synch word to two bytes long
   SpiWriteRegister(0x33, 0x02 );														//write 0x02 to the Header Control2 register    
	//Disable the receive header filters
   SpiWriteRegister(0x32, 0x00 );														//write 0x00 to the Header Control1 register            
	
	//Set the sync word pattern to 0x2DD4
	SpiWriteRegister(0x36, 0x2D);															//write 0x2D to the Sync Word 3 register
	SpiWriteRegister(0x37, 0xD4);															//write 0xD4 to the Sync Word 2 register

	//Enable the receive packet handler and CRC-16 (IBM) check
	SpiWriteRegister(0x30, 0x85);															//write 0x85 to the Data Access Control register
	//Enable FIFO mode and GFSK modulation
	SpiWriteRegister(0x71, 0x63);										 					//write 0x63 to the Modulation Mode Control 2 register
	//set preamble detection threshold to 20bits
	SpiWriteRegister(0x35, 0x28); 														//write 0x28 to the Preamble Detection Control  register
	//set the preamble length to 10bytes if the antenna diversity is used and set to 5bytes if not 
#ifdef ANTENNA_DIVERSITY
	SpiWriteRegister(0x34, 0x14); 														//write 0x14 to the Preamble Length register 	
#else	
	SpiWriteRegister(0x34, 0x0A);															//write 0x0A to the Preamble Length register
#endif

#ifdef ANTENNA_DIVERSITY
	//Enable antenna diversity mode
	SpiWriteRegister(0x08, 0x80);															//write 0x80 to the Operating Function Control 2 register
#endif

											/*set the GPIO's according the testcard type*/
#ifdef ANTENNA_DIVERSITY
  	SpiWriteRegister(0x0C, 0x17);															//write 0x17 to the GPIO1 Configuration(set the Antenna 1 Switch used for antenna diversity )
	SpiWriteRegister(0x0D, 0x18);															//write 0x18 to the GPIO2 Configuration(set the Antenna 2 Switch used for antenna diversity ) 
#endif
#ifdef ONE_SMA_WITH_RF_SWITCH
  	SpiWriteRegister(0x0C, 0x12);															//write 0x12 to the GPIO1 Configuration(set the TX state)
	SpiWriteRegister(0x0D, 0x15);															//write 0x15 to the GPIO2 Configuration(set the RX state) 
#endif

											/*set the non-default Si4432 registers*/
	//set the VCO and PLL
	SpiWriteRegister(0x5A, 0x7F); 														//write 0x7F to the VCO Current Trimming register 
	SpiWriteRegister(0x58, 0x80);															//write 0x80 to the ChargepumpCurrentTrimmingOverride register
	SpiWriteRegister(0x59, 0x40);															//write 0x40 to the Divider Current Trimming register 	
	//set the AGC
	SpiWriteRegister(0x6A, 0x0B);															//write 0x0B to the AGC Override 2 register
	//set ADC reference voltage to 0.9V
	SpiWriteRegister(0x68, 0x04);															//write 0x04 to the Deltasigma ADC Tuning 2 register
  	SpiWriteRegister(0x1F, 0x03);															//write 0x03 to the Clock Recovery Gearshift Override register
	
	//set  Crystal Oscillator Load Capacitance register
	SpiWriteRegister(0x09, 0xD7);															//write 0xD7 to the Crystal Oscillator Load Capacitance register

    
	/*enable receiver chain*/
	SpiWriteRegister(0x07, 0x05);															//write 0x05 to the Operating Function Control 1 register
	// a) one which shows that a valid packet received: 'ipkval'
	// b) second shows if the packet received with incorrect CRC: 'icrcerror' 
	SpiWriteRegister(0x05, 0x03); 														//write 0x03 to the Interrupt Enable 1 register
	SpiWriteRegister(0x06, 0x00); 														//write 0x00 to the Interrupt Enable 2 register
	//read interrupt status registers to release all pending interrupts
	ItStatus1 = SpiReadRegister(0x03);													//read the Interrupt Status1 register
	ItStatus2 = SpiReadRegister(0x04);													//read the Interrupt Status2 register

	/*MAIN Loop*/
	while(1)
	{
	    //wait for the interrupt event
		if( NIRQ == 0 )
		{
			//disable the receiver chain 
			SpiWriteRegister(0x07, 0x01);													//write 0x01 to the Operating Function Control 1 register
			//read interrupt status registers 		
			ItStatus1 = SpiReadRegister(0x03);											//read the Interrupt Status1 register
			ItStatus2 = SpiReadRegister(0x04);											//read the Interrupt Status2 register

			/*CRC Error interrupt occured*/
			if( (ItStatus1 & 0x01) == 0x01 )
			{
				//reset the RX FIFO
       	   SpiWriteRegister(0x08, 0x02);												//write 0x02 to the Operating Function Control 2 register
        	   SpiWriteRegister(0x08, 0x00);												//write 0x00 to the Operating Function Control 2 register
				//blink all LEDs to show the error
				LED1 = 1;
				LED2 = 1;
				LED3 = 1;
				LED4 = 1;
				for(delay = 0; delay < 10000;delay++);
				LED1 = 0;
				LED2 = 0;
				LED3 = 0;
				LED4 = 0;
			}

			/*packet received interrupt occured*/
			if( (ItStatus1 & 0x02) == 0x02 )
			{
 				//Read the length of the received payload
				length = SpiReadRegister(0x4B);											//read the Received Packet Length register
				//check whether the received payload is not longer than the allocated buffer in the MCU
				if(length < 11)
				{
					//Get the reeived payload from the RX FIFO
					for(temp8=0;temp8 < length;temp8++)
					{
						payload[temp8] = SpiReadRegister(0x7F);						//read the FIFO Access register
					}
			
					//check whether the content of the packet is what the demo expects
					if( length == 8 ) 
	            	{
	            		if( memcmp(&payload[0], "BUTTON1", 7) == 0 )
					   	{
	                  		//blink LED1 to show that the packet received
	     					LED1 = 1;
							for(delay = 0; delay < 10000;delay++);
	  		   				LED1 = 0;
	               		}
						if( memcmp(&payload[0], "BUTTON2", 7) == 0 )
					   	{
	                  	//blink LED2 to show that the packet received
	     					LED2 = 1;
							for(delay = 0; delay < 10000;delay++);
	  		   				LED2 = 0;
	               		}
						if( memcmp(&payload[0], "BUTTON3", 7) == 0 )
					   	{
	                  	//blink LED3 to show that the packet received
	     					LED3 = 1;
							for(delay = 0; delay < 10000;delay++);
	  		   				LED3 = 0;
	               		}
						if( memcmp(&payload[0], "BUTTON4", 7) == 0 )
					   	{
	                  	//blink LED4 to show that the packet received
	     					LED4 = 1;
							for(delay = 0; delay < 10000;delay++);
	  		   				LED4 = 0;
	               		}
					}
				}	
				//reset the RX FIFO
       	   SpiWriteRegister(0x08, 0x02);												//write 0x02 to the Operating Function Control 2 register
        	   SpiWriteRegister(0x08, 0x00);												//write 0x00 to the Operating Function Control 2 register
				//enable the receiver chain again
				SpiWriteRegister(0x07, 0x05);												//write 0x05 to the Operating Function Control 1 register
			}
		} 	
	}
}

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  	void MCU_Init(void)
  +
  + DESCRIPTION:   	This function configures the MCU 
  + 						P0.0  -  skipped, PB0
  + 						P0.1  -  skipped  PB1
  + 						P0.2  -  skipped, analog input, XTAL
  + 						P0.3  -  skipped, analog input, XTAL
  + 						P0.4  -  UART TX, push-pull output
  + 						P0.5  -  UART RX, open drain input
  + 						P0.6  -  skipped
  + 						P0.7  -  skipped
  +
  + 						P1.0  -  SCK  (SPI1), Push-Pull,  Digital
  + 						P1.1  -  MISO (SPI1), Open-Drain, Digital
  + 						P1.2  -  MOSI (SPI1), Push-Pull,  Digital
  + 						P1.3  -  NSS, Push-Pull,  Digital
  + 						P1.4  -  skipped LED1 
  + 						P1.5  -  skipped LED
  + 						P1.6  -  skipped LED
  + 						P1.7  -  skipped LED
  +	
  + 						P2.0  -  skipped, PB3
  + 						P2.1  -  skipped  PB4
  +
  + INPUT:			  None
  +
  + RETURN:         None
  +
  + NOTES:          None
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void MCU_Init(void)
{
	//Disable the Watch Dog timer of the MCU
   PCA0MD   &= ~0x40;                  
	
	// Set the clock source of the MCU: 10MHz, using the internal RC osc.
   CLKSEL    = 0x14;

   // Initialize the the IO ports and the cross bar
   P0SKIP  |= 0xCF;                    // skip P0.0-3 & 0.6-7
   XBR1    |= 0x40;                    // Enable SPI1 (3 wire mode)
   P1MDOUT |= 0x01;                    // Enable SCK push pull
   P1MDOUT |= 0x04;                    // Enable MOSI push pull
   P1SKIP  |= 0x08;                    // skip NSS
   P1MDOUT |= 0x08;                    // Enable NSS push pull
   P1SKIP  |= 0xF0;                    // skip LEDs
   P1MDOUT |= 0xF0;                    // Enable LEDS push pull
   P2SKIP  |= 0x03;                    // skip PB3 & 4
	P2 = 0x03;
	SFRPAGE = CONFIG_PAGE;
   P1DRV   = 0xFD;		               // MOSI, SCK, NSS, LEDs high current mode
   SFRPAGE = LEGACY_PAGE;
	XBR2    |= 0x40;                    // enable Crossbar

   // For the SPI communication the hardware peripheral of the MCU is used 
   //in 3 wires Single Master Mode. The select pin of the radio is controlled
	//from software
	SPI1CFG   = 0x40;							// Master SPI, CKPHA=0, CKPOL=0
   SPI1CN    = 0x00;							// 3-wire Single Master, SPI enabled
   SPI1CKR   = 0x00;
   SPI1EN 	 = 1;                     	// Enable SPI interrupt
   NSS = 1;
	
	// Turn off the LEDs
   LED1 = 0;
   LED2 = 0;
   LED3 = 0;
   LED4 = 0;
}

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME: void SpiWriteRegister(U8 reg, U8 value)
  +
  + DESCRIPTION:   This function writes the registers 
  + 					
  + INPUT:			 U8 reg - register address   
  +					 U8 value - value write to register	
  +
  + RETURN:        None
  +
  + NOTES:         Write uses a Double buffered transfer
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void SpiWriteRegister (U8 reg, U8 value)
{
   // Send SPI data using double buffered write

	//Select the radio by pulling the nSEL pin to low
   NSS = 0;                            
   
   //write the address of the register into the SPI buffer of the MCU
	//(important to set the MSB bit)
	SPI1DAT = (reg|0x80);				//write data into the SPI register
	//wait until the MCU finishes sending the byte
	while( SPIF1 == 0);					
	SPIF1 = 0;
	//write the new value of the radio register into the SPI buffer of the MCU
	SPI1DAT = value;					//write data into the SPI register
	//wait until the MCU finishes sending the byte
	while( SPIF1 == 0);					//wait for sending the data
	SPIF1 = 0;	       
   //Deselect the radio by pulling high the nSEL pin
	NSS = 1;                            

}

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME: U8 SpiReadRegister(U8 reg)
  +
  + DESCRIPTION:   This function reads the registers 
  + 					
  + INPUT:			 U8 reg - register address   
  +
  + RETURN:        SPI1DAT - the register content 
  +
  + NOTES:         none
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
U8 SpiReadRegister (U8 reg)
{

	//Select the radio by pulling the nSEL pin to low
	NSS = 0;                            
	//Write the address of the register into the SPI buffer of the MCU 
	//(important to clear the MSB bit)
   SPI1DAT = reg;								//write data into the SPI register
	//Wait untill the MCU finishes sending the byte
	while( SPIF1 == 0);						
	SPIF1 = 0;
	//Write a dummy data byte into the SPI buffer of the MCU. During sending 
	//this byte the MCU will read the value of the radio register and save it 
	//in its SPI buffer.
	SPI1DAT = 0xFF;							//write dummy data into the SPI register
	//Wait untill the MCU finishes sending the byte
	while( SPIF1 == 0);						
	SPIF1 = 0;
	//Deselect the radio by pulling high the nSEL pin
	NSS = 1;                            

	//Read the received radio register value and return with it
   return SPI1DAT;
}
