//-----------------------------------------------------------------------------
// Fxxx_EEPROM_Interface.c
//-----------------------------------------------------------------------------
// Copyright (C) 2010 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// File Description:
//
// This file defines an emulated EEPROM interface for Silicon Labs Flash-based
// 8051 MCUs.
//
// Release 1.0 / 10NOV2010 (BD)
//    -Initial Revision
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <si_toolchain.h>
#include "Fxxx_Flash_Interface.h"
#include "Fxxx_EEPROM_Configuration.h"

//-----------------------------------------------------------------------------
// Internal Constants
//-----------------------------------------------------------------------------

#define NEW_TAG         0xFF        // Value for New Sector
#define WIP_TAG         0x7F        // Value for WIP Sector

#define DONE_TAG_MIN    0x01        // Done tag minimum
#define DONE_TAG_MAX    0x7E        // Done tag maximum

#define NO_TAG          0x00

//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------
uint8_t currentEepromSector = 0, currentEepromPage = 0;

//-----------------------------------------------------------------------------
// Function Prototypes
//-----------------------------------------------------------------------------
void copySector(uint16_t, uint16_t, uint8_t, uint8_t);
uint16_t getBaseAddress(uint8_t, uint8_t);
uint8_t findNextSector(void);
uint8_t findCurrentSector(void);

uint8_t EEPROM_WriteBlock (uint8_t, uint8_t *, uint8_t);
uint8_t EEPROM_ReadBlock (uint8_t, uint8_t *, uint8_t);

//-----------------------------------------------------------------------------
// EEPROM_WriteBlock
//-----------------------------------------------------------------------------
//
// This routine writes the specified array into the emulated EEPROM by
// decoding the address and calling flash write functions.
//
// Arguments :  uint8_t address - 8-bit EEPROM address to start at
//              uint8_t *dataBuf - pointer to data array
//              uint8_t length - number of bytes to write
//
// Returns uint8_t : EE_NO_ERROR = Write was sucessful
//              EE_WRITE_ERROR = Write did not succeed, due to either a page
//                erase problem or an out-of-bounds address.
//              EE_SECTOR_ERROR = Could not determine current sector.
//
uint8_t EEPROM_WriteBlock (uint8_t address, uint8_t *dataBuf, uint8_t length)
{
   uint16_t readAddr, writeAddr;
   uint8_t nextSector;
   uint8_t currentTag;
   uint8_t byteCount;
   uint8_t pageEraseCounter;

   if ((uint16_t)(address+length) > EE_SIZE)
   {
      return EE_WRITE_ERROR;
   }

   currentTag = findCurrentSector();
   if (currentTag == NO_TAG)
   {
      return EE_SECTOR_ERROR;
   }

   // Get the current sector's base address for read (copy) later
   readAddr = getBaseAddress(currentEepromPage, currentEepromSector);

   nextSector = findNextSector();

   // Get base address of sector where we will be writing
   writeAddr = getBaseAddress(currentEepromPage, currentEepromSector);

   if (nextSector == 0x00)
   {
      pageEraseCounter = 0;

      while (pageEraseCounter < FL_ERASE_LIMIT)
      {
         FLASH_WriteErase (writeAddr, 0xFF, FL_ERASE);
         if (!FLASH_BlankCheck(writeAddr))
         {
            if (++pageEraseCounter == FL_ERASE_LIMIT)
            {
               return EE_WRITE_ERROR;
            }
         }
         else
         {
            break;
         }
      }
   }

   // Mark new sector as copy in progress
   FLASH_WriteErase (writeAddr + TAG_OFFSET, WIP_TAG, FL_WRITE);

   // Copy from old to new sector excluding copy addresses
   copySector(readAddr, writeAddr, address, length);

   // Write new info to sector
   for (byteCount = 0; byteCount < length; byteCount++)
   {
      FLASH_WriteErase (writeAddr + address + byteCount, *dataBuf++, FL_WRITE);
   }
   if (currentTag == DONE_TAG_MAX)
   {
      currentTag = DONE_TAG_MIN;
   }
   else
   {
      currentTag++;
   }

   // Mark new sector with valid tag
   FLASH_WriteErase (writeAddr + TAG_OFFSET, currentTag, FL_WRITE);

   return EE_NO_ERROR;
}

//-----------------------------------------------------------------------------
// EEPROM_ReadBlock
//-----------------------------------------------------------------------------
//
// This routine reads the emulated EEPROM information into the specified
// buffer.
//
// Arguments :  uint8_t address - 8-bit EEPROM address to start at
//              uint8_t *dataBuf - pointer to data array
//              uint8_t length - number of bytes to read
//
// Returns uint8_t : EE_NO_ERROR = Read was sucessful
//              EE_READ_ERROR = Read did not succeed, due to  out-of-bounds
//                 address.
//              EE_SECTOR_ERROR = Could not determine current sector.
//
uint8_t EEPROM_ReadBlock (uint8_t address, uint8_t *dataBuf, uint8_t length)
{
   uint16_t sectorAddress;
   uint8_t byteCount;

   if ((uint16_t)(address+length) > EE_SIZE)
   {
      return EE_READ_ERROR;
   }
   if (findCurrentSector() == 0x00)
   {
      return EE_SECTOR_ERROR;
   }
   sectorAddress = getBaseAddress(currentEepromPage, currentEepromSector);
   // Read data bytes
   for (byteCount = 0; byteCount < length; byteCount++)
   {
      *dataBuf++ = FLASH_Read(sectorAddress+address+byteCount);
   }
   return EE_NO_ERROR;
}


//-----------------------------------------------------------------------------
// copySector
//-----------------------------------------------------------------------------
//
// This internal routine copies the contents of one sector to another, while
// excluding the addresses about to be written.
//
// Arguments :  uint16_t fromAddr - 16-bit Flash address to copy from
//              uint16_t toAddr - 16-bit Flash addres to copy to
//              uint8_t exclude - Starting EEPROM address to exclude for copy.
//              uint8_t length - Number of bytes to exlude from copy.
//
// Returns :    None
//
void copySector(uint16_t fromAddr, uint16_t toAddr, uint8_t exclude, uint8_t length)
{
   uint8_t offset_addr;
   uint8_t copy_byte;

   for (offset_addr = 0; offset_addr < EE_SIZE; offset_addr++)
   {
      //copy_byte = *((uint8_t SI_SEG_CODE *) (fromAddr+offset_addr));
      copy_byte = FLASH_Read(fromAddr+offset_addr);
      
      if ((offset_addr < exclude)||(offset_addr >= exclude+length))
      {
         FLASH_WriteErase ((toAddr+offset_addr), copy_byte, FL_WRITE);
      }
   }
}

//-----------------------------------------------------------------------------
// findNextSector
//-----------------------------------------------------------------------------
//
// This internal routine finds the next available sector in the EEPROM area
// and updates the currentEepromPage and currentEepromSector variables.
//
// Arguments :  None
//
// Returns uint8_t : 0 if next page needs to be erased, 1 if empty sector found
//
uint8_t findNextSector(void)
{
   uint8_t pageNow = currentEepromPage, sectorNow = currentEepromSector;
   uint8_t sectorFound = 2;
   uint8_t sectTag;
   uint16_t sectorAddress;

   while (sectorFound == 2)
   {
      currentEepromSector++;
      if (currentEepromSector == EE_SECTORS)
      {
         currentEepromSector = 0;
         currentEepromPage++;
         if (currentEepromPage == FL_PAGES)
         {
            currentEepromPage = 0;
         }
      }
      sectorAddress = getBaseAddress(currentEepromPage, currentEepromSector);
      sectTag = FLASH_Read(sectorAddress+TAG_OFFSET);
      if (sectTag == NEW_TAG)
      {
         sectorFound = 1;
      }
      else if ((pageNow == currentEepromPage)&&
               (sectorNow == currentEepromSector))
      {
         currentEepromSector = 0;
         currentEepromPage++;
         if (currentEepromPage == FL_PAGES)
         {
            currentEepromPage = 0;
         }
         sectorFound = 0;
      }
   }
   return sectorFound;
}

//-----------------------------------------------------------------------------
// findCurrentSector
//-----------------------------------------------------------------------------
//
// This internal routine finds the currently used sector in the EEPROM area
// and updates the currentEepromPage and currentEepromSector variables.
//
// Arguments :  None
//
// Returns uint8_t : 0 if sector could not be determined, sector tag if sector
//              was found
//
uint8_t findCurrentSector(void)
{
   uint8_t pgCount, sectCount;
   uint8_t emptySectors = 0;
   uint8_t sectTag = NO_TAG, latestTag = NO_TAG, nextTag = NO_TAG;

   for (pgCount = 0; pgCount < FL_PAGES; pgCount++)
   {
      for (sectCount = 0; sectCount < EE_SECTORS; sectCount++)
      {
         sectTag = FLASH_Read(getBaseAddress(pgCount, sectCount)+TAG_OFFSET);

         if (sectTag == NEW_TAG)
         {
            emptySectors++;
         }
         if ((sectTag >= DONE_TAG_MIN)&&(sectTag <= DONE_TAG_MAX))
         {
            if ((latestTag == NO_TAG)||(sectTag == nextTag))
            {
               latestTag = sectTag;
               currentEepromPage = pgCount;
               currentEepromSector = sectCount;

               if (sectTag == DONE_TAG_MAX)
               {
                  nextTag = DONE_TAG_MIN;
               }
               else
               {
                  nextTag = sectTag + 1;
               }
            }
         }
      }
   }
   // If all sectors are empty, start fresh
   if (emptySectors == FL_PAGES*EE_SECTORS)
   {
      latestTag = DONE_TAG_MIN;
      currentEepromPage = FL_PAGES-1;
      currentEepromSector = EE_SECTORS-1;
   }

   return latestTag;
}

//-----------------------------------------------------------------------------
// getBaseAddress
//-----------------------------------------------------------------------------
//
// This internal routine calculates the base address for a Flash sector in the
// emulated EEPROM area.
//
// Arguments :  uint8_t page - zero-based page number to calculate from.
//              uint8_t sector - zero-based sector number to calculate from.
//
// Returns uint16_t: 16-bit Flash address corresponding to the page and sector.
//
uint16_t getBaseAddress(uint8_t page, uint8_t sector)
{
   return (EE_BASE_ADDR+(FL_PAGE_SIZE*page)+(EE_SIZE+EE_TAG_SIZE)*sector);
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------