//-----------------------------------------------------------------------------
// F330_Blinky.c
//-----------------------------------------------------------------------------
// Copyright (C) 2014 Silicon Laboratories, Inc.
//
// AUTH: HF
// DATE: 04 FEB 2003
//
// This program flashes the green LED on the C8051F33x target board about
// five times a second using the interrupt handler for Timer2.
//
// Target: C8051F330/1/2/3/4/5
//
// Tool chain: KEIL C51
// INTVECTOR(0x000) INTERVAL(3)
//
// Release 1.1 / 22Sept2014 (SHY)
//    -Merge Fxxx_Target_Config.h and Fxxx_Target_Interface.h
//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <si_toolchain.h>
#include <C8051F330_defs.h> //   TODO: "nnn" - Fill-in value for actual 
                            //   MCU SFR definition header file name.
#include "Fxxx_Target_Config.h"

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

#define SYSCLK       24500000/8             // SYSCLK frequency in Hz

sbit LED = P1^3;                          // LED='1' means ON
sbit SW2 = P0^7;                          // SW2='0' means switch pressed

#define TGT_CMD_ENTER_BL_MODE       0x05
#define RESERVED_SPACE_ADDR         (APP_FW_END_ADDR + 0x800)

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void SYSCLK_Init (void);
void PORT_Init (void);
void Timer2_Init (int counts);
void Timer2_ISR (void);

// Const on flash to make the code size bigger than 1 page to test the BL
code char ConstZero[20][64] = {{ 0}, { 1}, { 2}, { 3},    { 4}, { 5}, { 6}, { 7}, { 8}, { 9},
                               {10}, {11}, {12}, {13},    {14}, {15}, {16}, {17}, {18}, {19}};

//-----------------------------------------------------------------------------
// Enter_BL_Mode
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// This function will cause a Flash Error Reset to enter BL mode.
//
//-----------------------------------------------------------------------------
unsigned char Enter_BL_Mode(void)
{
    // The return value is just to prevent the compiler from optimizing out
    // the code read

    return *(unsigned char code*)(RESERVED_SPACE_ADDR);
}

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
void main (void) {

   // disable watchdog timer
   PCA0MD &= ~0x40;                       // WDTE = 0 (clear watchdog timer
                                          // enable)

   SYSCLK_Init ();                        // Initialize system clock to
                                          // 24.5MHz
   PORT_Init ();                          // Initialize crossbar and GPIO
   Timer2_Init (SYSCLK / 12 / 10);        // Init Timer2 to generate
                                          // interrupts at a 10Hz rate.

   EA = 1;							      // enable global interrupts

   while (1) {                            // spin forever
   }
}

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use the internal 24.5MHz / 8
// oscillator as its clock source.  Also enables missing clock detector reset.
//
void SYSCLK_Init (void)
{

   OSCICN = 0x80;                         // configure internal oscillator for
                                          // its lowest frequency
   RSTSRC = 0x04;                         // enable missing clock detector
}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports.
// P1.3 - LED (push-pull)
//
void PORT_Init (void)
{

   XBR0     = 0x00;                       // no digital peripherals selected
   XBR1     = 0x40;                       // Enable crossbar and weak pull-ups
   P1MDOUT |= 0x08;                       // enable LED as a push-pull output
}

//-----------------------------------------------------------------------------
// Timer2_Init
//-----------------------------------------------------------------------------
//
// Configure Timer2 to 16-bit auto-reload and generate an interrupt at
// interval specified by <counts> using SYSCLK/48 as its time base.
//
void Timer2_Init (int counts)
{
   TMR2CN  = 0x00;                        // Stop Timer2; Clear TF2;
                                          // use SYSCLK/12 as timebase
   CKCON  &= ~0x60;                       // Timer2 clocked based on T2XCLK;

   TMR2RL  = -counts;                     // Init reload values
   TMR2    = 0xffff;                      // set to reload immediately
   ET2     = 1;                           // enable Timer2 interrupts
   TR2     = 1;                           // start Timer2
}

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Timer2_ISR
//-----------------------------------------------------------------------------
// This routine changes the state of the LED whenever Timer2 overflows.
//
void Timer2_ISR (void) interrupt 5
{
   TF2H = 0;                              // clear Timer2 interrupt flag
   LED = ~LED;                            // change state of LED
}

