//-----------------------------------------------------------------------------
// F36x_Blinky.c
//-----------------------------------------------------------------------------
// Copyright (C) 2014 Silicon Laboratories, Inc.
//
// Program Description:
//
// This program flashes the green LED on the C8051F36x target board about
// five times a second using the interrupt handler for Timer2.
//
//
// How To Test:
//
// 1) Download code to a 'F36x target board
// 2) Ensure that pins 5 and 6 are shorted together on the J12 header
// 3) Run the program.  If the LED flashes, the program is working
//
//
// FID:            36X000045
// Target:         C8051F36x
// Tool chain:     Keil C51 7.50 / Keil EVAL C51
// Command Line:   None
//
// Release 1.1 / 22Sept2014 (SHY)
//    -Merge Fxxx_Target_Config.h and Fxxx_Target_Interface.h
// Release 1.0
//    -Initial Revision (MarkDing)
//    -10 May 2013
//  INTVECTOR(0x400) INTERVAL(3) CD
//  CODE (0x400-0x77FF, ?CO?F36X_INFOBLOCK(0x77F5))

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <si_toolchain.h>
#include <C8051F360_defs.h> //   TODO: "nnn" - Fill-in value for actual 
                            //   MCU SFR definition header file name.
#include "Fxxx_Target_Config.h"


//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

#define SYSCLK 3062500                 // Approximate SYSCLK frequency in Hz
#define RESERVED_SPACE_ADDR            (APP_FW_END_ADDR + 0x800)  // This adreess is a invalid of Flash

sbit LED = P3^2;                       // Green LED: '1' = ON; '0' = OFF

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------

void PORT_Init (void);
void Timer2_Init (int counts);
void Timer2_ISR (void);

// Const on flash to make the code size bigger than 1 page to test the BL
code char ConstZero[20][64] = {{ 0}, { 1}, { 2}, { 3},    { 4}, { 5}, { 6}, { 7}, { 8}, { 9},
                               {10}, {11}, {12}, {13},    {14}, {15}, {16}, {17}, {18}, {19}};

//-----------------------------------------------------------------------------
// Enter_BL_Mode
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// This function will cause a Flash Error Reset to enter BL mode.
//
//-----------------------------------------------------------------------------
unsigned char Enter_BL_Mode(void)
{
    // The return value is just to prevent the compiler from optimizing out
    // the code read

    return *(unsigned char code*)(RESERVED_SPACE_ADDR);
}

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
void main (void) 
{
   PCA0MD &= ~0x40;                    // WDTE = 0 (clear watchdog timer
                                       // enable)
   PORT_Init ();

   Timer2_Init (SYSCLK / 12 / 10);     // Init Timer2 to generate interrupts
                                       // at a 10 Hz rate.

   EA = 1;                             // Enable global interrupts

   SFRPAGE = LEGACY_PAGE;              // Page to sit in for now

   while (1) {}                         // Spin forever
}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Configure the Crossbar and GPIO ports
//
void PORT_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   XBR1    = 0x40;                     // Enable crossbar and weak pull-ups
   P3MDOUT |= 0x04;                    // enable P3.2 (LED) as push-pull output

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// Timer2_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   :
//   1)  int counts - calculated Timer overflow rate
//                    range is postive range of integer: 0 to 32767
//
// Configure Timer2 to auto-reload and generate an interrupt at interval
// specified by <counts> using SYSCLK/12 as its time base.
//
//
void Timer2_Init (int counts)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   TMR2CN = 0x00;                      // Stop Timer2; Clear TF2;
                                       // Use SYSCLK/12 as timebase
   TMR2RL = -counts;                   // Init reload values
   TMR2 = 0xFFFF;                      // Set to reload immediately

   ET2 = 1;                            // Enable Timer2 interrupts

   TR2 = 1;                            // Start Timer2

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Timer2_ISR
//-----------------------------------------------------------------------------
// This routine changes the state of the LED whenever Timer2 overflows.
//
// NOTE: The SFRPAGE register will automatically be switched to the Timer 2
// Page when an interrupt occurs.  SFRPAGE will return to its previous setting
// on exit from this routine.
//
void Timer2_ISR (void) interrupt 5
{
   TF2H = 0;                           // Clear Timer2 interrupt flag
   LED = ~LED;                         // Change state of LED
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------