//-----------------------------------------------------------------------------
// Fxxx_TargetBL_Main.c
//-----------------------------------------------------------------------------
// Copyright (C) 2014 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
//
// The main routine for the Target Bootloader Firmware.
//
// 
//
//
// Target:         C8051Fxxx (Any Silicon Labs Flash MCU)
// Tool chain:     Generic
// Command Line:   None
//
//
// Release 1.0 / 22Sept2014 (SHY)
//    -Ported from F330
//
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <si_toolchain.h>
#include "Fxxx_SFR_Defs_Wrapper.h"     // This header file will include the
// real MCU register definition file
#include "Fxxx_Target_Config.h"
#include "Fxxx_Target_Interface.h"
#include "Fxxx_TargetBL_Config.h"
#include "Fxxx_TargetBL_Interface.h"
#include "Fxxx_BL129_UART_Interface.h"
#include "F85x_CRC.h"
//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------
SI_SBIT(BL_Override_Pin, SFR_P0, 7);
sbit LED = P1^3;                          // LED='1' means ON
SI_SEGMENT_VARIABLE( Page_Buf[TGT_FLASH_PAGE_SIZE], uint8_t, SI_SEG_XDATA);

uint8_t data SRC_Response;
uint16_t data SRC_Page_CRC;
uint8_t data rx_buf[32];
uint8_t Last_Error = 0;

#define APP_MODE        0	
#define BOOTLOADER_MODE 1

// Bit masks for the RSTSRC SFR
#define PORSF  0x02
#define FERROR 0x40

void SRC_Validate_Response(uint8_t response);
void SRC_Validate_PageCRC(uint16_t page_addr);


//=============================================================================
// Main Routine
//=============================================================================
void main(void)
{
    uint8_t device_mode = BOOTLOADER_MODE;
	uint8_t code* codeptr;
	uint16_t data page_addr = 0;
	uint16_t addr = 0;
	uint8_t packets_for_page = 1;
    //---------------------------------------
    // Check the bootloader consition.
    //---------------------------------------
    codeptr = (uint8_t code*)(APP_FW_SIG3_ADDR);
   // The Signature (in Flash) should be valid to allow application FW execution.
   // This is written at the end of the bootloading process by the bootloader.
    if((codeptr[0] == SIG_BYTE3) &&(codeptr[1] == SIG_BYTE2) &&(codeptr[2] == SIG_BYTE1) && (codeptr[3] == SIG_BYTE0))
    {
   		device_mode = APP_MODE;
    }

    if ((!BL_Override_Pin) || (((RSTSRC & PORSF) == 0) && (RSTSRC & FERROR)))
    {
        device_mode = BOOTLOADER_MODE;
    }

    if (device_mode == APP_MODE)
    {
        // If not in BL Override, jump to application
        START_APPLICATION();
    }

    //-------------------------------------------
    // ** BL Mode ** Initialize MCU and Variables
    //-------------------------------------------
    Device_Init();

    //-------------------------------------------
    // Main Loop
    //-------------------------------------------
	while(1)
	{
		while(SRC_Disp_TGT_Info() != SRC_RSP_OK);
		SRC_Response = SRC_Get_Info();
        SRC_Validate_Response(SRC_Response);
        if (Last_Error != 0)
	        goto error;
		Flash_Key0 = rx_buf[5];
		Flash_Key1 = rx_buf[6];
		while(1)
		{
			SRC_Response = SRC_Get_Page_Info();
			SRC_Page_CRC = rx_buf[4] | (rx_buf[5] << 8);
			page_addr = rx_buf[1] | (rx_buf[2] << 8);
            SRC_Validate_Response(SRC_Response);
            if (Last_Error != 0)
	            break;
	        // Exit this loop if no more pages are available from source
    	    if (SRC_Response == SRC_RSP_DATA_END) 
	    	    break;

			TGT_Erase_Page(page_addr);
			addr = page_addr;

            packets_for_page = TGT_BL_PACKETS_FOR_GET_PAGE;
            while(packets_for_page--) {
                SRC_Response = SRC_Get_Page(Page_Buf);
                SRC_Validate_Response(SRC_Response); // this will not check for CRC match                

                if (Last_Error != 0)
	                break;

                TGT_Write_Flash(Page_Buf, addr);
				addr += SRC_CMD_GET_PAGE_RX_SZ;
            }
            SRC_Validate_PageCRC(page_addr); // this will check for CRC match                
            if (Last_Error != 0)
	            break;
		}
        // Set flash keys to 0
        Flash_Key0 = 0;
        Flash_Key1 = 0;
error:
        if (Last_Error != 0)
        {
            SRC_Response = SRC_Disp_Info_Code(Last_Error);
            Last_Error = 0;
        }
        RSTSRC = 0x12; // Initiate software reset with vdd monitor enabled
	}
}

//-----------------------------------------------------------------------------
// SRC_Validate_Response
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
//
//
//-----------------------------------------------------------------------------
void SRC_Validate_Response(uint8_t response)
{
    if ((response != SRC_RSP_OK) && (response != SRC_RSP_DATA_END))
    {
	    Last_Error = ERR_SRC_UNEXPECTED_RSP;
		return;
    }
}

void SRC_Validate_PageCRC(uint16_t page_addr)
{
    uint16_t buf_crc;    
    buf_crc = Get_Buf_CRC((uint8_t code *)page_addr, TGT_FLASH_PAGE_SIZE);
    if (buf_crc != SRC_Page_CRC)
    {
       Last_Error = ERR_SRC_CRC_MISMATCH;
    }
}


//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------
