//-----------------------------------------------------------------------------
// SPI_F300_Test.c
//-----------------------------------------------------------------------------
// Copyright 2001 Cygnal Integrated Products, Inc.
//
// AUTH: BD
// DATE: 14 DEC 01
//
// This program demonstrates how a collection of SPI master
// routines for the 8051F30x processors can be used in a C program.
//
// This program sets up the GPIO pins on the C8051F30x device for the correct
// functionality, then uses the SPI_Transfer function to send and receive
// information through the SPI pins. As information is sent, the progress of
// the program is sent out through the UART to be monitored on a connected
// terminal program.
//
// For this code to be functional, *one* of the following files should also be
// compiled or assembled, and the resulting object file must be linked to the
// object file produced from this file:
//
//    SPI_MODE0.c    Mode 0 SPI Master Implementation in C
//    SPI_MODE0.asm  Mode 0 SPI Master Implementation in Assembly
//    SPI_MODE1.c    Mode 1 SPI Master Implementation in C
//    SPI_MODE1.asm  Mode 1 SPI Master Implementation in Assembly
//    SPI_MODE2.c    Mode 2 SPI Master Implementation in C
//    SPI_MODE2.asm  Mode 2 SPI Master Implementation in Assembly
//    SPI_MODE3.c    Mode 3 SPI Master Implementation in C
//    SPI_MODE3.asm  Mode 3 SPI Master Implementation in Assembly
//
// Target: C8051F30x
// Tool chain: KEIL C51 6.03 / KEIL EVAL C51
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include <c8051f300.h>                 // SFR declarations
#include <stdio.h>                     // Standard I/O
#include "SPI_defs.h"                  // SPI port definitions

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F30x
//-----------------------------------------------------------------------------

sfr16 DP       = 0x82;                 // data pointer
sfr16 TMR2RL   = 0xca;                 // Timer2 reload value
sfr16 TMR2     = 0xcc;                 // Timer2 counter
sfr16 PCA0CP1  = 0xe9;                 // PCA0 Module 1 Capture/Compare
sfr16 PCA0CP2  = 0xeb;                 // PCA0 Module 2 Capture/Compare
sfr16 PCA0     = 0xf9;                 // PCA0 counter
sfr16 PCA0CP0  = 0xfb;                 // PCA0 Module 0 Capture/Compare

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

#define SYSCLK       24500000          // SYSCLK frequency in Hz
#define BAUDRATE     115200            // Baud rate of UART in bps

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------

void PORT_Init (void);                 // Port I/O configuration
void SYSCLK_Init (void);               // SYSCLK Initialization
void UART0_Init (void);                // UART0 Initialization

extern char SPI_Transfer (char);       // SPI Transfer routine
                                                      
//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main (void) {

   unsigned char test_counter, SPI_return;   // used to test SPI routine

   // Disable Watchdog timer
   PCA0MD &= ~0x40;                    // WDTE = 0 (clear watchdog timer 
                                       // enable)

   SYSCLK_Init ();                     // initialize oscillator
   PORT_Init ();                       // initialize ports and GPIO
   UART0_Init ();                      // initialize UART0
   EA = 1;                             // enable global interrupts

   while (1)
   {
      for (test_counter = 0; test_counter <= 0xFF; test_counter++)
      {

         NSS = 0x00;                   // select SPI Slave device

         SPI_return = SPI_Transfer(test_counter); // send/receive SPI byte

         NSS = 0x01;                   // de-select SPI Slave device
         
         printf("\nSPI Out = 0x%02X, SPI In = 0x%02X", (unsigned)test_counter, 
         (unsigned)SPI_return);  
                                       // send SPI data out to UART
                                       // for verification purposes
      }
   }
}

//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports.
// P0.0 - MOSI (push-pull)
// P0.1 - MISO
// P0.2 - SCK (push-pull)
// P0.3 - NSS (push-pull)
// P0.4 - UART TX (push-pull)
// P0.5 - UART RX
// P0.6 - 
// P0.7 - 
//
void PORT_Init (void)
{
   XBR0    =  0x0F;                    // skip SPI pins in XBAR
   XBR1    =  0x03;                    // UART0 TX and RX pins enabled
   XBR2    =  0x40;                    // Enable crossbar and weak pull-ups
   P0MDOUT |= 0x1D;                    // enable TX0, MOSI, SCK, and NSS as
                                       // push-pull outputs
}

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use the internal 24.5 MHz clock
// as its clock source.
//
void SYSCLK_Init (void)
{

   OSCICN = 0x07;                      // select internal oscillator as SYSCLK
                                       // source
}

//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------
//
// Configure the UART0 using Timer1, for <BAUDRATE> and 8-N-1.
//
void UART0_Init (void)
{
   SCON0 = 0x10;                       // SCON0: 8-bit variable bit rate
                                       //        level of STOP bit is ignored
                                       //        RX enabled
                                       //        ninth bits are zeros
                                       //        clear RI0 and TI0 bits
   if (SYSCLK/BAUDRATE/2/256 < 1) 
   {
      TH1 = -(SYSCLK/BAUDRATE/2);
      CKCON &= ~0x13;                  
      CKCON |= 0x10;                   // T1M = 1; SCA1:0 = xx
   } 
   else if (SYSCLK/BAUDRATE/2/256 < 4) 
   {
      TH1 = -(SYSCLK/BAUDRATE/2/4);
      CKCON &= ~0x13;
      CKCON |=  0x01;                  // T1M = 0; SCA1:0 = 01
   } 
   else if (SYSCLK/BAUDRATE/2/256 < 12) 
   {
      TH1 = -(SYSCLK/BAUDRATE/2/12);
      CKCON &= ~0x13;                  // T1M = 0; SCA1:0 = 00
   } 
   else 
   {
      TH1 = -(SYSCLK/BAUDRATE/2/48);
      CKCON &= ~0x13;
      CKCON |=  0x02;                  // T1M = 0; SCA1:0 = 10
   }

   TL1 = 0xff;                         // set Timer1 to overflow immediately
   TMOD |= 0x20;                       // TMOD: timer 1 in 8-bit autoreload
   TMOD &= ~0xD0;                      // mode
   TR1 = 1;                            // START Timer1
   TI0 = 1;                            // Indicate TX0 ready
}


