//-----------------------------------------------------------------------------
// Stepper.c
//-----------------------------------------------------------------------------
// Copyright 2003 Silicon Laboratories, Inc.
//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f300.h>                 // include SFR declarations
//-----------------------------------------------------------------------------
// defines and typedefs
//-----------------------------------------------------------------------------
#define SYSCLK       24500000          // SYSCLK frequency in Hz
#define BAUDRATE     57600             // Baud rate of UART in bps

#define READ_BUFFER_SIZE 16            // change UART receive buffer size here
#define WRITE_BUFFER_SIZE 16           // change UART receive buffer size here

#define ON 0                           // LED is active low
#define OFF 1
#define ACCELERATE 3                   // motor states
#define SLEWING 2
#define DECELERATE 1
#define DONE 0
#define INIT_TZERO 80                  // Tzero used at start-up and with
                                       // manual mode, change value here for
                                       // different stepper motors

typedef union                          // union used for writing to TL0 & TH0
    {
        struct
        {
            unsigned char hi;
            unsigned char lo;
        } b;
        unsigned int w;
    }udblbyte;
//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------
//
// half step stepper motor step pattern
// transistors are in order  B-,B+,A-,A+
//
// 0x01 = 0001
// 0x05 = 0101
// 0x04 = 0100
// 0x06 = 0110
// 0x02 = 0010
// 0x0A = 1010
// 0x08 = 1000
// 0x09 = 1001

const unsigned char code StepPattern[8]=
   {0x01,0x05,0x04,0x06,0x02,0x0A,0x08,0x09};
//
// stepper motor linear velocity profile
// 255 * [sqrt(n+1)-sqrt(n)]
// n = 0 to 255
//
const unsigned char code StepTable[256]=
{
   0xFF, 0x69, 0x51, 0x44, 0x3C, 0x36, 0x32, 0x2E,
   0x2B, 0x29, 0x27, 0x25, 0x24, 0x22, 0x21, 0x20,
   0x1F, 0x1E, 0x1D, 0x1C, 0x1C, 0x1B, 0x1A, 0x1A,
   0x19, 0x19, 0x18, 0x18, 0x17, 0x17, 0x17, 0x16,
   0x16, 0x16, 0x15, 0x15, 0x15, 0x14, 0x14, 0x14,
   0x14, 0x13, 0x13, 0x13, 0x13, 0x12, 0x12, 0x12,
   0x12, 0x12, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
   0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
   0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
   0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E,
   0x0E, 0x0E, 0x0E, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
   0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
   0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
   0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
   0x0C, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
   0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
   0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0A, 0x0A,
   0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
   0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
   0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
   0x0A, 0x0A, 0x0A, 0x09, 0x09, 0x09, 0x09, 0x09,
   0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
   0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
   0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
   0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
   0x09, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
   0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
   0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
   0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
   0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
   0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
   0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x07, 0x07
};
//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
//
// main and user interface functions
void SYSCLK_Init (void);               // initialize port pins and crossbar
void PORT_Init (void);                 // initialize system clock and watchdog
void Timer_Init (void);                // initialize timer for Motor and UART
void UART_Init (void);                 // initialize UART
void Motor_Init(void);                 // initialize Motor variables
void doneMsg(void);                    // display done message
void error(void);                      // display error message
void delay (void);                     // delay used for position updating
void INT0_ISR (void);                  // external interrupt used for SW2
//
// stepper motor control functions
void move (unsigned int);
void Timer_ISR (void);
unsigned int MUL8x8(unsigned char, unsigned char);
//
//  BUFFERED UART FUNCTION PROTOTYPES
//--Top Level - User Functions ------------------------------------------------

// put a string into transmit buffer, used to display text messages
void puts (char *);

// converts a 16 bit number to a string of ASCII characters
// and stores the characters into the transmit buffer
void putuint(unsigned int);

// converts an 8 bit number to a string of ASCII characters
// and stores the characters into the transmit buffer
void putuchar(unsigned char);

// retrieves ASCII characters from the receiver buffer
// and converts a string of digits into an 8-bit number
unsigned char getuchar(void);

// retrieves ASCII characters from the receiver buffer
// and converts a string of digits into an 16bit number
unsigned int getuint(void);

void newline(void);                    // outputs carriage return & line feed


//----------Middle Level - Primitive User Functions ---------------------------

// retrieves one character from the receive buffer using readc and echoes
// the character to the transmit buffer
char getc (void);

// stores one character into the transmit buffer using writec
void putc (char);

void readClear(void);                  // clears read buffer

void writeClear(void);                 // clears write buffer

//----------------Lowest level - Hardware access functions --------------------
char readc (void);                     // pulls one character directly
                                       // from the receive buffer

void writec (char);                    // writes one character directly
                                       // to the transmit buffer

//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------
//
// user interface variables
sbit LED     = P0 ^ 6;                 // bit set high to turn LED on
//
// Stepper Motor Variables
unsigned int Position;                 // current motor position
unsigned char TableIndex;              // index for stepper motor table
unsigned char MaxTableIndex;           // maximum index for desired profile
unsigned int SlewCount;                // down counter for slewing phase
unsigned char Tzero;                   // initial period, sqrt(2/alpha)
unsigned char PatternIndex;            // index for step pattern, modulus 8 counter
bit Forward;                           // 1 for forward, 0 for reverse
bit doneFlag;                          // done flag used for Timer ISR handshake
unsigned char motorState;              // motor state variable
//
// UART Global Variables

// UART receiver buffer
char idata readBuffer[READ_BUFFER_SIZE];
// UART transmitter buffer
char idata writeBuffer[WRITE_BUFFER_SIZE];
unsigned char writeIndex;              // points to next free write byte
unsigned char readIndex;               // points to next free read byte
unsigned char writeCount;              // number of bytes in write buffer
unsigned char readCount;               // number of bytes in read buffer
bit TX_Idle;                           // flag set when TX buffer is empty
//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
//
// The main function initializes everything and then services the user
// interface. The user interface parses characters from the UART and
// executes basic commands. There are three interrupt service routines
// which function in the background - The Timer_ISR which controls the
// stepper motor; The UART_ISR which moves characters to/from the buffers;
// and the INT0_ISR that handles the pushbutton switch commands.
//
void main (void)
{
   char theChar;                       // the character to be parsed
   unsigned int newP;                  // new position to move to

   SYSCLK_Init ();                     // Init system clock to 24.5MHz
   PORT_Init ();                       // Init crossbar and GPIO
   Timer_Init();                       // Init T0 for stepper motor and
                                       // T1 for Baud rate
   UART_Init();                        // Init UART
   Motor_Init();                       // Init motor global variables
   EA = 1;                             // enable global interrupts

   putc('>');                          // display prompt

    while(1)                           // main loop
    {
      if(LED==ON)                      // display position while moving
      {
         putuint(Position);            // display position
         puts("    ");                 // blank out trailing characters
         putc('\r');                   // overwrite line by not using linefeed
         delay();                      // delay sets display update rate
      }
      else
      {
         if(doneFlag)
         {
            doneMsg();                 // if done display message
         }
         theChar = getc();             // get character to be parsed
         switch(theChar)               // parse character
         {
            case '\r':                 // if return character
               putc('\n');             // linefeed with no carriage return
               putc('>');              // display prompt
               break;
            case 'p':                  // p for new position
               newP = getuint();       // get number as unsigned integer
               newline();
               puts("Moving...");      // display moving status indicator
               newline();
               puts("Position:");      // display position tag
               newline();
               move(newP);             // initiate move to new position
               break;
            case 'a':                  // a for change acceleration
               Tzero = getuchar();     // get number as unsigned character
               newline();
               puts("Acceleration:");  // confirm acceleration changed
               newline();
               putuint(Tzero);         // display new acceleration
               newline();
               putc('>');
               break;
            case 's':                  // s for display status
               newline();
               puts("Position:");      // display position tag
               newline();
               putuint(Position);      // display position value
               newline();
               puts("Acceleration:");  // display acceleration tag
               newline();
               putuint(Tzero);         // display acceleration value
               newline();
               putc('>');              // display prompt
               break;
            default:
               error();                // display error message
         }
      }
    }
}
//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use the internal 24.5MHz
// oscillator as its clock source.  Also enables missing clock detector reset.
//
void SYSCLK_Init (void)
{
   OSCICN = 0x07;                      // set SYSCLK to OSC frequency
   RSTSRC = 0x04;                      // enable missing clock detector
   PCA0MD &= ~0x40;                    // disable watchdog timer
}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports.
// P0.0 - A+
// P0.1 - A-
// P0.2 - B+
// P0.3 - B-
// P0.4 - Txd
// P0.5 - Rxd
// P0.6 - LED
// P0.7 - C2D/Switch
//
void PORT_Init (void)
{
   XBR0 = 0x4F;                        // Crossbar Register 1
   XBR1 = 0x03;                        // Crossbar Register 2
   XBR2 = 0x40;                        // Crossbar Register 3
   P0 = 0xC0;                          // Turn off MOSFETs, Set P0.6 & P0.7
   P0MDOUT = 0x5F;                     // Output configuration for P0
   P0MDIN = 0xFF;                      // Input configuration for P0
   IP = 0x02;                          // T0 high priority, others low
   IT01CF = 0x70;                      // use P0.7 for INT1
   IT1 = 1;                            // edge sensitive interrupt
   IE1 = 0;                            // clear external interrupt
   EX1 = 1;                            // enable external interrupt 1
}
//-----------------------------------------------------------------------------
void Timer_Init (void)
{
   CKCON = 0x12;                       // T1 uses sysclk, T0 uses /48,
   TMOD = 0x21;                        // T1 mode 2, T0 mode 1
}
//-----------------------------------------------------------------------------
void UART_Init(void)
{
   SCON0   = 0x10;                     // mode 1, 8-bit UART, disable receiver
   TH1    = 0x2C;                      // set Timer1 reload value for 57600
   TR1    = 1;                         // start Timer1
   TX_Idle = 1;                        // set idle flag
   readClear();                        // zero out read buffer
   writeClear();                       // zero out write buffer
   ES0 = 1;                            // enable UART interrupt
}
//-----------------------------------------------------------------------------
void Motor_Init()
{
   Tzero = INIT_TZERO;                       // initialize acceleration
   Position = 0x0000;                  // zero position
   LED = OFF;                          // turn off LED
   doneFlag = 0;                       // clear done flag
}
//-----------------------------------------------------------------------------
void error (void)                      // used by main
{
   newline();
   puts("invalid character");          // display error message
   newline();
   putc('>');
}
//-----------------------------------------------------------------------------
void delay (void)                      // used by main
{
   unsigned char i,j;

   for(i=255;i>0;i--)                  // simple delay for display update
   {
      for(j=255;j>0;j--);
   }
}
//-----------------------------------------------------------------------------
void doneMsg(void)                     // called from Timer_ISR
{
   putc('\r');                         // overwrite final position
   putuint(Position);                  // display int as string
   newline();
   puts("done!");                      // indicate move complete
   newline();
   putc('>');                          // display prompt
   doneFlag=0;

}
//-----------------------------------------------------------------------------
void INT0_ISR (void) interrupt 2       // external interrupt 0 used for switch
{
      if(LED==OFF)                     // ignore switch if moving
      {
         if(Position==0)               // if home move to 100
         {
            move(1600);
         }
         else                          // if not home move to home
         {
            move(0);
         }
      }
}
//-----------------------------------------------------------------------------
// move
// This function calculates the profile and initializes the stepper
// motor. Function will abort if the target equals the Position or if
// the motor is still moving. Forward is set to 1 if target>Position.
// the length is calculated as the absolute value of Position minus
// target. For short moves less than 1024, MaxTableIndex is length
// divided by 4. For long moves, MaxTableIndex is set to 255.
//
// slewCount = length - 2 * MaxTableIndex - 1
//
// The slewcount is calculated by subtracting MaxTableIndex twice and
// decrementing. The TableIndex is initialized to zero.
//
void move (unsigned int target)
{
   unsigned int length;                // used to calculate length of move

   if (target != Position)             // abort if already there
   {
      if (target > Position)
      {
         Forward = 1;                  // set forward flag
         length = target - Position;   // subtract smaller (position)
      }
      else
      {
         Forward = 0;                  // clear forward flag
         length = Position - target;   // subtract smaller (target)
      }
      if (length < 1024)               // if short move
         MaxTableIndex = length>>2;    // divide length by 4
      else
         MaxTableIndex = 0xff;         // else max is 255
      SlewCount = length;              // build value in SlewCount
      SlewCount -= MaxTableIndex;      // subtract MaxTableIndex twice
      SlewCount -= MaxTableIndex;
      SlewCount--;                     // Subtract one to account first step
      TableIndex = 0;                  // init table index
      motorState = ACCELERATE;               // init motor state
      TL0 = -100;                      // move starts in 100 ticks
      TH0 = 0xFF;                      // extend sign
      LED = ON;                        // turn on LED
      ET0 = 1;                         // enable Timer0 interrupts
      TR0 = 1;                         // start Timer0
   }
      else
   {
      doneFlag=1;                       // if done display message
   }
}
//-----------------------------------------------------------------------------
// Timer_ISR()
// This is the timer interrupt service routine for the stepper motor.
// First the PatternIndex and Position are incremented or decremented
// depending on the direction of the motor. Then the new pattern is
// output to the stepper motor. Nested if..else statements are used to
// determine the if the motor is in the acceleration, slewing, or
// deceleration phase. The TableIndex and SlewCount are modified
// accordingly. When the move is complete, further output compares and
// interrupts are disabled.
//
void Timer_ISR (void) interrupt 1
{
   unsigned char TableValue;
   unsigned int offset;
   udblbyte time;

   if (Forward)                        // if forward
   {
      PatternIndex++;                  // increment step pattern
      PatternIndex &= 0x07;            // fix modulus 8 counter
      Position++;                      // increment Position
   }
   else
   {
      PatternIndex--;                  // decrement step pattern
      PatternIndex &= 0x07;            // fix modulus 8 counter
      Position--;                      // increment Position
   }
   // output step pattern, set bit 7 because it is an input
   P0 = StepPattern[PatternIndex] | 0x80;

   // determine motor state based on counter values
   if (SlewCount == 0)
      if (TableIndex == 0)
         motorState = DONE;
      else
         motorState = DECELERATE;
   else
      if (TableIndex < MaxTableIndex)
         motorState = ACCELERATE;
      else
         motorState = SLEWING;

   if (motorState == DONE)
   {
      ET0     = 0;                     // disable T0 interrupts
      TR0     = 0;                     // stop T0
      LED = OFF;                       // turn off LED
      doneFlag = 1;                    // display done message
   }
   else
   {
      // get value from table, multiply by T zero
      TableValue = StepTable[TableIndex];
      offset = MUL8x8(Tzero, TableValue);
      TR0 = 0;                         // stop Timer0
      time.b.lo = TL0;                 // read lo byte first
      time.b.hi = TH0;                 // read hi byte second
      time.w = time.w - offset;        // calculate new time
      TL0 = time.b.lo;                 // write lo byte first
      TH0 = time.b.hi;                 // write hi byte second
      TR0 = 1;                         // start Timer0
     if (motorState == DECELERATE)     // if decelerating
         TableIndex--;                 // decrement table index
      else if(motorState == ACCELERATE)// if accelerating
         TableIndex++;                 // increment table index
      else if (motorState == SLEWING)  // if slewing
         SlewCount--;                  // decrement slewCount
      ET0     = 1;                     // enable Timer0 interrupts
   }
}
//-----------------------------------------------------------------------------
unsigned int MUL8x8(unsigned char a, unsigned char b)
{
   unsigned int ab;
   ab = (unsigned int) a * b;          // cast a to int to trick compiler
   return ab;                          // return int
}
//-----------------------------------------------------------------------------
void puts(char *string)                // puts a string into send buffer
{
   while(*string)                      // while not null character
   {
      putc(*string);                   // put character at pointer in buffer
      string++;                        // increment pointer
   }
}
//-----------------------------------------------------------------------------
unsigned int getuint(void)             // get string and convert to int
{
   char theChar;
   unsigned int i;

   i=0;                                // build value in i
   theChar=getc();                     // get next character
   while( theChar<'0' || theChar>'9')  // while not 0-9
      theChar=getc();
   while(theChar>='0' && theChar<='9') // while 0-9
   {
      theChar -= '0';                  // convert from ASCII
      i *= 10;                         // shift decimal point
      i += theChar;                    // add next digit
      theChar=getc();
   }
   return i;                           // return int value
}
//-----------------------------------------------------------------------------
void putuint(unsigned int i)
{
   char string[7];                     // string used to build output
   char *sp;                           // string pointer

   sp=string + 6;                      // build back to front
   *sp=0;                              // insert null character
   do
   {
      sp--;                            // predecrement pointer
      *sp=i%10 + 0x30;                 // take modulus add 30
      i/=10;                           // divide i by 10
   } while (i);                        // while i not zero
                                       // now output front to back
   while (*sp)                         // while not null character
   {
      putc(*sp);                       // put character in buffer
      sp++;                            // increment pointer
   }
}
//-----------------------------------------------------------------------------
unsigned char  getuchar ()             // same as getuint but returns uchar
{
   char theChar;
   unsigned char i;
   i=0;                                // build value in i
   theChar=getc();                     // get next character
   while( theChar<'0' || theChar>'9')  // while not 0-9
      theChar=getc();
   while(theChar>='0' && theChar<='9') // while 0-9
   {
      theChar -= '0';                  // convert from ASCII
      i *= 10;                         // shift decimal point
      i += theChar;                    // add next digit
      theChar=getc();
   }
   return i;                           // return uchar value
}
//-----------------------------------------------------------------------------
void newline(void)                     // normally cr and lf are used together
{
   putc('\r');                         // output carriage return
   putc('\n');                         // output linefeed
}
//-----------------------------------------------------------------------------
// getc
// Gets a character from the read buffer using readc().
// The getc() function also echos the incoming keystrokes to the display;
//
char getc(void)
{
   char theChar;
   theChar=readc();                    // get character using readc
   writec(theChar);                    // echo characters to display
   return theChar;
}
//-----------------------------------------------------------------------------
// putc
// This is a totally unnecessary layer of abstraction. It is only used to be
// consistent with the getc function which requires an additional layer of
// abstraction to handle character echo.
//
void putc(char theChar)
{
   writec(theChar);
}
//-----------------------------------------------------------------------------
void readClear(void)                   // clears read buffer
{
   unsigned char i;
   readCount=0;
   readIndex=0;
   i = READ_BUFFER_SIZE;
   do
   {
      i--;                             // predecrement
      readBuffer[i]=0;                 // zero all data
   } while (i != 0);
}
//-----------------------------------------------------------------------------
void writeClear(void)                  // clears write buffer
{
   unsigned char i;
   writeCount=0;
   writeIndex=0;
   i = WRITE_BUFFER_SIZE;
   do
   {
      i--;                             // predecrement
      writeBuffer[i]=0;                // zero all data
   } while (i != 0);
}
//-----------------------------------------------------------------------------
// readc()
//
// The readc() function is the lowest level function which provides
// direct access to the read buffer. It reads one character from the
// read buffer.
//
// Note that readc will wait if the read buffer is empty. This is usually
// desired if the program is waiting for user input.
//
// readc is a driver function and is closely integrated with the ISR.
// UART interrupts are temporarily disabled while the buffer is updated,
// This prevents the ISR from modifying the buffer and counter values
// during processing. That would be bad.

char readc(void)
{
   char theChar;                       // the character to return
   signed char i;                      // signed value to build location
   while (readCount==0);               // wait here if buffer empty [blocking]
   ES0 = 0;                            // disable UART interrupts
   i = readIndex - readCount;          // back up by readcount
   if (i < 0)                          // fix value if out of range
      i+=READ_BUFFER_SIZE;
   theChar = readBuffer[i];            // get character from read buffer
   readCount--;                        // one less character in the buffer
   ES0 = 1;                            // enable UARt interrupt
   return theChar;                     // return the character
}
//-----------------------------------------------------------------------------
//
// writec()
//
// The writec() function is the lowest level function which allows access
// to the write buffer. It writes one character to the write buffer.
//
// Note that writec will wait if the write buffer is full. This is usually
// desired to prevent the write buffer from overflowing.
//
// writec is a driver function and is closely integrated with the ISR.
// UART interrupts are temporarily disabled while the buffer is updated to
// prevent the ISR from modifying the buffer and counter values during
// processing. That would be bad.

void writec(char theChar)
{
   // wait here if full [blocking]
   while (writeCount >= WRITE_BUFFER_SIZE);
   ES0 = 0;                            // disable UART interrupts
   writeBuffer[writeIndex] = theChar;  // put character in buffer at writeIndex
   writeIndex++;                       // increment index
   if (writeIndex >= WRITE_BUFFER_SIZE)// fix if out of range
      writeIndex = 0;
   writeCount++;                       // one more character in buffer
   if(TX_Idle)                         // if transmitter idle flag set
   {
      TI0 = 1;                         // force TX interrupt
      TX_Idle = 0;                     // idle no more
   }
   ES0 = 1;                            // enable UART interrupts
}
//-----------------------------------------------------------------------------
void uartISR(void) interrupt 4         // main UART interrupt service routine
{
   char dummy;
   signed char i;
                                       // Who done it?
   if(RI0 == 1)                        // Was it the receiver?
   {
      RI0 = 0;                         // yep, clear receiver flag
      // if not full
      if (readCount != READ_BUFFER_SIZE)
      {
         readBuffer[readIndex]= SBUF0; // read char from UART
         readIndex++;                  // increment index
         if (readIndex == READ_BUFFER_SIZE)
            readIndex = 0;             // fix if out of range
         readCount++;                  // one more in the buffer
      }
      else
      {
         dummy = SBUF0;                // drop characters dummy!
      }
   }
   if(TI0 == 1)                        // Was it the transmitter?
   {
      TI0 = 0;                         // yep, clear transmitter flag
      if (writeCount>0)                // if not empty
      {
         i = writeIndex - writeCount;  // calculate where to get it
         if (i < 0)                    // fix if out of range
            i+=WRITE_BUFFER_SIZE;
         SBUF0 = writeBuffer[i];       // write character to UART
         writeCount--;                 // one less in the buffer
      }
      else
      {
      TX_Idle = 1;                     // set idle flag when empty
      }
   }
}
//-----------------------------------------------------------------------------
