/***********************************************************************
*                    SEGGER Microcontroller GmbH                       *
*                        The Embedded Experts                          *
************************************************************************
*                                                                      *
*                  (c) SEGGER Microcontroller GmbH                     *
*                        All rights reserved                           *
*                          www.segger.com                              *
*                                                                      *
************************************************************************
*                                                                      *
************************************************************************
*                                                                      *
*                                                                      *
*  Licensing terms                                                     *
*                                                                      *
* Redistribution and use in source and binary forms, with or without   *
* modification, are permitted provided that the following conditions   *
* are met:                                                             *
*                                                                      *
* 1. Redistributions of source code must retain the above copyright    *
* notice, this list of conditions and the following disclaimer.        *
*                                                                      *
* 2. Redistributions in binary form must reproduce the above           *
* copyright notice, this list of conditions and the following          *
* disclaimer in the documentation and/or other materials provided      *
* with the distribution.                                               *
*                                                                      *
*                                                                      *
* THIS SOFTWARE IS PROVIDED BY COPYRIGHT HOLDER "AS IS" AND ANY        *
* EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE    *
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR   *
* PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDER BE        *
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,     *
* OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,             *
* PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR   *
* PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY  *
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT         *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH.    *
* DAMAGE.                                                              *
*                                                                      *
************************************************************************

-------------------------- END-OF-HEADER -----------------------------

File    : FlashPrg.c
Purpose : Implementation of RAMCode template
*/

#include <stddef.h>
#include "stdint.h"
#include "string.h"
#include "FlashOS.h"
#include "rsi_qspi_flm.h"
#include "rsi_common_apis_flm.h"
#include "rsi_systick_config.h"
#include "rsi_bootup_config_flm.h"
#include "Flash_Intf.h"

/*********************************************************************
*
*       Defines (configurable)
*
**********************************************************************
*/

uint32_t RSI_FLASH_Initialize(void);
uint32_t RSI_FLASH_EraseSector(uint32_t sector_address);
uint32_t RSI_FLASH_Verify(uint32_t address, unsigned char *data, uint32_t length);
void GetQspiConfig(spi_config_t *spi_config);
void qspi_spi_init(qspi_reg_t *qspi_reg,spi_config_t *spi_config,uint32_t flash_init_req,uint32_t wr_reg_delay_ms,uint8_t fifo_thrsld);
uint32_t RSI_FLASH_Erasechip(void);
void qspi_flash_init(qspi_reg_t *qspi_reg, spi_config_t *spi_config, uint32_t wr_reg_delay_ms);
//
// Only compile in functions that make sense to keep RAMCode as small as possible
//
#define SUPPORT_NATIVE_VERIFY         (1)   // Non-memory mapped flashes only. Flash cannot be read memory-mapped
#define SUPPORT_NATIVE_READ_FUNCTION  (0)   // Non-memory mapped flashes only. Flash cannot be read memory-mapped
#define SUPPORT_ERASE_CHIP            (1)   // To potentially speed up production programming: Erases whole flash bank / chip with special command
#define SUPPORT_TURBO_MODE            (0)   // Currently available for Cortex-M only
#define SUPPORT_SEGGER_OPEN_ERASE     (0)   // Flashes with uniform sectors only. Speed up erase because 1 OFL call may erase multiple sectors
#define SUPPORT_SEGGER_OPEN_PROGRAM   (0)
#define SUPPORT_SEGGER_BLANK_CHECK    (1)
#define SUPPORT_ERASE_SECTOR			    (1) //For Dual Flash

/*********************************************************************
*
*       Defines (fixed)
*
**********************************************************************
*/

#define PAGE_SIZE_SHIFT             (12)   // Smallest amount of data that can be programmed. <PageSize> = 2 ^ Shift. Shift = 3 => <PageSize> = 2^3 = 8 bytes
#define SECTOR_SIZE_SHIFT           (12)   // Flashes with uniform sectors only. <SectorSize> = 2 ^ Shift. Shift = 12 => <SectorSize> = 2 ^ 12 = 4096 bytes

//
// Default definitions for optional functions if not compiled in
// Makes Api table code further down less ugly
//
#if (SUPPORT_ERASE_CHIP == 0)
  #define EraseChip NULL
#endif

#if (SUPPORT_ERASE_SECTOR == 0)
  #define EraseSector NULL
#endif

#if (SUPPORT_NATIVE_VERIFY == 0)
  #define Verify NULL
#endif
#if (SUPPORT_NATIVE_READ_FUNCTION == 0)
  #define SEGGER_OPEN_Read NULL
#endif
#if (SUPPORT_SEGGER_OPEN_ERASE == 0)
  #define SEGGER_OPEN_Erase NULL
#endif
#if (SUPPORT_TURBO_MODE == 0)
  #define SEGGER_OPEN_Start NULL
#endif

#if (SUPPORT_SEGGER_OPEN_PROGRAM == 0)
  #define SEGGER_OPEN_Program NULL
#endif

#if (SUPPORT_SEGGER_BLANK_CHECK == 0)
  #define BlankCheck NULL
#endif

//! Memory length for driver
#define GLOBAL_BUFF_LEN 12000

#define RSI_CHUNK_SIZE 4096

//!offset for code start address
#define START_ADDR_OFFSET 0x012000

//!offset for header start address
#define HEADER_START_ADDR_OFFSET 0x011000

//!qspi hardware address
#define QSPI_HW_ADDR 0x12000000

//!Header length
#define HEADER_LENGTH 64

//!checksum offset
#define CHECKSUM_OFFSET 236

//!24 BIT address mask
#define ADDRESS_MASK_24BIT 0x00FFFFFF

//!24 BIT address mask
#define ADDRESS_MASK_32BIT 0xFFFFFFFF

#define IMAGE_TYPE_MASK	   (0x1)

#define TA_IMAGE           (0x0)

#define M4_IMAGE	       (0x1)

//!4K CHUNK address mask
#define CHUNK_SIZE 4096

//! One ms count value for systick
#define ONE_MS (30030/1)

/* MBR releted Info macros*/

#define MBR_SIGNATURE 						0x5A5A				//MBR magic word

#define COMMON_FLASH_ENABLE_LOC 	0x800015B			//common flash enable bit location in the Dual flash MBR, This location is decides the board is common flash or dual flsh
#define DF_MBR_SIGNATURE_LOC			0x8000000			//MBR magic word location in the Dual flash MBR

#define COMMON_FLASH_ENABLE_LOC1 	0x41B015B			//common flash enable bit location in the common flash MBR, This location is decides the board is common flash or dual flsh
#define CF_MBR_SIGNATURE_LOC			0x41B0000			//MBR magic word location in the common flash MBR

#define DFB0_FLASH_SIZE_LOC							0x8000088		//Flash size location in B0 Dual flash MBR
#define DFB0_FLASH_VENDER_VARIENT_LOC		0x8000092		//VENDER_VARIENT location in B0 Dual flash MBR

#define DFA0_FLASH_SIZE_LOC							0x8000084		//Flash size location in A0 Dual flash MBR
#define DFA0_FLASH_VENDER_VARIENT_LOC		0x8000088		//VENDER_VARIENT location in B0 Dual flash MBR

volatile unsigned int Flash_erase_length=0,dual_flash_bit=0;

unsigned char *common_flash_enabled_ptr = (unsigned char *)COMMON_FLASH_ENABLE_LOC; 	//Reading the common flash enable bit from the Dual flash MBR location
short int *df_mbr_signature_ptr=(short int *)DF_MBR_SIGNATURE_LOC;										//Reading the MBR magic word form the Dual flash MBR location 

volatile unsigned int *dfb0_flash_size_ptr = (unsigned int *)DFB0_FLASH_SIZE_LOC;
volatile unsigned int *dfa0_flash_size_ptr = (unsigned int *)DFA0_FLASH_SIZE_LOC;

volatile unsigned int *dfb0_flash_vendor_varient_ptr = (unsigned int *)DFB0_FLASH_VENDER_VARIENT_LOC;
volatile unsigned int *dfa0_flash_vendor_varient_ptr = (unsigned int *)DFA0_FLASH_VENDER_VARIENT_LOC;
/*********************************************************************
*
*       Types
*
**********************************************************************
*/

typedef struct {
  U32 AddVariablesHere;
} RESTORE_INFO;

static void _FeedWatchdog(void);

/*********************************************************************
*
*       Static data
*
**********************************************************************
*/

static RESTORE_INFO _RestoreInfo;

/*********************************************************************
*
*       Public data
*
**********************************************************************
*/

volatile int PRGDATA_StartMarker __attribute__ ((section ("PrgData")));         // Mark start of <PrgData> segment. Non-static to make sure linker can keep this symbol. Dummy needed to make sure that <PrgData> section in resulting ELF file is present. Needed by open flash loader logic on PC side

const SEGGER_OFL_API SEGGER_OFL_Api __attribute__ ((section ("PrgCode"))) = {   // Mark start of <PrgCode> segment. Non-static to make sure linker can keep this symbol. 
  _FeedWatchdog,
  Init,
  UnInit,
  EraseSector,
  ProgramPage,
  BlankCheck,
  EraseChip,
  Verify,
  SEGGER_OPEN_CalcCRC,
  SEGGER_OPEN_Read,
  SEGGER_OPEN_Program,
  SEGGER_OPEN_Erase,
  SEGGER_OPEN_Start
};

/*Global variable for board ready*/
uint32_t board_ready = 0;

/*********************************************************************
*
*       Static code
*
**********************************************************************
*/

/*********************************************************************
*
*       _FeedWatchdog
*
*  Function description
*    Feeds the watchdog. Needs to be called during RAMCode execution in case of an watchdog is active.
*    In case no handling is necessary, it could perform a dummy access, to make sure that this function is linked in
*/
static void _FeedWatchdog(void) {
  *((volatile int*)&PRGDATA_StartMarker);  // Dummy operation
}

/*********************************************************************
*
*       Public code
*
**********************************************************************
*/
/*
 *  Default QSPI Configurations
 */
void GetQspiConfig_Comm(spi_config_t *spi_config)
{
  memset(spi_config, 0, sizeof(spi_config_t));
#ifdef GIGA_DEVICE
  spi_config->spi_config_1.inst_mode  = SINGLE_MODE; // For Radio Board , instr mode should be in SINGLE_MODE
  spi_config->spi_config_1.addr_mode  = QUAD_MODE;
  spi_config->spi_config_1.dummy_mode = QUAD_MODE;
#else
  spi_config->spi_config_1.inst_mode         = SINGLE_MODE;
  spi_config->spi_config_1.addr_mode         = SINGLE_MODE;
  spi_config->spi_config_1.dummy_mode        = SINGLE_MODE;
#endif
  spi_config->spi_config_1.data_mode       = QUAD_MODE;
  spi_config->spi_config_1.extra_byte_mode = QUAD_MODE;
  spi_config->spi_config_1.prefetch_en     = DIS_PREFETCH;
  spi_config->spi_config_1.dummy_W_or_R    = DUMMY_READS;
  spi_config->spi_config_1.d3d2_data       = 3;
  spi_config->spi_config_1.continuous      = DIS_CONTINUOUS;
#ifdef GIGA_DEVICE 
  spi_config->spi_config_1.read_cmd          = 0xEB;
  spi_config->spi_config_1.flash_type        = GIGA_DEVICE_FLASH;
  spi_config->spi_config_1.no_of_dummy_bytes = 2;
  spi_config->spi_config_1.extra_byte_en     = 1;
#else
  spi_config->spi_config_1.read_cmd          = 0x6B;
  spi_config->spi_config_1.flash_type        = MX_QUAD_FLASH;
  spi_config->spi_config_1.no_of_dummy_bytes = 1;
  spi_config->spi_config_1.extra_byte_en     = 0;
#endif
  spi_config->spi_config_2.auto_mode                   = EN_AUTO_MODE;
  spi_config->spi_config_2.cs_no                       = CHIP_ZERO;
  spi_config->spi_config_2.neg_edge_sampling           = NEG_EDGE_SAMPLING;
  spi_config->spi_config_2.qspi_clk_en                 = QSPI_FULL_TIME_CLK;
  spi_config->spi_config_2.protection                  = DNT_REM_WR_PROT;
  spi_config->spi_config_2.dma_mode                    = NO_DMA;
  spi_config->spi_config_2.swap_en                     = SWAP;
  spi_config->spi_config_2.full_duplex                 = IGNORE_FULL_DUPLEX;
  spi_config->spi_config_2.wrap_len_in_bytes           = NO_WRAP;
  spi_config->spi_config_2.addr_width_valid            = 0;
  spi_config->spi_config_2.addr_width                  = _24BIT_ADDR;
  spi_config->spi_config_2.pinset_valid                = 0;
  spi_config->spi_config_2.dummy_cycles_for_controller = 0;

  spi_config->spi_config_3.xip_mode          = 0;
  spi_config->spi_config_3._16bit_cmd_valid  = 0;
  spi_config->spi_config_3._16bit_rd_cmd_msb = 0;
  spi_config->spi_config_3.reserved          = 0;
  spi_config->spi_config_3.wr_cmd            = 0x2;
  spi_config->spi_config_3.wr_inst_mode      = SINGLE_MODE;
  spi_config->spi_config_3.wr_addr_mode      = SINGLE_MODE;
  spi_config->spi_config_3.wr_data_mode      = SINGLE_MODE;
  spi_config->spi_config_3.dummys_4_jump     = 1;

  spi_config->spi_config_4._16bit_wr_cmd_msb = 0;
  spi_config->spi_config_4.dual_flash_mode   = 0;
  spi_config->spi_config_4.secondary_csn     = 1;
  spi_config->spi_config_4.polarity_mode     = 0;
#ifdef GIGA_DEVICE
  spi_config->spi_config_4.valid_prot_bits = 5;
#else
  spi_config->spi_config_4.valid_prot_bits   = 4;
#endif
  spi_config->spi_config_4.no_of_ms_dummy_bytes = 0;
  spi_config->spi_config_4.continue_fetch_en    = 0;

  spi_config->spi_config_5.block_erase_cmd      = BLOCK_ERASE;
  spi_config->spi_config_5.busy_bit_pos         = 0;
  spi_config->spi_config_5.d7_d4_data           = 0xf;
  spi_config->spi_config_5.dummy_bytes_for_rdsr = 0x0;
  spi_config->spi_config_5.reset_type           = 0x0;

  spi_config->spi_config_6.chip_erase_cmd   = CHIP_ERASE;
  spi_config->spi_config_6.sector_erase_cmd = SECTOR_ERASE;

  spi_config->spi_config_7.status_reg_write_cmd = 0x1;
  spi_config->spi_config_7.status_reg_read_cmd  = 0x5;
}

/*********************************************************************
*
*       Init
*
*  Function description
*    Handles the initialization of the flash module.
*    It is called once per flash programming step (Erase, Program, Verify)
*
*  Parameters
*    Addr: Flash base address
*    Freq: Clock frequency in Hz
*    Func: Specifies the action followed by Init() (e.g.: 1 - Erase, 2 - Program, 3 - Verify / Read)
*
*  Return value
*    == 0  O.K.
*    == 1  Error
*
*  Notes
*    (1) This function is mandatory.
*    (2) Use "noinline" attribute to make sure that function is never inlined and label not accidentally removed by linker from ELF file.
*/
int Init(U32 Addr, U32 Freq, U32 Func) {
//  spi_config_t spi_configs_init;
//	static uint8_t config = 0;
	(void)Addr;
  (void)Freq;
  (void)Func;
  (void)_RestoreInfo;


#ifdef DEBUG_OFL	
if(config == 0)
{	
		*(volatile uint32_t *)0x24048628 |= ((1 << 3));
		*(volatile uint32_t *)0x24048628 &= ~(1 << 4);
		*(volatile uint32_t *)0x24048628 &= ~(0x7);
		
		*(volatile uint32_t *)0x24048624 |= ((1 << 3));
		*(volatile uint32_t *)0x24048624 &= ~(1 << 4);
		*(volatile uint32_t *)0x24048624 &= ~(0x7);
		
		*(volatile uint32_t *)0x24048620 |= ((1 << 3));
		*(volatile uint32_t *)0x24048620 &= ~(1 << 4);
		*(volatile uint32_t *)0x24048620 &= ~(0x7);
	config = 1;
}	


	*(volatile uint32_t *)0x24048620 |= (1<<5);
#endif

#ifdef DEBUG_OFL	
	*(volatile uint32_t *)0x24048620 &= ~(1<<5);
#endif

  SysTick_Config(ONE_MS);
  return (0);
  
}

/*********************************************************************
*
*       UnInit
*
*  Function description
*    Handles the de-initialization of the flash module.
*    It is called once per flash programming step (Erase, Program, Verify)
*
*  Parameters
*    Func  Caller type (e.g.: 1 - Erase, 2 - Program, 3 - Verify)
*
*  Return value
*    == 0  O.K.
*    == 1  Error
*
*  Notes
*    (1) This function is mandatory.
*    (2) Use "noinline" attribute to make sure that function is never inlined and label not accidentally removed by linker from ELF file.
*/
int UnInit(U32 Func) {
  (void)Func;

#ifdef DEBUG_OFL
	*(volatile uint32_t *)0x24048620 |= (1<<5);
	SysTick_Disable();
	*(volatile uint32_t *)0x24048620 &= ~(1<<5);
#else
	SysTick_Disable();
#endif
  return 0;
}

/*********************************************************************
*
*       EraseSector - For Dual Flash
*
*  Function description
*    Erases one flash sector.
*
*  Parameters
*    SectorAddr  Absolute address of the sector to be erased
*
*  Return value
*    == 0  O.K.
*    == 1  Error
*
*  Notes
*    (1) This function is mandatory.
*    (2) Use "noinline" attribute to make sure that function is never inlined and label not accidentally removed by linker from ELF file.
*/
// Will this be applicable for Common Flash?
#if SUPPORT_ERASE_SECTOR
int EraseSector(U32 SectorAddr) {
	(void)SectorAddr;
}
#endif

/*********************************************************************
*
*       ProgramPage
*
*  Function description
*    Programs one flash page.
*
*  Parameters
*    DestAddr  Address to start programming on
*    NumBytes  Number of bytes to program. Guaranteed to be == <FlashDevice.PageSize>
*    pSrcBuff  Pointer to data to be programmed
*
*  Return value
*    == 0  O.K.
*    == 1  Error
*
*  Notes
*    (1) This function is mandatory.
*    (2) Use "noinline" attribute to make sure that function is never inlined and label not accidentally removed by linker from ELF file.
*/
#define TA_RESET_ADDR      0x22000004
int ProgramPage(U32 DestAddr, U32 NumBytes, U8 *pSrcBuff) {

	int32_t status = 1;
  static int32_t x   = 0, size;
  uint32_t Imageheader[HEADER_LENGTH];


	*(uint32_t *)(TA_RESET_ADDR) = 0x0; //put TA in reset
	
#ifdef DEBUG_OFL	
  *(volatile uint32_t *)0x24048624 |= (1<<5);
#endif
memset(Imageheader,'\0', HEADER_LENGTH);
  if (x == 0) 
	{
		memcpy(Imageheader, pSrcBuff, HEADER_LENGTH);
		
    if (!board_ready) {
			
			if((uint32_t)(Imageheader[0] & IMAGE_TYPE_MASK) == (uint32_t)TA_IMAGE)
			{
				status = rsi_device_init(BURN_NWP_FW);
			}
			else
			{
				status = rsi_device_init(RSI_UPGRADE_IMAGE_I_FW);
			}
			
      if (status != RSI_SUCCESS) {
				
        return status;
      }	
    }

    size = Imageheader[2];

  	  size = (size) / CHUNK_SIZE;

    status = rsi_bl_upgrade_firmware(pSrcBuff, NumBytes, 1);
    size--;
    x = 1;
		
#ifdef DEBUG_OFL		
	*(volatile uint32_t *)0x24048624 &= ~(1<<5);
#endif
    return status;
  }

  if (size == 0) {
    status = rsi_bl_upgrade_firmware(pSrcBuff, NumBytes, 2);
    x      = 0;
  } else {
    status = rsi_bl_upgrade_firmware(pSrcBuff, NumBytes, 0);
    size--;
  }
	
#ifdef DEBUG_OFL		
	*(volatile uint32_t *)0x24048624 &= ~(1<<5);
#endif
	
  return status;
}

/*********************************************************************
*
*       BlankCheck
*
*  Function description
*    Checks if a memory region is blank
*
*  Parameters
*    Addr       Address to start checking
*    NumBytes   Number of bytes to be checked
*    BlankData  Blank (erased) value of flash (Most flashes have 0xFF, some have 0x00, some do not have a defined erased value)
*
*  Return value
*    == 0  O.K., blank
*    == 1  O.K., *not* blank
*     < 0  Error
*
*  Notes
*    (1) This function is optional. If not present, the J-Link software will assume that erased state of a sector can be determined via normal memory-mapped readback of sector.
*    (2) Use "noinline" attribute to make sure that function is never inlined and label not accidentally removed by linker from ELF file.
*/
#if SUPPORT_SEGGER_BLANK_CHECK
int BlankCheck(U32 Addr, U32 NumBytes, U8 BlankData) {
  volatile U8* pData;
  //
  // Simply read data from flash and compare against <BlankData>
  //
  _FeedWatchdog();
  pData = (volatile U8*)Addr;
  do {
    if (*pData++ != BlankData) {
      return 1;
    }
  } while (--NumBytes);
  return 0;
}
#endif

/*********************************************************************
*
*       SEGGER_OPEN_CalcCRC
*
*  Function description
*    Calculates the CRC over a specified number of bytes
*    Even more optimized version of Verify() as this avoids downloading the compare data into the RAMCode for comparison.
*    Heavily reduces traffic between J-Link software and target and therefore speeds up verification process significantly.
*
*  Parameters
*    CRC       CRC start value
*    Addr      Address where to start calculating CRC from
*    NumBytes  Number of bytes to calculate CRC on
*    Polynom   Polynom to be used for CRC calculation
*
*  Return value
*    CRC
*
*  Notes
*    (1) This function is optional
*    (2) Use "noinline" attribute to make sure that function is never inlined and label not accidentally removed by linker from ELF file.
*/
U32 SEGGER_OPEN_CalcCRC(U32 CRC, U32 Addr, U32 NumBytes, U32 Polynom) {
	(void)CRC;
	(void)Addr;
	(void)NumBytes;
	(void)Polynom;
  //CRC = SEGGER_OFL_Lib_CalcCRC(&SEGGER_OFL_Api, CRC, Addr, NumBytes, Polynom);   // Use lib function from SEGGER by default. Pass API pointer to it because it may need to call the read function (non-memory mapped flashes)
  return CRC;
}

/*********************************************************************
*
*       SEGGER_OPEN_Program
*
*  Function description
*    Optimized variant of ProgramPage() which allows multiple pages to be programmed in 1 RAMCode call.
*
*  Parameters
*    DestAddr  Address to start flash programming at.
*    NumBytes  Number of bytes to be program. Guaranteed to be multiple of <FlashDevice.PageSize>
*    pSrcBuff  Pointer to data to be programmed
*
*  Return value
*    == 0  O.K.
*    == 1  Error
*
*  Notes
*    (1) This function is optional. If not present, the J-Link software will use ProgramPage()
*    (2) Use "noinline" attribute to make sure that function is never inlined and label not accidentally removed by linker from ELF file.
*/
#if SUPPORT_SEGGER_OPEN_PROGRAM
int SEGGER_OPEN_Program(U32 DestAddr, U32 NumBytes, U8 *pSrcBuff) {
  U32 NumPages;
  int r;

  NumPages = (NumBytes >> PAGE_SIZE_SHIFT);
  r = 0;
  do {
    r = ProgramPage(DestAddr, (1uL << PAGE_SIZE_SHIFT), pSrcBuff);
    if (r < 0) {
      return r;
    }
    DestAddr += (1uL << PAGE_SIZE_SHIFT);
    pSrcBuff += (1uL << PAGE_SIZE_SHIFT);
  } while (--NumPages);
  return r;
}
#endif
/*********************************************************************
*
*       Verify
*
*  Function description
*    Verifies flash contents.
*    Usually not compiled in. Only needed for non-memory mapped flashes.
*
*  Parameters
*    Addr      Address to start verify on
*    NumBytes  Number of bytes to verify
*    pBuff     Pointer data to compare flash contents to
*
*  Return value
*    == (Addr + NumBytes): O.K.
*    != (Addr + NumBytes): *not* O.K. (ideally the fail address is returned)
*
*  Notes
*    (1) This function is optional. If not present, the J-Link software will assume that flash memory can be verified via memory-mapped readback of flash contents.
*    (2) Use "noinline" attribute to make sure that function is never inlined and label not accidentally removed by linker from ELF file.
*/
#if SUPPORT_NATIVE_VERIFY
U32 Verify(U32 Addr, U32 NumBytes, U8 *pBuff) {
  
	#ifndef DUAL_FLASH
		return RSI_FLASH_Verify(Addr, pBuff, NumBytes);
	#else
	unsigned char *pFlash;
  unsigned long r;
	
#ifdef DEBUG_OFL		
	*(volatile uint32_t *)0x24048628 |= (1<<5);
#endif
  pFlash = (unsigned char *)Addr;
  r = Addr + NumBytes;
  do {
      if (*pFlash != *pBuff) {
        r = (unsigned long)pFlash;
        break;
      }
      pFlash++;
      pBuff++;
  } while (--NumBytes);

#ifdef DEBUG_OFL	
	*(volatile uint32_t *)0x24048628 &= ~(1<<5);
#endif
  return r;
	#endif
}
#endif

/*********************************************************************
*
*       EraseChip
*
*  Function description
*    Erases the entire flash.
*
*  Return value
*    == 0  O.K.
*    == 1  Error
*
*  Notes
*    (1) This function is optional. If not present, J-Link will always use EraseSector() for erasing.
*    (2) Use "noinline" attribute to make sure that function is never inlined and label not accidentally removed by linker from ELF file.
*/
#if SUPPORT_ERASE_CHIP //Enable For Common Flash

int EraseChip(void) {
	
 	int status =RSI_SUCCESS;

		//Send the common flash erase command to the TA
		status = rsi_device_init(ERASE_COMMON_FLASH);
		if(status==RSI_SUCCESS)
			return RSI_OK;
	return 0;
}
#endif

/*********************************************************************
*
*       SEGGER_OPEN_Read
*
*  Function description
*    Reads a specified number of bytes from flash into the provided buffer.
*    Usually not compiled in. Only needed for non-memory mapped flashes.
*
*  Parameters
*    Addr      Address to start reading from
*    NumBytes  Number of bytes to read
*    pDestBuff Pointer to buffer to store read data
*
*  Return value
*    >= 0: O.K., NumBytes read
*    <  0: Error
*
*  Notes
*    (1) This function is optional. If not present, the J-Link software will assume that a normal memory-mapped read can be performed to read from flash.
*    (2) Use "noinline" attribute to make sure that function is never inlined and label not accidentally removed by linker from ELF file.
*/
#if SUPPORT_NATIVE_READ_FUNCTION
int SEGGER_OPEN_Read(U32 Addr, U32 NumBytes, U8 *pDestBuff) {
  //
  // Read function
  // Add your code here...
  //
  //_FeedWatchdog();
  return NumBytes;
}
#endif

/*********************************************************************
*
*       SEGGER_OPEN_Erase
*
*  Function description
*    Erases one or more flash sectors.
*    The implementation from this template only works on flashes that have uniform sectors.
*
*  Notes
*    (1) This function can rely on that at least one sector will be passed
*    (2) This function must be able to handle multiple sectors at once
*    (3) This function can rely on that only multiple sectors of the same sector
*        size will be passed. (e.g. if the device has two sectors with different
*        sizes, the DLL will call this function two times with NumSectors = 1)
*
*  Parameters
*    SectorAddr:  Address of the start sector to be erased
*    SectorIndex: Index of the start sector to be erased (1st sector handled by this flash bank: SectorIndex == 0)
*    NumSectors:  Number of sectors to be erased. Min. 1
*
*  Return value
*    == 0  O.K.
*    == 1  Error
*
*  Notes
*    (1) This function is optional. If not present, the J-Link software will use EraseSector()
*    (2) Use "noinline" attribute to make sure that function is never inlined and label not accidentally removed by linker from ELF file.
*/
#if SUPPORT_SEGGER_OPEN_ERASE
int SEGGER_OPEN_Erase(U32 SectorAddr, U32 SectorIndex, U32 NumSectors) {
  int r;

  (void)SectorIndex;
  _FeedWatchdog();
  r = 0;
  do {
    r = EraseSector(SectorAddr);
    if (r) {
      break;
    }
    SectorAddr += (1 << SECTOR_SIZE_SHIFT);
  } while (--NumSectors);
  return r;
}
#endif

/*********************************************************************
*
*       SEGGER_OPEN_Start
*
*  Function description
*    Starts the turbo mode of flash algo.
*    Currently only available for Cortex-M based targets.
*/
#if SUPPORT_TURBO_MODE
void SEGGER_OPEN_Start(volatile struct SEGGER_OPEN_CMD_INFO* pInfo) {
  SEGGER_OFL_Lib_StartTurbo(&SEGGER_OFL_Api, pInfo);
}
#endif

/**************************** End of file ***************************/
