/*******************************************************************************
* @file  Flash_Intf.c
* @brief 
*******************************************************************************
* # License
* <b>Copyright 2020 Silicon Laboratories Inc. www.silabs.com</b>
*******************************************************************************
*
* The licensor of this software is Silicon Laboratories Inc. Your use of this
* software is governed by the terms of Silicon Labs Master Software License
* Agreement (MSLA) available at
* www.silabs.com/about-us/legal/master-software-license-agreement. This
* software is distributed to you in Source Code format and is governed by the
* sections of the MSLA applicable to Source Code.
*
******************************************************************************/
#include "Flash_Intf.h"
#include "FlashOS.h"
#include "rsi_qspi_flm.h"
#include "RS1xxxx_9117.h"


volatile unsigned int DualFlash_erase_length=0;

void qspi_spi_init(qspi_reg_t *qspi_reg,spi_config_t *spi_config,uint32_t flash_init_req,uint32_t wr_reg_delay_ms,uint8_t fifo_thrsld);
void qspi_spi_erase(qspi_reg_t *qspi_reg,spi_config_t *spi_config,uint32_t erase_cmd,uint32_t blk_sec_addr,uint32_t dis_hw_ctrl,uint32_t wr_reg_delay_ms);

void qspi_usleep(uint32_t delay_us);

void egpio_set_dir(EGPIO_Type *pEGPIO, uint8_t port, uint8_t pin, boolean_t dir);

void egpio_set_pin(EGPIO_Type *pEGPIO, uint8_t port, uint8_t pin, uint8_t val);

typedef TIMERS_Type RSI_TIMERS_T;


#define FLASH_SIZE_LOC	0x8000088		//Flash size location in Dual flash MBR

/**
 * @fn           void egpio_set_dir(EGPIO_Type *pEGPIO, uint8_t port, uint8_t pin, boolean_t dir)
 * @brief        This API is used to set the EGPIO direction(Direction of the GPIO pin. '1' for INPUT, '0' for OUTPUT)
 * @param[in]    pEGPIO  : Pointer to the EGPIO register instance
 * @param[in]    port    : GPIO port number
 * @param[in]    pin     : GPIO pin number
 * @param[in]    dir     : boolean type pin direction
 *                \n '0' : Output
 *                \n '1' : Input
 * @return       None
 */
void egpio_set_dir(EGPIO_Type *pEGPIO, uint8_t port, uint8_t pin, boolean_t dir)
{
  pEGPIO->PIN_CONFIG[(port * 16) + pin].GPIO_CONFIG_REG_b.DIRECTION = dir;
}

/**
 * @fn           void egpio_set_pin(EGPIO_Type *pEGPIO, uint8_t port, uint8_t pin, uint8_t val)
 * @brief        This API is used to set the GPIO pin value.It Loads 0th bit on to the pin on write &
 *                reads the value on pin on read into 0th bit
 * @param[in]    pEGPIO  : Pointer to the EGPIO register instance
 * @param[in]    port    : GPIO port number
 * @param[in]    pin     : GPIO pin number
 * @param[in]    val     : value to be set for the pin
 *                \n '0' : Logic on Pin
 *                \n '1' : Logic on Pin
 * @return       None
 */
void egpio_set_pin(EGPIO_Type *pEGPIO, uint8_t port, uint8_t pin, uint8_t val)
{
  pEGPIO->PIN_CONFIG[(port * 16) + pin].BIT_LOAD_REG = val;
}

uint32_t  RSI_FLASH_UnInitialize(void)
{
	/* This function not supported so always returns RSI_OK */
	return RSI_OK;
}

void egpio_pad_selection_disable(uint8_t padNum)//required
{
  if (padNum < 22) {
    /*(tass_m4ss_gpio_sel)PAD selection (0 t0 21) A value of 0 on this gives control to TASS(by default it is 0 means ta control) */
    PADSELECTION &= ~BIT(padNum);
  }
	#ifdef CHIP_9117
		else {
			/*(tass_m4ss_gpio_sel)PAD selection (22 t0 33) A value of 0 on this gives control to TASS(by default it is 0 means ta control) */
			PADSELECTION_1 &= ~BIT(padNum - 22);
		}
	#endif
}

void egpio_set_pin_mux(EGPIO_Type *pEGPIO, uint8_t port, uint8_t pin, uint8_t mux)//required
{
  pEGPIO->PIN_CONFIG[(port * 16) + pin].GPIO_CONFIG_REG_b.MODE = (mux);
}

void egpio_pad_selection_enable(uint8_t padNum)//required
{
  if (padNum < 22) {
    /*(tass_m4ss_gpio_sel)PAD selection (0 t0 21) A value of 1 on this gives control to M4SS(by default it is 0 means ta control) */
    PADSELECTION |= BIT(padNum);
  }
	#ifdef CHIP_9117
  else {
    /*(tass_m4ss_gpio_sel)PAD selection (22 t0 33) A value of 1 on this gives control to M4SS(by default it is 0 means ta control) */
    PADSELECTION_1 |= BIT(padNum - 22);
  }
	#endif
}

/**
 * @fn           void egpio_pad_receiver_enable(uint8_t u8GpioNum)
 * @brief        This API is used to enable the receiver enable bit(REN)
 * @param[in]    u8GpioNum  :  GPIO num to be use
 * @return       None
 */
void egpio_pad_receiver_enable(uint8_t u8GpioNum)//required
{
  /*REN enable bit(this should be enable)*/
  PAD_CONFIG_REG(u8GpioNum) |= (0x1 << 4);
}



void qspi_gpio_revert_m4(void)
{
  uint32_t gpio_pin_no = 0;
  uint32_t gpio_mode = VAL_15;
  uint32_t i;

  //! Enable QSPI  clock and hclock
  CLK_ENABLE_SET_2_REG_QSPI = ((1 << VAL_12)|(1 << VAL_11));
  egpio_pad_selection_disable(0);
  for(i = 0; i < VAL_6; i++) {
     egpio_set_pin_mux(EGPIO, QSPI_PINS_PORT, (gpio_pin_no + i), gpio_mode);
   }
 }

void GetQspiConfig(spi_config_t *spi_config)
{
	
	memset(spi_config, 0, sizeof(spi_config_t));
	spi_config->spi_config_1.inst_mode         = SINGLE_MODE;
	spi_config->spi_config_1.addr_mode         = SINGLE_MODE;
	spi_config->spi_config_1.data_mode         = SINGLE_MODE;
	spi_config->spi_config_1.dummy_mode        = SINGLE_MODE;
	spi_config->spi_config_1.extra_byte_mode   = SINGLE_MODE;
	spi_config->spi_config_1.prefetch_en       = DIS_PREFETCH;
	spi_config->spi_config_1.dummy_W_or_R      = DUMMY_READS;
	spi_config->spi_config_1.extra_byte_en     = 0;
	spi_config->spi_config_1.d3d2_data         = 3;
	spi_config->spi_config_1.continuous        = DIS_CONTINUOUS;
	spi_config->spi_config_1.read_cmd          = READ;
	spi_config->spi_config_1.flash_type        = SST_SPI_FLASH;
	spi_config->spi_config_1.no_of_dummy_bytes = 0;


	spi_config->spi_config_2.auto_mode         = EN_AUTO_MODE;
	spi_config->spi_config_2.cs_no             = CHIP_ZERO;
	spi_config->spi_config_2.neg_edge_sampling = NEG_EDGE_SAMPLING;
	spi_config->spi_config_2.qspi_clk_en       = QSPI_FULL_TIME_CLK;
	spi_config->spi_config_2.protection        = DNT_REM_WR_PROT;
	spi_config->spi_config_2.dma_mode          = NO_DMA;
	spi_config->spi_config_2.swap_en           = SWAP;
	spi_config->spi_config_2.full_duplex       = IGNORE_FULL_DUPLEX;
	spi_config->spi_config_2.wrap_len_in_bytes = NO_WRAP;
	spi_config->spi_config_2.addr_width_valid  = 0;
	spi_config->spi_config_2.addr_width        = _24BIT_ADDR;
	spi_config->spi_config_2.pinset_valid      = 0;
	spi_config->spi_config_2.flash_pinset      = GPIO_58_TO_63;
	spi_config->spi_config_2.dummy_cycles_for_controller = 0;

	spi_config->spi_config_3.xip_mode                = 0;
	spi_config->spi_config_3._16bit_cmd_valid        = 0;
	spi_config->spi_config_3._16bit_rd_cmd_msb       = 0;
	#ifdef CHIP_9118
	spi_config->spi_config_3.ddr_mode_en             = 0;
	#endif
	spi_config->spi_config_3.wr_cmd                  = 0x2;
	spi_config->spi_config_3.wr_inst_mode            = SINGLE_MODE;
	spi_config->spi_config_3.wr_addr_mode            = SINGLE_MODE;
	spi_config->spi_config_3.wr_data_mode            = SINGLE_MODE;
	spi_config->spi_config_3.dummys_4_jump           = 1;

	spi_config->spi_config_4._16bit_wr_cmd_msb       = 0;
	#ifdef CHIP_9118
	spi_config->spi_config_4.qspi_manual_ddr_phasse  = 0;
	spi_config->spi_config_4.ddr_data_mode           = 0;
	spi_config->spi_config_4.ddr_addr_mode           = 0;
	spi_config->spi_config_4.ddr_inst_mode           = 0;
	spi_config->spi_config_4.ddr_dummy_mode          = 0;
	spi_config->spi_config_4.ddr_extra_byte          = 0;
	#endif
	spi_config->spi_config_4.dual_flash_mode         = 0;
	spi_config->spi_config_4.secondary_csn           = 1;
	spi_config->spi_config_4.polarity_mode           = 0;
	spi_config->spi_config_4.valid_prot_bits         = 4;
	spi_config->spi_config_4.no_of_ms_dummy_bytes    = 0;
	#ifdef CHIP_9118
	spi_config->spi_config_4.ddr_dll_en              = 0;
	#endif
	spi_config->spi_config_4.continue_fetch_en       = 0;


	spi_config->spi_config_5.block_erase_cmd           = BLOCK_ERASE;
	spi_config->spi_config_5.busy_bit_pos              = 0;
	spi_config->spi_config_5.d7_d4_data                = 0xf;
	spi_config->spi_config_5.dummy_bytes_for_rdsr      = 0x0;
	spi_config->spi_config_5.reset_type                = 0x0;

	spi_config->spi_config_6.chip_erase_cmd        = CHIP_ERASE;
	spi_config->spi_config_6.sector_erase_cmd      = SECTOR_ERASE;

	spi_config->spi_config_7.status_reg_write_cmd  = 0x1;
	spi_config->spi_config_7.status_reg_read_cmd   = 0x5;
	
}

uint32_t RSI_FLASH_Initialize(void)
{
	/*Init the QSPI configurations structure */
	spi_config_t spi_configs_init; 
	*(uint32_t *)(TA_RESET_ADDR) = 0x1; //put TA in reset
	qspi_gpio_revert_m4();
	GetQspiConfig(&spi_configs_init);
	
	#ifdef EXTERNAL_FLASH
	RSI_EGPIO_PadSelectionEnable(PadSelectionEnable_1);
	RSI_EGPIO_PadSelectionEnable(PadSelectionEnable_2);
	#else
	egpio_pad_selection_enable(PadSelectionEnable_CLK);
	egpio_pad_selection_enable(PadSelectionEnable_D0);
	egpio_pad_selection_enable(PadSelectionEnable_D1);
	egpio_pad_selection_enable(PadSelectionEnable_CSN0);
	egpio_pad_selection_enable(PadSelectionEnable_D2);
	egpio_pad_selection_enable(PadSelectionEnable_D3);
	
	/*Receive enable for QSPI GPIO*/
	egpio_pad_receiver_enable(M4SS_QSPI_CLK);
	egpio_pad_receiver_enable(M4SS_QSPI_CSN0);
	egpio_pad_receiver_enable(M4SS_QSPI_D0);
	egpio_pad_receiver_enable(M4SS_QSPI_D1);
	egpio_pad_receiver_enable(M4SS_QSPI_D2);
	egpio_pad_receiver_enable(M4SS_QSPI_D3);
	
	/*Set GPIO pin MUX for QSPI*/
	egpio_set_pin_mux(EGPIO, QSPI_PINS_PORT, M4SS_QSPI_CLK, QSPI_MODE);
	egpio_set_pin_mux(EGPIO, QSPI_PINS_PORT, M4SS_QSPI_CSN0 , QSPI_MODE);
	egpio_set_pin_mux(EGPIO, QSPI_PINS_PORT, M4SS_QSPI_D0 , QSPI_MODE);
	egpio_set_pin_mux(EGPIO, QSPI_PINS_PORT, M4SS_QSPI_D1 , QSPI_MODE);
	egpio_set_pin_mux(EGPIO, QSPI_PINS_PORT, M4SS_QSPI_D2 , QSPI_MODE);
	egpio_set_pin_mux(EGPIO, QSPI_PINS_PORT, M4SS_QSPI_D3 , QSPI_MODE);
	/*Initialize QSPI*/
	qspi_spi_init((qspi_reg_t *)(M4_QSPI_BASE_ADDRESS),&spi_configs_init,1,0,0);
	/* Above function always complete sucessfully */
	#endif
	return RSI_OK;
}



uint32_t RSI_FLASH_EraseSector(uint32_t sector_address)
{
	spi_config_t spi_configs_erase; 
	GetQspiConfig(&spi_configs_erase);
		/* The erase function does the erase and takes care of any missing configurations to successfully erase the sector*/
	qspi_spi_erase((qspi_reg_t *)(M4_QSPI_BASE_ADDRESS),&spi_configs_erase,SECTOR_ERASE, sector_address, 1, 0);
	return RSI_OK;   
}

uint32_t RSI_FLASH_Erasechip()
{
	volatile int sector_off_set_addr = 0;
//	volatile unsigned int *flash_size_ptr = (unsigned int *)FLASH_SIZE_LOC; // Flash size location in the MBR 
	
	spi_config_t spi_configs_erase_chip; 
	GetQspiConfig(&spi_configs_erase_chip);//TODO : Read the configs from the MBR

  /*Note : This function will not erase the MBR content of the flash but erases the rest of the flash from 68K(INIT_FLASH_OFFSET)*/	
	sector_off_set_addr = INIT_FLASH_OFFSET;
	
	
//	/*flash_size_ptr:- variable is used to read the flash size from the MBR*/
//	
//	if(flash_size_ptr[0]==0x20)				//0x20 means 4mb external flash
//	{
		DualFlash_erase_length=0x003EEFFF; 	//4 MB Dual flash erase length (0x3F_FFFF-0x11000(68KB)) , not erasing the MBR size(68KB)
//	}
//	else if(flash_size_ptr[0]==0x40)	//0x40 means 8MB external flash
//	{
//		DualFlash_erase_length=0x007EEFFF; 	//8 MB Dual flash erase length (0x7F_FFFF-0x11000(68KB)) , not erasing the MBR size(68KB)
//	}
//	
//	else if(flash_size_ptr[0]==0x80)	//0x80 means 16MB external flash
//	{
//		DualFlash_erase_length=0x00FEEFFF; 	//16 MB Dual flash erase length (0xFF_FFFF-0x11000(68KB)) , not erasing the MBR size(68KB)
//	}
//	else
//	{
//		return 1;
//	}
//	
	do{
//		_FeedWatchdog();
		qspi_spi_erase((qspi_reg_t *)(M4_QSPI_BASE_ADDRESS),&spi_configs_erase_chip,SECTOR_ERASE, sector_off_set_addr, 1, 0);
		
		//! Increment the sector address by 4K
		sector_off_set_addr = (sector_off_set_addr + (4*SIZE_1K));
		//! This loop will be terminated once it reaches to the Flash_erase_length
	
	}while(sector_off_set_addr < DualFlash_erase_length);
	// The erase function does the erase and takes care of any missing configurations to successfully erase the sector of the flash
	return RSI_OK;
}
uint32_t checksum_addition(uint8_t *buf, uint32_t size, uint32_t prev_sum)
{
	uint32_t sum = prev_sum;
	uint32_t cnt;
	uint32_t cnt_limit;
	uint32_t dword;

	if (size == 0)
	{
 		return sum;
	}

	cnt_limit = (size & (~0x3));
	/* Accumulate checksum */
	for (cnt = 0; cnt < cnt_limit; cnt += 4)
	{
		dword = *(uint32_t *) &buf[cnt];
		sum += dword;
		if(sum < dword)
		{
			/* In addition operation, if result is lesser than any one of the operand
			 * it means carry is generated. 
			 * Incrementing the sum to get ones compliment addition */

			sum++;
		}
	}
	
	/* Handle non dword-sized case */
  if(size & 0x3) {
		dword = DWORD_ADDR_COMP;
		dword = ~(dword << (8 * (size & 0x3)));
		/* Keeping only valid bytes and making upper bytes zeroes. */
		dword = (*(uint32_t *) &buf[cnt]) & dword;
		sum += dword;
		if(sum < dword)
		{
			sum++;
		}
	}

	return ~sum;
//	return RSI_OK;
}


uint32_t RSI_FLASH_Verify(uint32_t address, unsigned char *data, uint32_t length)
{
	uint32_t check_sum = 0, i; 
	uint8_t *buffer = (uint8_t *)address;
	
	if((address & FLASH_VERIFY_ADDR)== INIT_FLASH_OFFSET)	{
		check_sum  = checksum_addition(data , CRC_LEN_VAL , 1);
		memcpy(&data[CRC_LEN_VAL] ,(uint8_t*)&check_sum,4);	
		for(i=0;i<length;i++)
			if(buffer[i] != data[i])
				break;
	} else {
		for(i=0;i<length;i++)
			if(buffer[i] != data[i])
				break;
	}
	return (address+i);
	//return RSI_OK;
}

uint32_t RSI_FLASH_Read(uint32_t address, unsigned char *data, uint32_t length, uint8_t auto_mode)
{
	/* This function not supported so always returns zero */
	return RSI_OK;
}


void initialise_m4_efuse_in_io_mode()
{
  M4SS_CLK_ENABLE_SET_3_REG = EFUSE_CLK_BIT | EFUSE_PCLK_BIT;
  // Program Timing params
  M4_EFUSE_RD_TMNG_PARAM_REG = 0x5A2;
  // Program Mem map mode to byte read
  M4_EFUSE_MEM_MAP_LENGTH = 0;
  // Enable Efuse
  M4_EFUSE_CTRL_REG = 0x1;
}

void rsi_cmemcpy(uint8_t *dst, uint8_t *src, uint32_t len)
{
  while (len--) {
    *dst++ = *src++;
  }
}



void qspi_write_to_flash(qspi_reg_t *qspi_reg, uint32_t len_in_bits, uint32_t cmd_addr_data, uint32_t cs_no)
{
  // length of the word to be programmed
  qspi_reg->QSPI_MANUAL_WRITE_DATA_2_REG = len_in_bits;
  // cmd/data/addr to be written
  qspi_reg->QSPI_MANUAL_RD_WR_DATA_REG = cmd_addr_data;
  qspi_reg->QSPI_MANUAL_CONFIG_REG =
    (((qspi_reg->QSPI_MANUAL_CONFIG_REG & ~0x6007) & ~TAKE_LEN_FRM_REG) | WRITE_TRIGGER | (cs_no << 13));
  // wait till QSPI becomes idle
  while (qspi_reg->QSPI_STATUS_REG & 1)
    ;
}


void qspi_switch_qspi2(qspi_reg_t *qspi_reg, uint32_t mode, uint32_t cs_no)
{
  uint32_t qspi_manual_config_reg;

  // clearing existing read count and bus mode
  qspi_reg->QSPI_MANUAL_CONFIG_REG &= ~0x1ffe;
  if (cs_no == CHIP_ZERO) {
    // for chip select zero, configure bus mode
    qspi_reg->QSPI_BUS_MODE_REG = ((qspi_reg->QSPI_BUS_MODE_REG & ~0x6) | (mode << 1));
  } else {
    // read the reg
    qspi_manual_config_reg = qspi_reg->QSPI_MANUAL_CONFIG_2_REG;
    // mask the bus mode bits
    qspi_manual_config_reg &= ~(0x3 << (((cs_no - 1) * 2) + 8));
    // write the bus mode bits
    qspi_manual_config_reg |= ((mode) << (((cs_no - 1) * 2) + 8));
    // write back to the register
    qspi_reg->QSPI_MANUAL_CONFIG_2_REG = qspi_manual_config_reg;
  }
}

uint32_t qspi_wait_flash_status_Idle(qspi_reg_t *qspi_reg, spi_config_t *spi_config, uint32_t wr_reg_delay_ms)
{
  uint32_t busy, tmp_dummy;
  uint32_t qspi_operational_mode;
  uint32_t cs_no, flash_type, status_reg_read_cmd, busy_bit_pos;
  uint32_t cmd_len, dummy_bytes = 0;
  volatile uint32_t flash_status;

  cs_no               = spi_config->spi_config_2.cs_no;
  flash_type          = spi_config->spi_config_1.flash_type;
  status_reg_read_cmd = spi_config->spi_config_7.status_reg_read_cmd;

  qspi_operational_mode = QSPI_MANUAL_BUS_SIZE(cs_no);

  // SST_QFLASH supports status reg read in Quad mode
  busy_bit_pos = spi_config->spi_config_5.busy_bit_pos;
  if (QSPI_DUAL_FLASH_MODE) {
    if ((busy_bit_pos < 4)) {
      busy = (BIT(busy_bit_pos)) | (BIT(busy_bit_pos) << 4);
    } else {
      busy_bit_pos -= 4;
      busy = (((BIT(busy_bit_pos)) | (BIT(busy_bit_pos) << 4)) << 8);
    }
  } else {
    // flashes which support status reg read only in SPI mode
    busy = BIT(busy_bit_pos);
  }
  // ensure previous operation is terminated
  DEASSERT_CSN;

  // status reg read cmd

  cmd_len = ((spi_config->spi_config_3._16bit_cmd_valid) && (qspi_operational_mode == OCTA_MODE)) ? 16 : 8;
  if ((cmd_len == 8) && (spi_config->spi_config_3._16bit_cmd_valid)) {
    status_reg_read_cmd >>= 8;
  }
#if 0
	if(cmd_len > 8) {
		status_reg_read_cmd = status_reg_read_cmd | (RDSR << 8);
	}
#endif

  qspi_write_to_flash(qspi_reg, cmd_len, status_reg_read_cmd, cs_no);

  if ((flash_type == MX_OCTA_FLASH) && (qspi_operational_mode == OCTA_MODE)) {
    qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN * 4, 0x00, cs_no);
  }

  if ((qspi_operational_mode == OCTA_MODE) || (qspi_operational_mode == QUAD_MODE)) {
    dummy_bytes = spi_config->spi_config_5.dummy_bytes_for_rdsr;
  }
  if (dummy_bytes) {
    do {
      tmp_dummy = (dummy_bytes & 0x3) ? (dummy_bytes & 3) : 4;

      qspi_write_to_flash(qspi_reg, (QSPI_8BIT_LEN * tmp_dummy), 0x00, cs_no);
      dummy_bytes -= tmp_dummy;

    } while (dummy_bytes);
  }

  if (spi_config->spi_config_4.dual_flash_mode) {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 &= ~BIT(19);
    qspi_switch_qspi2(qspi_reg, OCTA_MODE, cs_no);
  }

  do {
#ifdef CHIP_9118
    if (QSPI_DATA_DDR_MODE || QSPI_DUAL_FLASH_MODE)
#else
    if (QSPI_DUAL_FLASH_MODE)
#endif
    {
      READ_4M_FLASH(2, cs_no, _16BIT);
    } else {
      READ_4M_FLASH(1, cs_no, 0);
    }
    // wait till the fifo empty is deasserted
    while (qspi_reg->QSPI_STATUS_REG & QSPI_FIFO_EMPTY_RFIFO_S)
      ;

      // read status
#ifdef CHIP_9118
    if (QSPI_DATA_DDR_MODE || QSPI_DUAL_FLASH_MODE)
#else
    if (QSPI_DUAL_FLASH_MODE)
#endif
    {
      flash_status = (uint16_t)qspi_reg->QSPI_MANUAL_RD_WR_DATA_REG;
    } else {
      flash_status = (uint8_t)qspi_reg->QSPI_MANUAL_RD_WR_DATA_REG;
    }
    // if flash is busy, continue reading till it becomes idle
  } while (flash_status & busy);

  if (spi_config->spi_config_4.dual_flash_mode) {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 |= BIT(19);
  }
  qspi_switch_qspi2(qspi_reg, qspi_operational_mode, cs_no);
  DEASSERT_CSN;

  return flash_status;
}

void qspi_enable_status_reg_write(qspi_reg_t *qspi_reg, uint32_t flash_type, spi_config_t *spi_config, uint32_t cs_no)
{
  uint32_t qspi_operational_mode;
  qspi_operational_mode = QSPI_MANUAL_BUS_SIZE(cs_no);
  // write enable added by Samson after verifying with LiteFi
  qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN, cs_no);
  if (spi_config->spi_config_3._16bit_cmd_valid && (qspi_operational_mode == OCTA_MODE)) {
    qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN2, cs_no);
  }
  DEASSERT_CSN;
}

void qspi_status_reg_write(qspi_reg_t *qspi_reg,
                           uint32_t write_value,
                           spi_config_t *spi_config,
                           uint32_t wr_reg_delay_ms)
{
  uint32_t cs_no;
  uint32_t cmd_len;
  uint32_t flash_type, qspi_operational_mode, status_reg_write_cmd;
  uint32_t send_16bit_data = 0, send_24bit_data = 0, do_non_volatile_write = 0;

  cs_no                 = spi_config->spi_config_2.cs_no;
  flash_type            = spi_config->spi_config_1.flash_type;
  qspi_operational_mode = QSPI_MANUAL_BUS_SIZE(cs_no);
  status_reg_write_cmd  = spi_config->spi_config_7.status_reg_write_cmd;

  if (wr_reg_delay_ms & BIT(29)) {
    wr_reg_delay_ms &= ~BIT(29);
    wr_reg_delay_ms &= ~BIT(31);
    send_24bit_data = 1;
  }
  if (wr_reg_delay_ms & BIT(31)) {
    wr_reg_delay_ms &= ~BIT(31);
    send_16bit_data = 1;
  }

  if (wr_reg_delay_ms & BIT(30)) {
    wr_reg_delay_ms &= ~BIT(30);
    do_non_volatile_write = 1;
  }
  if (spi_config->spi_config_4.dual_flash_mode) {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 |= BIT(19);
  }
  if (((flash_type == SST_SPI_FLASH) || (flash_type == GIGA_DEVICE_FLASH)) && (!do_non_volatile_write)) {
    // cmd to enable status reg write
    qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, EWSR, cs_no);
    DEASSERT_CSN;
  } else {
    // enable status reg write
    qspi_enable_status_reg_write(qspi_reg, spi_config->spi_config_1.flash_type, spi_config, cs_no);
#if 0       
		if ((flash_type == WBOND_QUAD_FLASH)) {
			qspi_func->write_to_flash(qspi_reg, CMD_LEN, WREN, cs_no);   
			DEASSERT_CSN;
		}
#endif
  }

  cmd_len = ((flash_type == MX_OCTA_FLASH) && (qspi_operational_mode == OCTA_MODE)) ? 16 : 8;
  cmd_len = ((spi_config->spi_config_3._16bit_cmd_valid) && (qspi_operational_mode == OCTA_MODE)) ? 16 : 8;
  if ((cmd_len == 8) && (spi_config->spi_config_3._16bit_cmd_valid)) {
    status_reg_write_cmd >>= 8;
  }
  qspi_write_to_flash(qspi_reg, cmd_len, status_reg_write_cmd, cs_no);

  if ((flash_type == MX_OCTA_FLASH) && (qspi_operational_mode == OCTA_MODE)) {
    qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN * 4, 0x00, cs_no);
  }

  if ((flash_type == WBOND_QUAD_FLASH) || (flash_type == GIGA_DEVICE_FLASH) || send_16bit_data) {
    cmd_len = 16;
  } else {
    cmd_len = 8;
  }
  if (send_24bit_data) {
    cmd_len = 24;
  }
// status reg is written with 0 to remove protection of memory
#ifdef CHIP_9118
  if (QSPI_DATA_DDR_MODE) {
    qspi_write_to_flash(qspi_reg, QSPI_16BIT_LEN, (write_value << 8), cs_no);
  } else {
    qspi_write_to_flash(qspi_reg, cmd_len, write_value, cs_no);
  }
#else
  qspi_write_to_flash(qspi_reg, cmd_len, write_value, cs_no);
#endif
  DEASSERT_CSN;

  // wait till qspi_status_reg_write is done
  qspi_wait_flash_status_Idle(qspi_reg, spi_config, wr_reg_delay_ms);

  if (spi_config->spi_config_4.dual_flash_mode) {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 &= ~BIT(19);
  }
}

uint32_t qspi_flash_reg_read(qspi_reg_t *qspi_reg, uint8_t reg_read_cmd, uint32_t cs_no, spi_config_t *spi_config)
{
  uint32_t rd_config;
  uint32_t read_len = 1;
#ifdef CHIP_9117
  rd_config = 0;
#endif
  if (cs_no & BIT(31)) {
    cs_no &= ~BIT(31);
    read_len = 2;
  }

  // config reg read cmd
  qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, reg_read_cmd, cs_no);

  // trigger qspi to read one byte from flash
  READ_4M_FLASH(read_len, cs_no, 0);

  // read the config reg
  do {
    // wait till the fifo empty is deasserted
    while (qspi_reg->QSPI_STATUS_REG & QSPI_FIFO_EMPTY_RFIFO_S)
      ;
#ifdef CHIP_9117
    //! This is a bug fix as this func is returning 0th byte as 0 always in readl_len=2
    //! This Fix is required for Giga flash only and Not necessary for Macronix
    rd_config <<= 8;
    rd_config |= (uint8_t)qspi_reg->QSPI_MANUAL_RD_WR_DATA_REG;
#endif
#ifdef CHIP_9118
    rd_config = (uint8_t)qspi_reg->QSPI_MANUAL_RD_WR_DATA_REG;
    rd_config <<= 8;
#endif
  } while (--read_len);

  return rd_config;
}

void qspi_flash_reg_write(qspi_reg_t *qspi_reg,
                          uint32_t reg_write_cmd,
                          uint32_t reg_write_value,
                          uint32_t cs_no,
                          uint32_t wr_reg_delay_ms)
{
  // write enable
  qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN, cs_no);
  DEASSERT_CSN;

  // write cmd + reg configuration
  qspi_write_to_flash(qspi_reg, QSPI_16BIT_LEN, ((reg_write_cmd << 8) | reg_write_value), cs_no);

  DEASSERT_CSN;
  if (wr_reg_delay_ms) {
    qspi_usleep(wr_reg_delay_ms);
  }
}

void RSI_QSPI_UpdateOperatingMode_and_ResetType(qspi_reg_t *qspi_reg, uint32_t operating_mode)
{
  uint32_t bbff_storage2;
  if (qspi_reg == (qspi_reg_t *)QSPI_BASE_ADDRESS) {
    bbff_storage2 = M4_BBFF_STORAGE2;
    bbff_storage2 &= ~0xff;
    bbff_storage2 |= operating_mode;
    M4_BBFF_STORAGE2 = bbff_storage2;

  } else {
    bbff_storage2 = TA_BBFF_STORAGE2;
    bbff_storage2 &= ~0xff;
    bbff_storage2 |= operating_mode;
    TA_BBFF_STORAGE2 = bbff_storage2;
  }
}


void RSI_QSPI_ResetFlash(qspi_reg_t *qspi_reg, uint32_t cs_no)
{
  uint32_t operating_mode, reset_type, flash_oper_mode;
#ifdef CHIP_9118
  uint32_t ddr_mode;
#endif
  if (qspi_reg == (qspi_reg_t *)QSPI_BASE_ADDRESS) {
    operating_mode = M4_BBFF_STORAGE2;
  } else {
    operating_mode = TA_BBFF_STORAGE2;
  }
  reset_type      = operating_mode & 0xf;
  flash_oper_mode = (operating_mode >> 5) & 0x3;
#ifdef CHIP_9118
  ddr_mode = operating_mode & BIT(7);
#endif
  if (!reset_type) {
    // Giving reset in all possible modes(Single/Quad/Octa) for resetting flash.
    qspi_switch_qspi2(qspi_reg, SINGLE_MODE, cs_no);
    qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, 0xFF, cs_no);
    DEASSERT_CSN;
    qspi_switch_qspi2(qspi_reg, QUAD_MODE, cs_no);
    qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, 0xFF, cs_no);
    DEASSERT_CSN;
#if 0      

		qspi_switch_qspi2(qspi_reg, OCTA_MODE, cs_no);
		qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, 0xFF, cs_no);
		DEASSERT_CSN;
#endif
  } else {
#ifdef CHIP_9118
    if (ddr_mode) {
      qspi_reg->QSPI_MANUAL_CONFIG_2_REG |= QSPI_DDR_CLK_EN;
      qspi_reg->QSPI_MANUAL_CONFIG_2_REG |= QSPI_MANUAL_DDR_PHASSE;
    }
#endif
    qspi_switch_qspi2(qspi_reg, (flash_oper_mode & 0x3), cs_no);
    if (reset_type == 1) {
      // Switch off flash LDO
    } else if (reset_type == 2) { // Godavari mode of resetting
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, 0xFF, cs_no);
      DEASSERT_CSN;
      qspi_write_to_flash(qspi_reg, QSPI_16BIT_LEN, 0xFFFF, cs_no);
      DEASSERT_CSN;
    } else if (reset_type == 3) { // giga device flash
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, 0xFF, cs_no);
      DEASSERT_CSN;
    } else if (reset_type == 4) { // macronix quad
      if (flash_oper_mode != SINGLE_MODE) {
        qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, 0xF5, cs_no);
        DEASSERT_CSN;
        if (operating_mode & BIT(4)) {
          qspi_usleep(150);
        } else {
          qspi_usleep(50);
        }
      }
    } else if (reset_type == 5) { // Adesto_flash
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN, cs_no);
      DEASSERT_CSN;
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, 0xFF, cs_no);
      DEASSERT_CSN;
    } else if (reset_type == 6) { // mx octa flash in single
      if (flash_oper_mode != SINGLE_MODE) {
        qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, 0x66, cs_no);
        DEASSERT_CSN;
        qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, 0x99, cs_no);
        DEASSERT_CSN;
        if (operating_mode & BIT(4)) {
          qspi_usleep(150);
        } else {
          qspi_usleep(50);
        }
      }
    } else if (reset_type == 7) { // mx octa flash in octa. it can be used for ddr mode also
      // This is intended for Macronix OCTA flash if already flash is in single bit mode not giving reset.
      if (flash_oper_mode != SINGLE_MODE) {
        qspi_write_to_flash(qspi_reg, QSPI_16BIT_LEN, 0x6699, cs_no);
        DEASSERT_CSN;
        qspi_write_to_flash(qspi_reg, QSPI_16BIT_LEN, 0x9966, cs_no);
        DEASSERT_CSN;
        if (operating_mode & BIT(4)) {
          qspi_usleep(150);
        } else {
          qspi_usleep(50);
        }
      }
    } else if (reset_type == 8) {
      egpio_set_pin_mux(EGPIO, 0, 13, 0);
      egpio_set_pin(EGPIO, 0, 13, 0);
      egpio_set_dir(EGPIO, 0, 13, 0);
      if (operating_mode & BIT(4)) {
        qspi_usleep(150);
      } else {
        qspi_usleep(50);
      }
      egpio_set_pin(EGPIO, 0, 13, 1);
    } else if (reset_type == 9) {
      egpio_set_pin_mux(EGPIO, 0, 14, 0);
      egpio_set_pin(EGPIO, 0, 14, 0);
      egpio_set_dir(EGPIO, 0, 14, 0);
      if (operating_mode & BIT(4)) {
        qspi_usleep(150);
      } else {
        qspi_usleep(50);
      }
      egpio_set_pin(EGPIO, 0, 14, 1);
    } else if (reset_type == 10) {
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, 0xFF, cs_no);
      DEASSERT_CSN;
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, 0xF5, cs_no);
      DEASSERT_CSN;
    } else if (reset_type == 11) {

      /*****DWORD0******/
      // No of commands[1:0],
      // cmd1_len[3:2],
      // cmd2_len[5:4],
      // Reserved[7:6],
      // delay[13:8],
      // Reserved[15:14]
      // reserved cmd1[15:0]
      /*****DWORD1******/
      // cmd2[15:0]
      uint16_t flash_reset_info[3];
      uint32_t delay, no_of_commands, cmd_len, command, inx, src;

      initialise_m4_efuse_in_io_mode();

      src = M4_EFUSE_IO_BASE_ADDR;

      rsi_cmemcpy((uint8_t *)flash_reset_info, (uint8_t *)(src), 6);

      no_of_commands = flash_reset_info[0] & 0x3;
      inx            = 0;

      while (inx < no_of_commands) {
        // 0 : 32-bit, 1 : 8-bit, 2 : 16-bit
        cmd_len = (flash_reset_info[0] >> ((inx + 1) * 2)) & 0x3;
        command = flash_reset_info[inx + 1];

        qspi_write_to_flash(qspi_reg, (8 * cmd_len), command, cs_no);
        DEASSERT_CSN;
        inx++;
      }

      // If Bit(13) is set delay is ms with 1ms ganularity else delay is in us with 16us granularity.
      if (flash_reset_info[0] & BIT(13)) {
        delay = (((flash_reset_info[0] >> 8) & 0x1f) * 1000);
      } else {
        delay = (((flash_reset_info[0] >> 8) & 0x1f) * 16);
      }

      if (delay) {
        qspi_usleep(delay);
      }
    }
#ifdef CHIP_9118
    if (ddr_mode) {
      qspi_reg->QSPI_MANUAL_CONFIG_2_REG &= ~QSPI_DDR_CLK_EN;
      qspi_reg->QSPI_MANUAL_CONFIG_2_REG &= ~QSPI_MANUAL_DDR_PHASSE;
    }
#endif
  }
  qspi_switch_qspi2(qspi_reg, SINGLE_MODE, cs_no);
}

void qspi_set_flash_mode(qspi_reg_t *qspi_reg,
                         uint32_t data_mode,
                         uint32_t cs_no,
                         uint32_t ddr_mode_en,
                         uint32_t flash_type)
{
  uint32_t enable_bus_mode         = 0;
  volatile uint32_t reset_bus_mode = 0;
  // FIXME AS SST26VF016 supports only SINGLE and QUAD MODE
  // support for Dual mode is not added
  // Add it if flash supports

  if (flash_type == MX_QUAD_FLASH) {
    enable_bus_mode = 0x35;
    reset_bus_mode  = 0xF5;
  } else if (flash_type == MX_OCTA_FLASH) {
    reset_bus_mode = 0x9966;
  } else {
    enable_bus_mode = EQIO;
    reset_bus_mode  = RSTQIO;
  }
  if ((data_mode == QUAD_MODE)) {
    qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, enable_bus_mode, cs_no);

    DEASSERT_CSN;
    qspi_switch_qspi2(qspi_reg, data_mode, cs_no);

  } else {

#if 1
    RSI_QSPI_ResetFlash(qspi_reg, cs_no);
  }
#else

    if (flash_type == ADESTO_OCTA_FLASH) {
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN, cs_no);
      DEASSERT_CSN;
    }
    if (flash_type == MX_OCTA_FLASH) {
      qspi_switch_qspi2(qspi_reg, OCTA_MODE, cs_no);
      qspi_write_to_flash(qspi_reg, QSPI_16BIT_LEN, ((reset_bus_mode << 8) | (reset_bus_mode >> 8)), cs_no);
      DEASSERT_CSN;
      qspi_write_to_flash(qspi_reg, QSPI_16BIT_LEN, reset_bus_mode, cs_no);
      DEASSERT_CSN;
      qspi_switch_qspi2(qspi_reg, SINGLE_MODE, cs_no);
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, reset_bus_mode, cs_no);
      DEASSERT_CSN;
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, (reset_bus_mode >> 8), cs_no);
      DEASSERT_CSN;
    } else {
      qspi_switch_qspi2(qspi_reg, SINGLE_MODE, cs_no);
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, reset_bus_mode, cs_no);
      DEASSERT_CSN;
      qspi_switch_qspi2(qspi_reg, QUAD_MODE, cs_no);
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, reset_bus_mode, cs_no);
      DEASSERT_CSN;
      qspi_switch_qspi2(qspi_reg, OCTA_MODE, cs_no);
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, reset_bus_mode, cs_no);
      DEASSERT_CSN;
    }
    qspi_switch_qspi2(qspi_reg, SINGLE_MODE, cs_no);
  }
#endif
}

void qspi_status_control_reg_write(spi_config_t *spi_config,
                                   qspi_reg_t *qspi_reg,
                                   uint16_t write_command,
                                   uint32_t addr,
                                   uint16_t write_value,
                                   uint32_t cs_no,
                                   uint32_t wr_reg_delay_ms)
{
  // write enable
  qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN, cs_no);
  DEASSERT_CSN;
  // write cmd + reg configuration
  qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, write_command, cs_no);

  if (spi_config->spi_config_1.flash_type == MX_OCTA_FLASH) {
    qspi_write_to_flash(qspi_reg, QSPI_32BIT_ADDR, addr, cs_no);
  } else {
    qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, addr, cs_no);
  }
  qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, write_value, cs_no);
  DEASSERT_CSN;
  if (spi_config->spi_config_1.flash_type == ADESTO_OCTA_FLASH) {
    qspi_wait_flash_status_Idle(qspi_reg, spi_config, 0);
  }

  if (wr_reg_delay_ms) {
    qspi_usleep(wr_reg_delay_ms);
  }
}

void RSI_QSPI_AutoModeEn(qspi_reg_t *qspi_reg)
{
  if (!(qspi_reg->QSPI_STATUS_REG & HW_CTRLD_QSPI_MODE_CTRL_SCLK)) {
    while (!(qspi_reg->QSPI_MANUAL_CONFIG_REG & CSN_ACTIVE)) {
      //wait till manual becomes idle
    }
    qspi_reg->QSPI_BUS_MODE_REG |= AUTO_MODE;
  }
}

/* This function initializes auto mode */
void qspi_auto_init(qspi_reg_t *qspi_reg, spi_config_t *spi_config)
{
  uint32_t offset;
  uint32_t *auto_1_ptr;
  uint32_t *auto_2_ptr;
  uint32_t *auto_3_ptr;
  uint32_t auto_3_data;
  uint32_t dummy_count = 0;
  // enable or disable prefetch
  if (spi_config->spi_config_1.prefetch_en) {
    qspi_reg->QSPI_BUS_MODE_REG |= QSPI_PREFETCH_EN;
  } else {
    qspi_reg->QSPI_BUS_MODE_REG &= ~QSPI_PREFETCH_EN;
  }

  if (spi_config->spi_config_2.addr_width == _32BIT_ADDR) {
    qspi_reg->QSPI_AUTO_CONFIG3 |= QSPI_ADR_SIZE_32BIT_AUTO_MODE;
  } else {
    qspi_reg->QSPI_AUTO_CONFIG3 &= ~QSPI_ADR_SIZE_32BIT_AUTO_MODE;
  }

  if (spi_config->spi_config_2.cs_no) {
    auto_3_ptr = ((uint32_t *)&qspi_reg->QSPI_AUTO_CONFIG3_CSN1);
  } else {
    auto_3_ptr = ((uint32_t *)&qspi_reg->QSPI_AUTO_CONFIG3);
  }
  auto_3_data = *auto_3_ptr;

  if (spi_config->spi_config_3._16bit_cmd_valid) {
    auto_3_data |= QSPI_CMD_SIZE_16BIT_CSN0 | (spi_config->spi_config_1.read_cmd << QSPI_RD_INST_CSN0_MSB);
  } else {
    auto_3_data &= ~QSPI_CMD_SIZE_16BIT_CSN0;
  }
#ifdef CHIP_9118
  if (spi_config->spi_config_4.ddr_data_mode) {
    auto_3_data |= (1 << DDR_DATA);
  } else {
    auto_3_data &= ~(1 << DDR_DATA);
  }
  if (spi_config->spi_config_4.ddr_addr_mode) {
    auto_3_data |= (1 << DDR_ADDR);
  } else {
    auto_3_data &= ~(1 << DDR_ADDR);
  }
  if (spi_config->spi_config_4.ddr_inst_mode) {
    auto_3_data |= (1 << DDR_CMD);
  } else {
    auto_3_data &= ~(1 << DDR_CMD);
  }
  if (spi_config->spi_config_4.ddr_dummy_mode) {
    auto_3_data |= (1 << DDR_DUMMY);
  } else {
    auto_3_data &= ~(1 << DDR_DUMMY);
  }
  if (spi_config->spi_config_4.ddr_extra_byte) {
    auto_3_data |= (1 << DDR_EXTRA_BYTE);
  } else {
    auto_3_data &= ~(1 << DDR_EXTRA_BYTE);
  }
#endif
  dummy_count = (spi_config->spi_config_1.no_of_dummy_bytes) | (spi_config->spi_config_4.no_of_ms_dummy_bytes << 4);

  if (spi_config->spi_config_2.dummy_cycles_for_controller) {

    // Program dummy cycles in bit mode
    auto_3_data |= DUMMY_BYTE_OR_BIT_MODE;

    dummy_count =
      ((dummy_count * 8)
       + (spi_config->spi_config_2.dummy_cycles_for_controller * (1 << spi_config->spi_config_1.dummy_mode)));
  } else {
    auto_3_data &= ~DUMMY_BYTE_OR_BIT_MODE;
  }

  auto_3_data |= (((dummy_count >> 4) & 0xF) << 1);

  if (spi_config->spi_config_3.en_word_swap) {
    auto_3_data |= (1 << WORD_SWAP_EN);
  } else {
    auto_3_data &= ~(1 << WORD_SWAP_EN);
  }
  *auto_3_ptr = auto_3_data;

  // address difference between auto config reg for csn 0 and 1 is 12regs
  offset     = 12 * spi_config->spi_config_2.cs_no;
  auto_1_ptr = ((uint32_t *)&qspi_reg->QSPI_AUTO_CTRL_CONFIG_1_REG + offset);
  auto_2_ptr = ((uint32_t *)&qspi_reg->QSPI_AUTO_CTRL_CONFIG_2_REG + offset);

  if (spi_config->spi_config_3._16bit_cmd_valid) {
    *auto_2_ptr = (spi_config->spi_config_3._16bit_rd_cmd_msb << 8);
  } else {
    *auto_2_ptr = (spi_config->spi_config_1.read_cmd << 8);
  }

  // enable read data swapping for auto mode
  if (spi_config->spi_config_2.swap_en == SWAP) {
    *auto_2_ptr |= AUTO_RD_SWAP;
  } else {
    *auto_2_ptr &= ~AUTO_RD_SWAP;
  }

  *auto_2_ptr |= (AUTO_ADDR_WIDTH) //< address width 8, 9, 16 or 24 bit
                 //	|  (spi_config->spi_config_2.jump_inst << 24)          //< cmd to be used in case of jump
                 | (spi_config->spi_config_1.read_cmd << 16)     //< read cmd is used for wrap reads too
                 | (spi_config->spi_config_3.dummys_4_jump << 4) //< no. of dummy bytes in case of jump reads
                 | (spi_config->spi_config_1.dummy_W_or_R << 3); //< dummy writes or reads
#ifdef CHIP_9117
  if (spi_config->spi_config_1.flash_type != MX_QUAD_FLASH) {
    *auto_2_ptr |= (spi_config->spi_config_1.continuous << 2); //< continuous read mode enable
  }
#endif

  *auto_1_ptr = ((dummy_count & 0xF) << 24)                               //< no. of dummy bytes for read
                | (spi_config->spi_config_3.no_of_dummy_bytes_wrap << 28) //< no. of dummy bytes for wrap
                //		| (spi_config->spi_config_2.jump_en << 23)  					//< jump enable
                | (0 << 23)                                        //< removed jump enable
                | (spi_config->spi_config_1.extra_byte_en << 18)   //< extra byte enable
                | (EXTRA_BYTE << 10)                               //< extra byte value
                | (spi_config->spi_config_1.inst_mode << 6)        //< cmd mode
                | (spi_config->spi_config_1.addr_mode << 4)        //< addr mode
                | (spi_config->spi_config_1.dummy_mode << 2)       //< dummy mode
                | (spi_config->spi_config_1.extra_byte_mode << 0); //< extra mode

  if (spi_config->spi_config_4.dual_flash_mode) {
    *auto_1_ptr |= (OCTA_MODE << 8); //< data mode
  } else {
    *auto_1_ptr |= (spi_config->spi_config_1.data_mode << 8); //< data mode
  }
  if (spi_config->spi_config_2.wrap_len_in_bytes != NO_WRAP) {
    // Enable burst mode read in qspi
    qspi_reg->QSPI_BUS_MODE_REG |= QSPI_WRAP_EN;
    // no. of dummy bytes for wrap
    *auto_1_ptr |= 1 << 28;
    // read cmd for wrap mode
    *auto_2_ptr |= READ_BURST << 16;
  } else {
    // disable burst mode read in qspi
    qspi_reg->QSPI_BUS_MODE_REG &= ~QSPI_WRAP_EN;
  }
  RSI_QSPI_AutoModeEn(qspi_reg);
}

void qspi_flash_init(qspi_reg_t *qspi_reg, spi_config_t *spi_config, uint32_t wr_reg_delay_ms)
{
  uint32_t reg_cfg  = 0;
  uint32_t xip_mode = DIS_XIP;
  uint32_t is_quad_mode;
  uint32_t is_octa_mode;
  uint32_t flash_type;
  uint32_t opi_qpi, wr_status_reg = 0, flash_status2 = 0;
  uint32_t dummy_7_0, dummy_7_4;
  uint32_t str_dtr, dummy_clks, status, operating_mode;
	uint32_t total_dummy_bytes=0;

  // store quad indication to optimize
  is_quad_mode = CHK_QUAD_MODE;
  is_octa_mode = CHK_OCTA_MODE;
  opi_qpi      = (spi_config->spi_config_1.inst_mode == OCTA_MODE)
              ? BIT(3)
              : ((spi_config->spi_config_1.inst_mode == QUAD_MODE) ? BIT(2) : 0);
  flash_type = spi_config->spi_config_1.flash_type;

  // Reset flash to single bit mode
  qspi_set_flash_mode(qspi_reg, 0, spi_config->spi_config_2.cs_no, 0, flash_type);

// Store flash operating mode and reset type in battery backup flipflops.
#ifdef CHIP_9118
  operating_mode = (spi_config->spi_config_1.inst_mode | (spi_config->spi_config_3.ddr_mode_en ? BIT(2) : 0));
#else
  operating_mode = (spi_config->spi_config_1.inst_mode);
#endif
  RSI_QSPI_UpdateOperatingMode_and_ResetType(qspi_reg, ((operating_mode << 5) | (spi_config->spi_config_5.reset_type)));

  switch (flash_type) {
    case SST_QUAD_FLASH:
    case SST_SPI_FLASH:
      // Configure flash to quad/single mode as requested
      //qspi_set_flash_mode(qspi_reg, spi_config->spi_config_1.data_mode,
      //		spi_config->spi_config_2.cs_no, 0, flash_type);
#if 0      
		if(spi_config->spi_config_2.protection) {
			// Removes write protection, so that memory can be written
			// OR Enables write protection, so that unwanted writes are ignored
			qspi_write_block_protect(qspi_reg, SST_PROTECTION,
					spi_config->spi_config_2.cs_no,
					spi_config->spi_config_2.num_prot_bytes, wr_reg_delay_ms);
		}
#endif
      break;

    case AT_QUAD_FLASH:
      // QUAD_MODE en is at BIT(7), so enable it if asked
      if (is_quad_mode) {
        qspi_flash_reg_write(qspi_reg, WCON, ATMEL_QEN, spi_config->spi_config_2.cs_no, wr_reg_delay_ms);
      }
#if 0      
		if(spi_config->spi_config_2.protection) {
			// Removes write protection, so that memory can be written
			// OR Enables write protection, so that unwanted writes are ignored
			qspi_status_reg_write(qspi_reg, AT_PROT,
					spi_config, wr_reg_delay_ms);
		}
#endif
      break;

    case MX_QUAD_FLASH:

      if ((is_quad_mode) || (spi_config->spi_config_4.prot_top_bottom)
          || (spi_config->spi_config_4.high_perf_mode_en)) {
        status = qspi_wait_flash_status_Idle(qspi_reg, spi_config, wr_reg_delay_ms);

        if ((!!(status & BIT(6))) != is_quad_mode) {
          wr_status_reg = 1;
        }

        status |= (is_quad_mode << 6);
        if ((spi_config->spi_config_4.prot_top_bottom) || (spi_config->spi_config_4.high_perf_mode_en)) {
          flash_status2 = qspi_flash_reg_read(qspi_reg, 0x15, spi_config->spi_config_2.cs_no | BIT(31), spi_config);
          status        = (status << 16) | flash_status2;
          if (spi_config->spi_config_4.high_perf_mode_en) {
            status |= HIGH_PERF_MODE;
            wr_reg_delay_ms |= BIT(29);
            wr_status_reg = 1;
          } else {
            status >>= 8;
          }

          if (spi_config->spi_config_4.prot_top_bottom) {

            status |= (PROT_FROM_TOP << (spi_config->spi_config_4.high_perf_mode_en) ? 8 : 0);

            wr_reg_delay_ms |= BIT(31);
            wr_status_reg = 1;
          }

          else if (!(spi_config->spi_config_4.high_perf_mode_en)) {
            status >>= 8;
          }
        }

        if (wr_status_reg) {
          qspi_status_reg_write(qspi_reg, status, spi_config, wr_reg_delay_ms);
        }
      }
      wr_reg_delay_ms &= ~BIT(31);
      if (spi_config->spi_config_1.inst_mode == QUAD_MODE) {
        // Send QPI/OPI enable command
        qspi_set_flash_mode(qspi_reg,
                            spi_config->spi_config_1.inst_mode,
                            spi_config->spi_config_2.cs_no,
                            0,
                            flash_type);
      }
      break;

    case MX_OCTA_FLASH:
#ifdef CHIP_9118
      str_dtr = is_octa_mode ? (spi_config->spi_config_3.ddr_mode_en ? BIT(1) : BIT(0)) : 0;
#else
      str_dtr = is_octa_mode ? BIT(0) : 0;
#endif
      //qspi_set_flash_mode(qspi_reg, 0, spi_config->spi_config_2.cs_no, 0, flash_type);
      dummy_7_4 = spi_config->spi_config_4.no_of_ms_dummy_bytes;
      dummy_7_0 = (spi_config->spi_config_1.no_of_dummy_bytes | (dummy_7_4 << 4));
      if (dummy_7_0 <= 8) {
        dummy_clks = 6;
      } else if (dummy_7_0 <= 16) {
        dummy_clks = 2;
      } else {
        dummy_clks = 0;
      }

      if (spi_config->spi_config_4.prot_top_bottom) {
        status        = qspi_wait_flash_status_Idle(qspi_reg, spi_config, wr_reg_delay_ms);
        flash_status2 = qspi_flash_reg_read(qspi_reg, 0x15, spi_config->spi_config_2.cs_no, spi_config);
        if (flash_status2 & BIT(3)) {
        } else {
          status = ((status << 8) | 0xF);
          wr_reg_delay_ms |= BIT(31);
          qspi_status_reg_write(qspi_reg, status, spi_config, wr_reg_delay_ms);
        }
      }
      wr_reg_delay_ms &= ~BIT(31);

      qspi_status_control_reg_write(spi_config,
                                    qspi_reg,
                                    WCFG2,
                                    0x300,
                                    dummy_clks,
                                    spi_config->spi_config_2.cs_no,
                                    wr_reg_delay_ms);
      qspi_status_control_reg_write(spi_config,
                                    qspi_reg,
                                    WCFG2,
                                    0,
                                    str_dtr,
                                    spi_config->spi_config_2.cs_no,
                                    wr_reg_delay_ms);
      break;

    case MICRON_QUAD_FLASH:
      //qspi_set_flash_mode(qspi_reg, 0, spi_config->spi_config_2.cs_no, 0, flash_type);

      if (!spi_config->spi_config_3.xip_mode) {
        xip_mode = DIS_XIP;
      } else {
        xip_mode = XIP_MODE;
      }
      dummy_clks = spi_config->spi_config_1.no_of_dummy_bytes;
      if (spi_config->spi_config_1.dummy_mode == DUAL_MODE) {
        dummy_clks = dummy_clks * 4;
      } else if (spi_config->spi_config_1.dummy_mode == QUAD_MODE) {
        dummy_clks = dummy_clks * 2;
      } else {
        dummy_clks = spi_config->spi_config_1.no_of_dummy_bytes * 8;
      }
      // configure the no_of_dummy_bytes
      qspi_flash_reg_write(qspi_reg,
                           WR_VOL_CON_REG,

                           ((dummy_clks << 4) | xip_mode | spi_config->spi_config_2.wrap_len_in_bytes),
                           spi_config->spi_config_2.cs_no,
                           wr_reg_delay_ms);

      if (is_quad_mode) {
        // default configs + Quad mode enable
        reg_cfg = 0x5F;
      } else if (CHK_DUAL_MODE) {
        // default configs + Dual mode enable
        reg_cfg = 0x9F;
      } else {
        // default configs + SPI mode enable
        reg_cfg = 0xDF;
      }
      if (reg_cfg) {
        /** writing enhanced volatile config reg only if
			 *  Quad/dual mode are enabled
			 */
        qspi_flash_reg_write(qspi_reg, WR_ENHN_VOL_CON_REG, reg_cfg, spi_config->spi_config_2.cs_no, wr_reg_delay_ms);
      }
      break;

    case GIGA_DEVICE_FLASH:
    case WBOND_QUAD_FLASH:
      //qspi_set_flash_mode(qspi_reg, 0,
      //		spi_config->spi_config_2.cs_no, 0, flash_type);
      // Set QUAD ENABLE bit in status register(BIT(9))
      status = qspi_wait_flash_status_Idle(qspi_reg, spi_config, wr_reg_delay_ms);
#ifndef CHIP_9117
      status <<= 8;
      status |= (is_quad_mode << 1);

      qspi_status_reg_write(qspi_reg, status, spi_config, wr_reg_delay_ms);
#else
      flash_status2 = qspi_flash_reg_read(qspi_reg, SR2_READ, spi_config->spi_config_2.cs_no, spi_config);
      //! Checking whether flash configuration and user configurations are same, if not same we make it equal below
      if (is_quad_mode != ((flash_status2 & QUAD_EN) >> 1)) {
        //! Preparing 2byte status reg data; upper byte will be fed first
        status <<= 8;
        //! If quad mode is set in spi config we enable quad in flash otherwise we disable
        if (is_quad_mode) {
          status |= (flash_status2 | QUAD_EN);
        } else {
          status |= (flash_status2 & ~QUAD_EN);
        }
        wr_status_reg = 1;
        //! This 31st bit is to enable qspi_status_reg_write func to initiate 2byte status reg write
        wr_reg_delay_ms |= BIT(31);
        //! This programming enables the status_re_write func to do non-volatile reg write
        wr_reg_delay_ms |= BIT(30);
      }
      if (wr_status_reg)
        qspi_status_reg_write(qspi_reg, status, spi_config, wr_reg_delay_ms);
      wr_reg_delay_ms &= ~(BIT(30) | BIT(31));
#endif
      if (spi_config->spi_config_1.inst_mode == QUAD_MODE) {
        // Send QPI enable command
        qspi_set_flash_mode(qspi_reg,
                            spi_config->spi_config_1.inst_mode,
                            spi_config->spi_config_2.cs_no,
                            0,
                            flash_type);
        // Set read parameters. Setting number of dummy bytes and Wrap bytes.
#ifdef CHIP_9117
        //! This is to fix the dummy cycles configuration which was not happening properly
        total_dummy_bytes =
          spi_config->spi_config_1.no_of_dummy_bytes + spi_config->spi_config_1.extra_byte_en;
        if (total_dummy_bytes == 4) {
          //! 8 dummy cycles
          dummy_clks = (3 << 4);
        } else if (total_dummy_bytes == 3) {
          //! 6 dummy cycles
          dummy_clks = (2 << 4);
        } else {
          //! 4 dummy cycles
          dummy_clks = 0;
        }
#else
        if (spi_config->spi_config_1.no_of_dummy_bytes > 2) {
          if (spi_config->spi_config_1.no_of_dummy_bytes == 3) {
            // 6 clocks dummy cycles
            dummy_clks = (0x2 << 4);
          } else {
            // 8 clocks dummy cycles
            dummy_clks = (0x3 << 4);
          }
        } else {
          // 4 clocks dummy cycles
          dummy_clks = 0;
        }
#endif
        qspi_flash_reg_write(qspi_reg, 0xC0, (dummy_clks | 0x00), 0, wr_reg_delay_ms);
      }
      break;

#ifdef CHIP_9117
    case ADESTO_QUAD_FLASH:
      //Writing control reg for quad mode
      if (is_quad_mode) {
        qspi_status_control_reg_write(spi_config,
                                      qspi_reg,
                                      STS_CTRL,
                                      2,
                                      (is_quad_mode << 1),
                                      spi_config->spi_config_2.cs_no,
                                      wr_reg_delay_ms);
      }
      //Writing control reg for xip mode and wrap lenth
      if (spi_config->spi_config_3.xip_mode) {

        qspi_status_control_reg_write(spi_config,
                                      qspi_reg,
                                      STS_CTRL,
                                      4,
                                      (spi_config->spi_config_2.wrap_len_in_bytes | BIT(3)),
                                      spi_config->spi_config_2.cs_no,
                                      wr_reg_delay_ms);
      }
      //Writing control reg for dummy cycles incase of 0-4-4 command
      if (spi_config->spi_config_2.dummy_cycles_for_controller) {
        qspi_status_control_reg_write(spi_config,
                                      qspi_reg,
                                      STS_CTRL,
                                      5,
                                      (spi_config->spi_config_2.dummy_cycles_for_controller << 4),
                                      spi_config->spi_config_2.cs_no,
                                      wr_reg_delay_ms);
      }
      break;
#endif
    case ADESTO_OCTA_FLASH:
      str_dtr = 0;
      if (opi_qpi) {
        if (spi_config->spi_config_1.dummy_mode == OCTA_MODE) {
          dummy_clks = spi_config->spi_config_1.no_of_dummy_bytes;
        } else {
          dummy_clks = spi_config->spi_config_1.no_of_dummy_bytes * 2;
        }
      } else {
        dummy_clks = 0;
      }
      if (dummy_clks <= 8) {
        dummy_clks = 0;
      } else if (dummy_clks <= 16) {
        dummy_clks = 4;
      }
      //qspi_set_flash_mode(qspi_reg, 0,
      //		spi_config->spi_config_2.cs_no, 0, flash_type);
      // config dummy and wrap bytes
      qspi_status_control_reg_write(spi_config,
                                    qspi_reg,
                                    STS_CTRL,
                                    3,
                                    (dummy_clks | (spi_config->spi_config_2.wrap_len_in_bytes << 5)),
                                    spi_config->spi_config_2.cs_no,
                                    wr_reg_delay_ms);
      // enabling opi or qpi and str or dtr
      if (opi_qpi || str_dtr) {
        qspi_flash_reg_write(qspi_reg, STS_BYT2, (opi_qpi | str_dtr), spi_config->spi_config_2.cs_no, wr_reg_delay_ms);
      }
      break;

    default:;
      // XXX UNKNOWN FLASH TYPE
      // If the flash type is other than the current list
      // then caller must take care of flash configuration
  }
}

void qspi_spi_init(qspi_reg_t *qspi_reg,
                   spi_config_t *spi_config,
                   uint32_t flash_init_req,
                   uint32_t wr_reg_delay_ms,
                   uint8_t fifo_thrsld)
{
  uint32_t dataline_pos, d7_d4_data_pos;
  uint32_t cs_no         = spi_config->spi_config_2.cs_no;
  uint32_t secondary_csn = spi_config->spi_config_4.secondary_csn;

  // d2d3 dataline initialization
  dataline_pos   = GET_POS;
  d7_d4_data_pos = GET_POS_D7_D4;
  qspi_reg->QSPI_BUS_MODE_REG =
    ((qspi_reg->QSPI_BUS_MODE_REG & (MASK_D3_D2(dataline_pos))) | (spi_config->spi_config_1.d3d2_data << dataline_pos)
     | (spi_config->spi_config_2.neg_edge_sampling << 16));

  qspi_reg->OCTA_SPI_BUS_CONTROLLER = ((qspi_reg->OCTA_SPI_BUS_CONTROLLER & (MASK_D7_D4(d7_d4_data_pos)))
                                       | (spi_config->spi_config_5.d7_d4_data << d7_d4_data_pos));

  //for micron, qspi should not be switched to spi mode, as flash might be in quad mode already
  qspi_switch_qspi2(qspi_reg, SINGLE_MODE, spi_config->spi_config_2.cs_no);

  if (spi_config->spi_config_4.dual_flash_mode) {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 |= (DUAL_FLASH_MODE | (secondary_csn << (4 + (cs_no * 2))));
  } else {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 &= ~(DUAL_FLASH_MODE | (secondary_csn << (4 + (cs_no * 2))));
  }

  if (spi_config->spi_config_4.dual_flash_mode) {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 |= BIT(19);
  }

  if (flash_init_req) {
    // flash initialization
    qspi_flash_init(qspi_reg, spi_config, wr_reg_delay_ms);
    if (spi_config->spi_config_2.wrap_len_in_bytes != NO_WRAP) {
      // send SET_BURST cmd to flash to initialize wrap
      qspi_write_to_flash(qspi_reg,
                          16,
                          ((SET_BURST << 8) | (spi_config->spi_config_2.wrap_len_in_bytes)),
                          spi_config->spi_config_2.cs_no);
      DEASSERT_CSN;
    }
  }
  if (spi_config->spi_config_4.dual_flash_mode) {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 &= ~BIT(19);
  }

  qspi_switch_qspi2(qspi_reg, spi_config->spi_config_1.inst_mode, spi_config->spi_config_2.cs_no);

// only enabling synchros logic & dynamic clk
#ifdef CHIP_9118
  if (spi_config->spi_config_4.ddr_dll_en) {
    RSI_QSPI_ConfigQspiDll(spi_config, qspi_reg);
  } else {
    qspi_reg->QSPI_CLK_CONFIG_REG = (spi_config->spi_config_2.qspi_clk_en << 8);
  }
#else
  qspi_reg->QSPI_CLK_CONFIG_REG = (spi_config->spi_config_2.qspi_clk_en << 8);
#endif

  if (spi_config->spi_config_4.polarity_mode) {
    qspi_reg->QSPI_CLK_CONFIG_REG |= BIT(20);
  } else {
    qspi_reg->QSPI_CLK_CONFIG_REG &= ~BIT(20);
  }

  if (spi_config->spi_config_4.auto_csn_based_addr_en) {
    qspi_reg->QSPI_BUS_MODE_REG |= AUTO_CSN_BASED_ADDR_ENABLE;
    qspi_reg->QSPI_AUTO_BASE_ADDR_UNMASK_CSN0 = 0xFC000000;
  } else {
    qspi_reg->QSPI_BUS_MODE_REG &= ~AUTO_CSN_BASED_ADDR_ENABLE;
  }

// Keeping csn high for one SOC clock cycle, during CSN assertions. Range is 0 to 31. /
// This is fix for QSPI read issue at > 75 degrees, if prefetch is enabled.
#ifdef CHIP_9118
  if (!(spi_config->spi_config_4.ddr_dll_en)) {
    qspi_reg->QSPI_CLK_CONFIG_REG |= 1;
  }
#else
  qspi_reg->QSPI_CLK_CONFIG_REG |= 1;
#endif
  // Enable HW_CTRL_MODE
  qspi_reg->QSPI_MANUAL_CONFIG_REG |= (spi_config->spi_config_2.cs_no << 13) | HW_CTRL_MODE;

  if (spi_config->spi_config_4.qspi_loop_back_mode_en) {
    qspi_reg->QSPI_MANUAL_CONFIG_2_REG |= QSPI_LOOP_BACK_MODE_EN;
  } else {
    qspi_reg->QSPI_MANUAL_CONFIG_2_REG &= ~QSPI_LOOP_BACK_MODE_EN;
  }
#ifdef CHIP_9118
  if (spi_config->spi_config_4.qspi_manual_ddr_phasse) {
    qspi_reg->QSPI_MANUAL_CONFIG_2_REG |= QSPI_MANUAL_DDR_PHASSE;
  } else {
    qspi_reg->QSPI_MANUAL_CONFIG_2_REG &= ~QSPI_MANUAL_DDR_PHASSE;
  }

  if (spi_config->spi_config_3.ddr_mode_en) {
    qspi_reg->QSPI_MANUAL_CONFIG_2_REG |= QSPI_DDR_CLK_EN;
  } else {
    qspi_reg->QSPI_MANUAL_CONFIG_2_REG &= ~QSPI_DDR_CLK_EN;
  }
#endif
  if (fifo_thrsld) {
    // configure QSPI fifo thresholds
    qspi_reg->QSPI_FIFO_THRESHOLD_REG = (QSPI_FIFO_AFULL_TH << 4) | QSPI_FIFO_AEMPTY_TH;
  }

  // Enable full duplex mode if asked
  if (spi_config->spi_config_2.full_duplex == EN_FULL_DUPLEX) {
    qspi_reg->QSPI_MANUAL_CONFIG_REG |= FULL_DUPLEX_EN;
  } else if (spi_config->spi_config_2.full_duplex == DIS_FULL_DUPLEX) {
    qspi_reg->QSPI_MANUAL_CONFIG_REG &= ~FULL_DUPLEX_EN;
  }

  if (spi_config->spi_config_4.continue_fetch_en) {
    qspi_reg->QSPI_AUTO_CONITNUE_FETCH_CTRL_REG |= CONTINUE_FETCH_EN;
  } else {
    qspi_reg->QSPI_AUTO_CONITNUE_FETCH_CTRL_REG &= ~CONTINUE_FETCH_EN;
  }

  qspi_auto_init(qspi_reg, spi_config);
  // If auto mode is requested call auto_init
  if (spi_config->spi_config_2.auto_mode) {
    RSI_QSPI_AutoModeEn(qspi_reg);
  }
}


void qspi_spi_erase(qspi_reg_t *qspi_reg,
                    spi_config_t *spi_config,
                    uint32_t erase_cmd,
                    uint32_t blk_sec_addr,
                    uint32_t dis_hw_ctrl,
                    uint32_t wr_reg_delay_ms)
{
  uint32_t cs_no;
  uint32_t prev_state = 0, flash_type, cmd_len;
  uint32_t cmd_to_drive;
  cs_no      = spi_config->spi_config_2.cs_no;
  flash_type = spi_config->spi_config_1.flash_type;

  if (spi_config->spi_config_2.addr_width == 4) {
    // Shifting more than 31 bit won't work in our processor. So not using below logic.
    if (dis_hw_ctrl & BIT(31)) {
    } else {
      blk_sec_addr &= 0x3FFFFFF;
    }
  } else {
    blk_sec_addr &= ((1 << (spi_config->spi_config_2.addr_width * 8)) - 1);
  }
  dis_hw_ctrl &= ~BIT(31);

  /* Check if already auto mode enabled */
  if (qspi_reg->QSPI_BUS_MODE_REG & AUTO_MODE) {
    qspi_reg->QSPI_BUS_MODE_REG &= ~AUTO_MODE;
    while (qspi_reg->QSPI_STATUS_REG & AUTO_MODE_ENABLED)
      ;
    prev_state = 1;
  }
  if (spi_config->spi_config_4.continue_fetch_en) {
    qspi_reg->QSPI_AUTO_CONITNUE_FETCH_CTRL_REG &= ~CONTINUE_FETCH_EN;
  }
  if (spi_config->spi_config_4.dual_flash_mode) {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 |= BIT(19);
  }

  // switch qspi to data mode
  qspi_switch_qspi2(qspi_reg, spi_config->spi_config_3.wr_inst_mode, cs_no);

  // if hardware control needs to be disabled, do it here.
  if (dis_hw_ctrl) {
    qspi_reg->QSPI_MANUAL_CONFIG_REG &= ~HW_CTRL_MODE;
    while (qspi_reg->QSPI_STATUS_REG & HW_CTRLD_QSPI_MODE_CTRL_SCLK)
      ;
  }

// write enable
#ifdef CHIP_9118
  if (spi_config->spi_config_3.ddr_mode_en) {
    if (flash_type == MX_OCTA_FLASH) {
      qspi_write_to_flash(qspi_reg, QSPI_16BIT_LEN, (WREN << 8) | WREN2, cs_no);
    } else {
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN, cs_no);
    }
  } else {
    qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN, cs_no);
    if (spi_config->spi_config_3._16bit_cmd_valid) {
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN2, cs_no);
    }
  }
#else
  qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN, cs_no);
  if (spi_config->spi_config_3._16bit_cmd_valid) {
    qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN2, cs_no);
  }
#endif
  DEASSERT_CSN;

  // erase command is sent to flash
  // erase command is sent to flash
  cmd_len = spi_config->spi_config_3._16bit_cmd_valid ? 16 : 8;

  if (erase_cmd == SECTOR_ERASE) {
    cmd_to_drive = spi_config->spi_config_6.sector_erase_cmd;
  } else if (erase_cmd == BLOCK_ERASE) {
    cmd_to_drive = spi_config->spi_config_5.block_erase_cmd;
  } else { // for sector erase and block erase
    cmd_to_drive = spi_config->spi_config_6.chip_erase_cmd;
  }

  qspi_write_to_flash(qspi_reg, cmd_len, cmd_to_drive, cs_no);

  // if not chip erase then send the address of sector/block
  if (erase_cmd != CHIP_ERASE) {
    qspi_write_to_flash(qspi_reg, ADDR_LEN, blk_sec_addr, cs_no);
  }
  DEASSERT_CSN;

  // wait for flash to become idle
  qspi_wait_flash_status_Idle(qspi_reg, spi_config, wr_reg_delay_ms);

  if (spi_config->spi_config_4.dual_flash_mode) {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 &= ~BIT(19);
  }

  // if hardware control was disabled, enable it here.
  if (dis_hw_ctrl) {
    qspi_reg->QSPI_MANUAL_CONFIG_REG |= HW_CTRL_MODE;
    while (!(qspi_reg->QSPI_STATUS_REG & HW_CTRLD_QSPI_MODE_CTRL_SCLK))
      ;
  }
  if (prev_state == 1) {
    qspi_reg->QSPI_BUS_MODE_REG |= AUTO_MODE;
    while (!(qspi_reg->QSPI_STATUS_REG & AUTO_MODE_ENABLED))
      ;
  }
  if (spi_config->spi_config_4.continue_fetch_en) {
    qspi_reg->QSPI_AUTO_CONITNUE_FETCH_CTRL_REG |= CONTINUE_FETCH_EN;
  }
}

uint32_t qspi_spi_write(qspi_reg_t *qspi_reg,
                        spi_config_t *spi_config,
                        uint32_t write_cmd,
                        uint32_t addr,
                        uint8_t *data,
                        uint32_t len_in_bytes,
                        uint16_t page_size,
                        uint32_t hsize,
                        uint32_t dis_hw_ctrl,
                        uint32_t wr_reg_delay_ms,
                        uint32_t check_en,
                        uint32_t dma_flags,
                        void *udmaHandle,
                        void *gpdmaHandle)
{

  uint16_t loop_count;
  uint16_t *data_16bit;
  uint32_t *data_32bit;
  uint32_t once;
  uint32_t cs_no;
  uint32_t prev_state     = 0;
  uint32_t qspi_data_mode = 0;
  uint32_t address;
  uint32_t length, status, len_in_loop, flash_type;
  volatile uint32_t auto_mode_address = 0;
  uint8_t *data_in;
  uint32_t ch_no;
  status     = RSI_OK;
  once       = 1;
  data_16bit = (uint16_t *)data;
  data_32bit = (uint32_t *)data;
  cs_no      = spi_config->spi_config_2.cs_no;
  flash_type = spi_config->spi_config_1.flash_type;
  ch_no      = (dma_flags & 0xFF);

  // Ignoring bits more than address width.
  if (spi_config->spi_config_2.addr_width == 4) {
    // Shifting more than 31 bit won't work in our processor. So not using below logic.
    if (dis_hw_ctrl & BIT(31)) {
    } else {
      addr &= 0x3FFFFFF;
    }
  } else {
    addr &= ((1 << (spi_config->spi_config_2.addr_width * 8)) - 1);
  }
  dis_hw_ctrl &= ~BIT(31);

  address = addr;
  length  = len_in_bytes;
  data_in = (uint8_t *)data;
  /* Check if already auto mode enabled */
  if (qspi_reg->QSPI_BUS_MODE_REG & AUTO_MODE) {
    qspi_reg->QSPI_BUS_MODE_REG &= ~AUTO_MODE;
    while (qspi_reg->QSPI_STATUS_REG & AUTO_MODE_ENABLED)
      ;
    prev_state = 1;
  }
  if (spi_config->spi_config_4.continue_fetch_en) {
    qspi_reg->QSPI_AUTO_CONITNUE_FETCH_CTRL_REG &= ~CONTINUE_FETCH_EN;
  }

  if (spi_config->spi_config_4.dual_flash_mode) {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 |= BIT(19);
  }

  while (len_in_bytes) {
    // pick loop_count as minimum of len_in_bytes and page_size
    loop_count = XMIN(len_in_bytes, page_size);
    // switch qspi to inst mode
    qspi_switch_qspi2(qspi_reg, spi_config->spi_config_3.wr_inst_mode, cs_no);
#ifdef CHIP_9118
    if (spi_config->spi_config_3.ddr_mode_en) {
      if (flash_type == MX_OCTA_FLASH) {
        qspi_write_to_flash(qspi_reg, QSPI_16BIT_LEN, (WREN << 8) | WREN2, cs_no);
      } else {
        qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WREN, cs_no);
      }
    } else {
      // write enable
      qspi_write_to_flash(qspi_reg, CMD_LEN, WREN, cs_no);
      if (spi_config->spi_config_3._16bit_cmd_valid) {
        qspi_write_to_flash(qspi_reg, CMD_LEN, WREN2, cs_no);
      }
    }
#else
    // write enable
    qspi_write_to_flash(qspi_reg, CMD_LEN, WREN, cs_no);
    if (spi_config->spi_config_3._16bit_cmd_valid) {
      qspi_write_to_flash(qspi_reg, CMD_LEN, WREN2, cs_no);
    }
#endif
    DEASSERT_CSN;

    // write command is send to flash
#ifdef CHIP_9118
    if (spi_config->spi_config_3.ddr_mode_en) {
      if (flash_type == MX_OCTA_FLASH) {
        qspi_write_to_flash(qspi_reg,
                            QSPI_16BIT_LEN,
                            ((spi_config->spi_config_3.wr_cmd << 8) | spi_config->spi_config_4._16bit_wr_cmd_msb),
                            cs_no);
      } else {
        qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, spi_config->spi_config_3.wr_cmd, cs_no);
      }
    } else {
      qspi_write_to_flash(qspi_reg, CMD_LEN, spi_config->spi_config_3.wr_cmd, cs_no);
      if (spi_config->spi_config_3._16bit_cmd_valid) {
        qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, spi_config->spi_config_4._16bit_wr_cmd_msb, cs_no);
      }
    }
#else
    qspi_write_to_flash(qspi_reg, CMD_LEN, spi_config->spi_config_3.wr_cmd, cs_no);
    if (spi_config->spi_config_3._16bit_cmd_valid) {
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, spi_config->spi_config_4._16bit_wr_cmd_msb, cs_no);
    }
#endif

    // if write cmd is not AAI then send addr or send atleast once
    if (once) {
      qspi_switch_qspi2(qspi_reg, spi_config->spi_config_3.wr_addr_mode, cs_no);
      qspi_write_to_flash(qspi_reg, ADDR_LEN, addr, cs_no);
      if (spi_config->spi_config_3.wr_cmd == AAI) {
        once = 0;
      }
    }

    if (spi_config->spi_config_4.dual_flash_mode) {
      qspi_reg->OCTA_SPI_BUS_CONTROLLER2 &= ~BIT(19);
      qspi_data_mode                        = spi_config->spi_config_3.wr_data_mode;
      spi_config->spi_config_3.wr_data_mode = OCTA_MODE;
    }
    if (ODD_PAGE_BOUNDARY) {
      loop_count = page_size - ODD_PAGE_BOUNDARY;
      loop_count = XMIN(loop_count, len_in_bytes);
    }
    // reducing total length by loop_count
    len_in_bytes -= loop_count;
    // increment addr by loop_count, so that during next iteration addr will be ready
    addr += loop_count;

    while (loop_count) {
      // start writing data
      qspi_switch_qspi2(qspi_reg, spi_config->spi_config_3.wr_data_mode, cs_no);
      // write swap en is done only for data
      qspi_reg->QSPI_MANUAL_CONFIG_2_REG = (qspi_reg->QSPI_MANUAL_CONFIG_2_REG & ~0xF)
                                           | (spi_config->spi_config_2.swap_en << cs_no);

      if (spi_config->spi_config_4.dma_write) {
        qspi_reg->QSPI_MANUAL_WRITE_DATA_2_REG = hsize;
        qspi_reg->QSPI_MANUAL_CONFIG_REG       = 0x200000 | WRITE_TRIGGER | (256 << 3);

//        if (dma_flags & USE_UDMA_MODE) {

//        //  RSI_UDMA_ChannelEnable(udmaHandle, ch_no);
//          /* Enable DMA controller  */
//         // RSI_UDMA_UDMAEnable(udmaHandle);
//        //  while (!(RSI_UDMA_ChannelIsEnabled(udmaHandle, ch_no)))
//            ;

//        } else {
//         // gpdma_dma_channel_trigger(gpdmaHandle, ch_no);
//        //  while ((gpdma_channel_is_enabled(gpdmaHandle, ch_no)))
//            ;
//        }

        loop_count -= 256;
        qspi_wait_flash_status_Idle(qspi_reg, spi_config, cs_no);

      } else {
        if (hsize == _8BIT) {
          qspi_write_to_flash(qspi_reg, 8, *(data), cs_no);
          data++;
          loop_count -= 1;
        } else if (hsize == _16BIT) {
          qspi_write_to_flash(qspi_reg, 16, *(data_16bit), cs_no);
          data_16bit++;
          loop_count -= 2;
        } else if (hsize == _32BIT) {
          qspi_write_to_flash(qspi_reg, 32, *(data_32bit), cs_no);
          data_32bit++;
          loop_count -= 4;
        }
      }
    }

    // if hardware control needs to be disabled, do it here
    if (dis_hw_ctrl) {
      qspi_reg->QSPI_MANUAL_CONFIG_REG &= ~HW_CTRL_MODE;
      while (qspi_reg->QSPI_STATUS_REG & HW_CTRLD_QSPI_MODE_CTRL_SCLK)
        ;
    }

    // dual flash mode
    if (spi_config->spi_config_4.dual_flash_mode) {
      qspi_reg->OCTA_SPI_BUS_CONTROLLER2 |= BIT(19);
      spi_config->spi_config_3.wr_data_mode = qspi_data_mode;
      qspi_switch_qspi2(qspi_reg, spi_config->spi_config_3.wr_data_mode, cs_no);
    }

    qspi_reg->QSPI_MANUAL_CONFIG_2_REG &= ~0xF;
    qspi_switch_qspi2(qspi_reg, spi_config->spi_config_1.inst_mode, cs_no);
    qspi_wait_flash_status_Idle(qspi_reg, spi_config, cs_no);

    // if required (non-zero) wait for the delay time
    if (wr_reg_delay_ms) {
      qspi_usleep(wr_reg_delay_ms);
    }

    // if hardware control was disabled, enable it here
    if (dis_hw_ctrl) {
      qspi_reg->QSPI_MANUAL_CONFIG_REG |= HW_CTRL_MODE;
      while (!(qspi_reg->QSPI_STATUS_REG & HW_CTRLD_QSPI_MODE_CTRL_SCLK))
        ;
    }
  }

// disable write command is issued
#ifdef CHIP_9118
  if (spi_config->spi_config_3.ddr_mode_en) {
    if (flash_type == MX_OCTA_FLASH) {
      qspi_write_to_flash(qspi_reg, QSPI_16BIT_LEN, (WRDI << 8) | WRDI2, cs_no);
    } else {
      qspi_write_to_flash(qspi_reg, QSPI_8BIT_LEN, WRDI, cs_no);
    }
  } else {
    qspi_write_to_flash(qspi_reg, CMD_LEN, WRDI, cs_no);
    if (spi_config->spi_config_3._16bit_cmd_valid) {
      qspi_write_to_flash(qspi_reg, CMD_LEN, WRDI2, cs_no);
    }
  }
#else
  qspi_write_to_flash(qspi_reg, CMD_LEN, WRDI, cs_no);
  if (spi_config->spi_config_3._16bit_cmd_valid) {
    qspi_write_to_flash(qspi_reg, CMD_LEN, WRDI2, cs_no);
  }
#endif
  DEASSERT_CSN;
  if (spi_config->spi_config_4.dual_flash_mode) {
    qspi_reg->OCTA_SPI_BUS_CONTROLLER2 &= ~BIT(19);
  }
  if (prev_state == 1) {
    qspi_reg->QSPI_BUS_MODE_REG |= AUTO_MODE;
    while (!(qspi_reg->QSPI_STATUS_REG & AUTO_MODE_ENABLED))
      ;
  }
  if (spi_config->spi_config_4.continue_fetch_en) {
    qspi_reg->QSPI_AUTO_CONITNUE_FETCH_CTRL_REG |= CONTINUE_FETCH_EN;
  }

  if (check_en) {
    if (qspi_reg == (qspi_reg_t *)TA_QSPI_BASE_ADDRESS) {
      auto_mode_address = TA_QSPI_AUTOM_CHIP0_ADDRESS;
    } else if (qspi_reg == (qspi_reg_t *)M4_QSPI_BASE_ADDRESS) {
      auto_mode_address = M4_QSPI_AUTOM_CHIP0_ADDRESS;
    }
    len_in_loop = 0;
    while (length >= 4) {
      if (*(volatile uint32_t *)(auto_mode_address + address + len_in_loop)
          == *(uint32_t *)((uint32_t)data_in + len_in_loop)) {
        len_in_loop += 4;
        length -= 4;
      } else {
        status = (uint32_t)RSI_FAIL;
        break;
      }
    }

    while (length > 0) {
      if (*(volatile uint8_t *)(auto_mode_address + address + len_in_loop)
          == *(uint8_t *)((uint32_t)data_in + len_in_loop)) {
        len_in_loop += 1;
        length -= 1;
      } else {
        status = (uint32_t)RSI_FAIL;
        break;
      }
    }
  }
  return status;
}

/**
 * @fn          error_t RSI_TIMERS_SetMatch( RSI_TIMERS_T *pTIMER, uint8_t timerNum, uint32_t match)
 * @brief		    This API is used to disable the timer interrupt 
 * @param[in]   pTIMER     : Pointer to the TIMERS instance register area
 * @param[in]   timerNum   : Timer number(0 to 3)
 * @param[in]   match      : delay time
 * @return 		  return the timer error code 
 */
STATIC INLINE error_t RSI_TIMERS_SetMatch(RSI_TIMERS_T *pTIMER, uint8_t timerNum, uint32_t match)
{
  if (timerNum <= TIMER_3) {
    pTIMER->MATCH_CTRL[timerNum].MCUULP_TMR_MATCH = match;
  } else {
    return ERROR_INVAL_TIMER_NUM;
  }
  return RSI_OK;
}

/**
 * @fn          error_t RSI_TIMERS_TimerStart(RSI_TIMERS_T *pTIMER, uint8_t timerNum)
 * @brief		    This API is used to start the timer 
 * @param[in]   pTIMER     : Pointer to the TIMERS instance register area
 * @param[in]   timerNum   : Timer number(0 to 3)                          
 * @return 		  return the timer error code
 */

STATIC INLINE error_t RSI_TIMERS_TimerStart(RSI_TIMERS_T *pTIMER, uint8_t timerNum)
{
  if (timerNum <= TIMER_3) {
    pTIMER->MATCH_CTRL[timerNum].MCUULP_TMR_CNTRL_b.TMR_START = ENABLE;
  } else {
    return ERROR_INVAL_TIMER_NUM;
  }
  return RSI_OK;
}

/**
 * @fn          error_t RSI_TIMERS_InterruptStatus(RSI_TIMERS_T *pTIMER , uint8_t timerNum)
 * @brief		    This API is used to get the timer interrupt status 
 * @param[in]   pTIMER     : Pointer to the TIMERS instance register area
 * @param[in]   timerNum   : Timer number(0 to 3)                          
 * @return 		  return the timer interrupt status if valid timer else 0. 
 */
STATIC INLINE uint8_t RSI_TIMERS_InterruptStatus(RSI_TIMERS_T *pTIMER, uint8_t timerNum)
{
  if (timerNum <= TIMER_3) {
    return (pTIMER->MCUULP_TMR_INTR_STAT & (1 << timerNum));
  } else {
    return RSI_OK;
  }
}

void qspi_usleep(uint32_t delay)
{
  /*  Micro seconds delay */
  RSI_TIMERS_SetMatch(TIMERS, TIMER_0, delay);
  /*Start timer */
  RSI_TIMERS_TimerStart(TIMERS, TIMER_0);
  /*Wait for time out*/
  while (!RSI_TIMERS_InterruptStatus(TIMERS, TIMER_0))
    ;
}


