/*******************************************************************************
* @file  rsi_device_init_flm.c
* @brief 
*******************************************************************************
* # License
* <b>Copyright 2020 Silicon Laboratories Inc. www.silabs.com</b>
*******************************************************************************
*
* The licensor of this software is Silicon Laboratories Inc. Your use of this
* software is governed by the terms of Silicon Labs Master Software License
* Agreement (MSLA) available at
* www.silabs.com/about-us/legal/master-software-license-agreement. This
* software is distributed to you in Source Code format and is governed by the
* sections of the MSLA applicable to Source Code.
*
******************************************************************************/

/**
 * Include files
 */
#include "stdint.h"
#include "rsi_bootup_config_flm.h"
#include "rsi_user_flm.h"
#include "rsi_api_flm.h"
#include "rsi_common_apis_flm.h"
#include "rsi_error_flm.h"
#include "rsi_systick_config.h"

#define BIT(a) ((uint32_t)1U << a)

int16_t rsi_bl_select_option(uint8_t cmd)
{
  uint16_t boot_cmd   = 0;
  int16_t retval      = 0;
  uint16_t read_value = 0;
  uint32_t commandResTimeout = 0;

  retval = rsi_mem_wr(RSI_HOST_INTF_REG_OUT, (uint8_t *)&boot_cmd);
  if (retval < 0) {
    return retval;
  }

  if (cmd == BURN_NWP_FW) {
    boot_cmd = RSI_HOST_INTERACT_REG_VALID_FW | cmd;
  } else {
    boot_cmd = RSI_HOST_INTERACT_REG_VALID | cmd;
  }
  retval = rsi_bootloader_instructions(RSI_REG_WRITE, &boot_cmd);
  if (retval < 0) {
    return retval;
  }

  commandResTimeout = 300;
  while ((cmd != LOAD_NWP_FW) && (cmd != LOAD_DEFAULT_NWP_FW_ACTIVE_LOW)) {
    retval = rsi_bootloader_instructions(RSI_REG_READ, &read_value);
    if (retval < 0) {
      return retval;
    }
    if (cmd == LOAD_NWP_FW) {
      if (read_value == (RSI_HOST_INTERACT_REG_VALID | RSI_SEND_RPS_FILE)) {
        break;
      }
    } else if (read_value == (RSI_HOST_INTERACT_REG_VALID | cmd)) {
      break;
    }
		
		if( isOneMillis() )
		{		
			if((commandResTimeout--) <= 0 )
			{
					return RSI_ERROR_FW_LOAD_OR_UPGRADE_TIMEOUT;
			}
		}	
  }
	
	commandResTimeout = 3000;
  
	if ((cmd == LOAD_NWP_FW) || (cmd == LOAD_DEFAULT_NWP_FW_ACTIVE_LOW)) {
			
    do {
      retval = rsi_bootloader_instructions(RSI_REG_READ, &read_value);
      if (retval < 0) {
        return retval;
      }
      if ((read_value & 0xF000) == (RSI_HOST_INTERACT_REG_VALID_FW & 0xF000)) {
        if ((read_value & 0xFF) == VALID_FIRMWARE_NOT_PRESENT) {
#ifdef RSI_DEBUG_PRINT
          RSI_DPRINT(RSI_PL4, "VALID_FIRMWARE_NOT_PRESENT\n");
#endif
          return RSI_ERROR_VALID_FIRMWARE_NOT_PRESENT;
        }
        if ((read_value & 0xFF) == RSI_INVALID_OPTION) {
#ifdef RSI_DEBUG_PRINT
          RSI_DPRINT(RSI_PL4, "INVALID CMD\n");
#endif

          return RSI_ERROR_INVALID_OPTION;
        }
        if ((read_value & 0xFF) == RSI_CHECKSUM_SUCCESS) {
#ifdef RSI_DEBUG_PRINT
          RSI_DPRINT(RSI_PL4, "LOAD SUCCESS\n");
#endif
          break;
        }
      }		
		
			if( isOneMillis() )
			{		
				if((commandResTimeout--) <= 0 )
				{
						return RSI_ERROR_FW_LOAD_OR_UPGRADE_TIMEOUT;
				}
			}
    } while (1);
  }
  return retval;
}


int16_t checkError(uint16_t read_value)
{
	int16_t retValue =0;
	read_value = read_value & 0xFF;
	
	switch(read_value)
	{
			case RSI_INVALID_OPTION	          :
#ifdef DEBUG_OFL
																		*(volatile uint32_t *)0x24048628 |= (1<<5);
																		*(volatile uint32_t *)0x24048628 &= ~(1<<5);
#endif
			case RSI_CHECKSUM_FAILURE         :			
			case RSI_CHECKSUM_INVALID_ADDRESS :
			case VALID_FIRMWARE_NOT_PRESENT	 	: 
#ifdef DEBUG_OFL				
																		*(volatile uint32_t *)0x24048620 |= (1<<5);
																		*(volatile uint32_t *)0x24048620 &= ~(1<<5);
#endif
																		retValue = -1;
			break;
			
			default														: 
																		retValue = 0;
			break;
	}
	
	return retValue;
}
/*==============================================*/
/**
 * @fn          int16_t rsi_bl_upgrade_firmware(uint8_t *firmware_image , uint32_t fw_image_size, uint8_t flags)
 * @brief       Upgrades firmware to WiFi module
 * @param[in]   uint8_t image_type, type of firmware image to upgrade
 * @param[in]   uint8_t *firmware_image, pointer to firmware
 * @param[in]   uint32_t fw_image_size, size of firmware image
 * @param[out]  none
 * @return      errCode
 *              <0 = Command issue failed
 *              0  = SUCCESS
 * @section description 
 * This API is used to upgrade firmware to WiFi module.
 */
int16_t rsi_bl_upgrade_firmware(uint8_t *firmware_image, uint32_t fw_image_size, uint8_t flags)
{
  static uint16_t boot_cmd;
  uint16_t read_value = 0;
  uint16_t prevReadValue =read_value;
  uint32_t offset     = 0;
  int16_t retval      = 0;
  uint32_t boot_insn = 0, poll_resp = 0;
  uint32_t commandResTimeout = 0;
  int16_t errorStatus =0;
	
  // If it is a start of file set the boot cmd to pong valid
  if (flags & RSI_FW_START_OF_FILE) {
    boot_cmd = RSI_HOST_INTERACT_REG_VALID | RSI_PONG_VALID;
  }

  // check for invalid packet
  if ((fw_image_size % (RSI_MIN_CHUNK_SIZE) != 0) && (!(flags & RSI_FW_END_OF_FILE))) {
    return RSI_ERROR_INVALID_PACKET;
  }

  // loop to execute multiple of 4K chunks
  while (offset < fw_image_size) {
    switch (boot_cmd) {
      case (RSI_HOST_INTERACT_REG_VALID | RSI_PING_VALID):
        boot_insn = RSI_PONG_WRITE;
        poll_resp = RSI_PING_AVAIL;
        boot_cmd  = RSI_HOST_INTERACT_REG_VALID | RSI_PONG_VALID;
        break;

      case (RSI_HOST_INTERACT_REG_VALID | RSI_PONG_VALID):
        boot_insn = RSI_PING_WRITE;
        poll_resp = RSI_PONG_AVAIL;
        boot_cmd  = RSI_HOST_INTERACT_REG_VALID | RSI_PING_VALID;
        break;
      default: {
      }
    }

    retval = rsi_bootloader_instructions(boot_insn, (uint16_t *)((uint8_t *)firmware_image + offset));
    if (retval < 0) {
      return retval;
    }
	//commandResTimeout = 1000;
		commandResTimeout=50000000;
    while (1) {
      retval = rsi_bootloader_instructions(RSI_REG_READ, &read_value);
      if (retval < 0) {
        return retval;
      }
      if (read_value == (RSI_HOST_INTERACT_REG_VALID | poll_resp)) {
        break;
      }
			
			if( isOneMillis() )
			{		
				if((commandResTimeout--) <= 0 )
				{
						return RSI_ERROR_FW_LOAD_OR_UPGRADE_TIMEOUT;
				}
			}	
    }
    offset += RSI_MIN_CHUNK_SIZE;			
  }

  // For last chunk set boot cmd as End of file reached
  if (flags & RSI_FW_END_OF_FILE) {
    boot_cmd = RSI_HOST_INTERACT_REG_VALID | RSI_EOF_REACHED;

    retval = rsi_bootloader_instructions(RSI_REG_WRITE, &boot_cmd);
    if (retval < 0) {
      return retval;
    }

		/*Review Comment : Turn the magic value into MACRO*/
	//commandResTimeout = 30000;
		commandResTimeout=50000000;
    do {
      retval = rsi_bootloader_instructions(RSI_REG_READ, &read_value);

      if (retval < 0) {
        return retval;
      }
						
			if(prevReadValue != read_value)
			{
				prevReadValue = read_value;
				errorStatus = checkError(read_value);
				if(errorStatus < 0)
				{
					return read_value & 0xFF;
				}
			}
			
	    if( isOneMillis() )
			{		
				if((commandResTimeout--) <= 0 )
				{
						return RSI_ERROR_FW_LOAD_OR_UPGRADE_TIMEOUT;
				}
			}				
    } while (read_value != (RSI_HOST_INTERACT_REG_VALID | RSI_FWUP_SUCCESSFUL));
  }
  return retval;
}

/*==============================================*/
/**
 * @fn          int16_t rsi_bootloader_instructions(uint8_t type, uint16_t *data)
 * @brief       Sends boot instructions to WiFi module
 * @param[in]   uint8_t type, type of the insruction to perform
 * @param[in]   uint32_t *data, pointer to data which is to be read/write
 * @param[out]  none
 * @return      errCode
 *              < 0  = Command issued failure/Invalid command 
 *                0  = SUCCESS
 *              > 0  = Read value
 * @section description 
 * This API is used to send boot instructions to WiFi module.
 */

int16_t rsi_bootloader_instructions(uint8_t type, uint16_t *data)
{
  int16_t retval     = 0;
  uint32_t cmd       = 0;
  uint32_t j         = 0;
  uint16_t len       = 0;
  uint16_t offset    = 0;
  uint16_t local     = 0;
  uint16_t read_data = 0;
  uint32_t commandResTimeout = 0;

  switch (type) {
    case RSI_REG_READ:
      retval = rsi_mem_rd(RSI_HOST_INTF_REG_OUT, (uint8_t *)&read_data);
      *data  = read_data;
      break;

    case RSI_REG_WRITE:
      retval = rsi_mem_wr(RSI_HOST_INTF_REG_IN, (uint8_t *)data);
      break;

    case RSI_PING_WRITE:

      for (j = 0; j < RSI_PING_PONG_CHUNK_SIZE / RSI_HAL_MAX_WR_BUFF_LEN; j++) {
        if (j == RSI_PING_PONG_CHUNK_SIZE / RSI_HAL_MAX_WR_BUFF_LEN) {
          len = (RSI_PING_PONG_CHUNK_SIZE % RSI_HAL_MAX_WR_BUFF_LEN);
          if (len == 0) {
            break;
          }
        } else {
          len = RSI_HAL_MAX_WR_BUFF_LEN;
        }
        retval = rsi_mem_wr(RSI_PING_BUFFER_ADDR + offset, (uint8_t *)((uint32_t)data + offset));
        if (retval < 0) {
          return retval;
        }
        offset += len;
      }

      local  = (RSI_PING_AVAIL | RSI_HOST_INTERACT_REG_VALID);
      retval = rsi_mem_wr(RSI_HOST_INTF_REG_IN, (uint8_t *)&local);
      break;

    case RSI_PONG_WRITE:

      for (j = 0; j < RSI_PING_PONG_CHUNK_SIZE / RSI_HAL_MAX_WR_BUFF_LEN; j++) {
        if (j == RSI_PING_PONG_CHUNK_SIZE / RSI_HAL_MAX_WR_BUFF_LEN) {
          len = (RSI_PING_PONG_CHUNK_SIZE % RSI_HAL_MAX_WR_BUFF_LEN);
          if (len == 0) {
            break;
          }
        } else {
          len = RSI_HAL_MAX_WR_BUFF_LEN;
        }
        retval = rsi_mem_wr(RSI_PONG_BUFFER_ADDR + offset, (uint8_t *)((uint32_t)data + offset));
        if (retval < 0) {
          return retval;
        }
        offset += len;
      }

      // Perform the write operation
      local = (RSI_PONG_AVAIL | RSI_HOST_INTERACT_REG_VALID);

      retval = rsi_mem_wr(RSI_HOST_INTF_REG_IN, (uint8_t *)&local);
      break;

    case BURN_NWP_FW:

      cmd = BURN_NWP_FW | RSI_HOST_INTERACT_REG_VALID;

      retval = rsi_mem_wr(RSI_HOST_INTF_REG_IN, (uint8_t *)&cmd);
      if (retval < 0) {
        return retval;
      }
    	commandResTimeout = 300;
      do {
        retval = rsi_mem_rd(RSI_HOST_INTF_REG_OUT, (uint8_t *)&read_data);
        if (retval < 0) {
          return retval;
        }
		    
			if( isOneMillis() )
			{		
				if((commandResTimeout--) <= 0 )
				{
						return RSI_ERROR_FW_LOAD_OR_UPGRADE_TIMEOUT;
				}
			}				
				
      } while (read_data != (RSI_SEND_RPS_FILE | RSI_HOST_INTERACT_REG_VALID));
      break;
    case LOAD_NWP_FW:
      cmd    = LOAD_NWP_FW | RSI_HOST_INTERACT_REG_VALID;
      retval = rsi_mem_wr(RSI_HOST_INTF_REG_IN, (uint8_t *)&cmd);
      break;
    case LOAD_DEFAULT_NWP_FW_ACTIVE_LOW:
      cmd    = LOAD_DEFAULT_NWP_FW_ACTIVE_LOW | RSI_HOST_INTERACT_REG_VALID;
      retval = rsi_mem_wr(RSI_HOST_INTF_REG_IN, (uint8_t *)&cmd);
      break;
    default:
      retval = RSI_ERROR_INVALID_PARAM;
      break;
  }
  return retval;
}
