//-----------------------------------------------------------------------------
// LCDInterface.c
//-----------------------------------------------------------------------------
// Copyright 2004 Silicon Laboratories, Inc.
//
// AUTH: GP
// DATE: 9 DEC 04
//
// This program interfaces a C8051F330 device with an example LCD.
//
// Target: C8051F33x
// Tool Chain : Keil
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include <c8051F330.h>
#include <stdio.h>

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F3xx
//-----------------------------------------------------------------------------

sfr16 TMR2RL   = 0xca;                 // Timer2 reload value
sfr16 TMR2     = 0xcc;                 // Timer2 counter

sfr16 TMR3RL   = 0x92;                 // Timer3 reload value
sfr16 TMR3     = 0x94;                 // Timer3 counter

//-----------------------------------------------------------------------------
// Structures, Unions, Enumerations, and Type Definitions
//-----------------------------------------------------------------------------

// The translation table provides the mapping between ASCII characters
// and the segment pin values

// The first 32 characters (except 10) just produce a space;
// Character 10 (newline) clears the LCD digits
// Characters that can't be translated produce a space
// The MSB in the byte is meaningless because there are only 7 segments

// If the bit is low, the corresponding bar in the digit is active
// The 'diagram' below shows which bit corresponds to which bar in the LCD digit

//      0
//      _
//   5 | | 1        the middle inner bar is bit 6
//      -
//   4 | | 2
//      -
//      3


unsigned char code translation_table[128] = {

0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,     // 0 - 7
0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,     // 8 - 15
0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,     // 16 - 23
0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,     // 24 - 31

0xFF, 0xCF, 0xDD, 0xFF, 0x92, 0x9B, 0xFF, 0xDF,     // 32 - 39
0xC6, 0xF0, 0xFF, 0xFF, 0xDF, 0xBF, 0xEF, 0xFF,     // 40 - 47
0x40, 0xF9, 0xA4, 0xB0, 0x99, 0x92, 0x02, 0xF8,     // 48 - 55
0x00, 0x10, 0xF9, 0xF9, 0x9E, 0xB7, 0xBC, 0x18,     // 56 - 63

0xFF, 0x08, 0x00, 0xC6, 0x21, 0x06, 0x0E, 0x02,     // 64 - 71
0x09, 0xF9, 0xE1, 0x07, 0xC7, 0xFF, 0x48, 0x40,     // 72 - 79
0x0C, 0x18, 0xAF, 0x92, 0xF8, 0x41, 0xE3, 0xFF,     // 80 - 87
0x09, 0x11, 0xA4, 0x06, 0xFF, 0xF0, 0xDC, 0xF7,     // 88 - 95

0xDF, 0x20, 0x03, 0xA7, 0x21, 0x04, 0x0E, 0x10,     // 96 - 103
0x0B, 0xFB, 0x61, 0x07, 0xF9, 0xFF, 0xAB, 0x23,     // 104 - 111
0x0C, 0x18, 0xAF, 0x92, 0x07, 0xE3, 0xE3, 0xFF,     // 112 - 119
0x09, 0x11, 0xA4, 0xC6, 0xF9, 0xF0, 0xBF, 0xFF,     // 120 - 127
};

//-----------------------------------------------------------------------------
// Global Constants
//-----------------------------------------------------------------------------

#define SYSCLK       24500000          // SYSCLK frequency in Hz
#define TIMER2_RATE  1000              // Timer 2 overflow rate in Hz
#define TIMER3_RATE  2000              // Timer 3 overflow rate in Hz
#define PULSE_LENGTH 25

//-----------------------------------------------------------------------------
// Port names
//-----------------------------------------------------------------------------

sbit SRCLK = P1^1;                     // shift register clock
sbit RCLK  = P1^2;                     // shift register latch
sbit SER   = P1^3;                     // shift register serial in

sbit COM1  = P1^4;                     // COM1 pin on LCD
sbit COM2  = P1^5;                     // COM2 pin on LCD
sbit COM3  = P1^6;                     // COM3 pin on LCD
sbit COM4  = P1^7;                     // COM4 pin on LCD


//-----------------------------------------------------------------------------
// Global LCD Variables
//-----------------------------------------------------------------------------

unsigned char com_cycle = 1;           // start at COM 1
unsigned char com_invert = 0;          // start with positive cycle

// Below are the bit maps for each of the bars on the LCD;  If the bit is low
// then the bar is opaque (ON). If the bit is high, the bar is transparent (OFF).

// one char per digit on the LCD; initialized to OFF
unsigned char bdata LCD_digits[6] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

// The naming scheme: D1A means the A segment of digit 1

// Digit 1 (D1) is the leftmost digit on the LCD
sbit D1A = LCD_digits[0] ^ 0;
sbit D1B = LCD_digits[0] ^ 1;          // D1 is controlled by S1 and S2
sbit D1C = LCD_digits[0] ^ 2;
sbit D1D = LCD_digits[0] ^ 3;
sbit D1E = LCD_digits[0] ^ 4;
sbit D1F = LCD_digits[0] ^ 5;
sbit D1G = LCD_digits[0] ^ 6;

sbit D2A = LCD_digits[1] ^ 0;          // D2 is controlled by S3 and S4
sbit D2B = LCD_digits[1] ^ 1;
sbit D2C = LCD_digits[1] ^ 2;
sbit D2D = LCD_digits[1] ^ 3;
sbit D2E = LCD_digits[1] ^ 4;
sbit D2F = LCD_digits[1] ^ 5;
sbit D2G = LCD_digits[1] ^ 6;

sbit D3A = LCD_digits[2] ^ 0;          // D3 is controlled by S5 and S6
sbit D3B = LCD_digits[2] ^ 1;
sbit D3C = LCD_digits[2] ^ 2;
sbit D3D = LCD_digits[2] ^ 3;
sbit D3E = LCD_digits[2] ^ 4;
sbit D3F = LCD_digits[2] ^ 5;
sbit D3G = LCD_digits[2] ^ 6;

sbit D4A = LCD_digits[3] ^ 0;          // D4 is controlled by S7 and int8_t
sbit D4B = LCD_digits[3] ^ 1;
sbit D4C = LCD_digits[3] ^ 2;
sbit D4D = LCD_digits[3] ^ 3;
sbit D4E = LCD_digits[3] ^ 4;
sbit D4F = LCD_digits[3] ^ 5;
sbit D4G = LCD_digits[3] ^ 6;

sbit D5A = LCD_digits[4] ^ 0;          // D5 is controlled by S9 and S10
sbit D5B = LCD_digits[4] ^ 1;
sbit D5C = LCD_digits[4] ^ 2;
sbit D5D = LCD_digits[4] ^ 3;
sbit D5E = LCD_digits[4] ^ 4;
sbit D5F = LCD_digits[4] ^ 5;
sbit D5G = LCD_digits[4] ^ 6;

sbit D6A = LCD_digits[5] ^ 0;          // D6 is controlled by S11 and S12
sbit D6B = LCD_digits[5] ^ 1;
sbit D6C = LCD_digits[5] ^ 2;
sbit D6D = LCD_digits[5] ^ 3;
sbit D6E = LCD_digits[5] ^ 4;
sbit D6F = LCD_digits[5] ^ 5;
sbit D6G = LCD_digits[5] ^ 6;

//-----------------------------------------------------------------------------
// Function Prototypes
//-----------------------------------------------------------------------------

void SYSCLK_Init (void);
void Port_IO_Init();
void Timer2_Init (int);
void Timer3_Init (int);
char putchar(char);
void Strobe();
void wait_one_ms(unsigned int);

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main(void)
{
   PCA0MD &= ~0x40;                    // WDTE = 0 (clear watchdog timer enable)

   SYSCLK_Init();                      // initialize the oscillator
   Port_IO_Init();                     // initialize the ports
   Timer2_Init (SYSCLK / TIMER2_RATE); // enable timer to interrupt at some Hz
   Timer3_Init (SYSCLK / TIMER3_RATE); // enable timer to overflow at some Hz

   // We first configure the COM ports to analog inputs.  This allows us
   // to set them to high impedance if we write a 1 to the COM port.  Along with
   // some external resistors, we can then create a VDD/2 voltage.  When it is
   // time for the corresponding COM cycle, we can set the pin to push-pull and
   // drive the output to VDD or GND.  These 3 levels (VDD, VDD/2, GND) are
   // necessary only for the backplane (Common) pins on the LCD

   COM1  = 1;                          // high impedance
   COM2  = 1;                          // high impedance
   COM3  = 1;                          // high impedance
   COM4  = 1;                          // high impedance
   RCLK  = 0;                          // don't output anything to LCD
   SRCLK = 0;                          // don't shift anything to registers yet

   EA = 1;                             // enable global interrupts

   while (1)
   {
      printf ("\nHELLO");

      wait_one_ms (1000);

      printf ("\n");

      wait_one_ms (1000);
   }
}


//-----------------------------------------------------------------------------
// Init Functions
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use the internal 24.5MHz
// oscillator as its clock source.  Also enables missing clock detector reset.
//
void SYSCLK_Init (void)
{
   OSCICN |= 0x03;                     // Configure internal osc to max freq
   RSTSRC  = 0x04;                     // Enable missing clock detector
}


//-----------------------------------------------------------------------------
// Port_IO_init
//-----------------------------------------------------------------------------
//
// This routine initializes the ports and enables the crossbar
//
void Port_IO_Init(void)
{
    // P0.0  -  Unassigned,  Open-Drain, Digital
    // P0.1  -  Unassigned,  Open-Drain, Digital
    // P0.2  -  Unassigned,  Open-Drain, Digital
    // P0.3  -  Unassigned,  Open-Drain, Digital
    // P0.4  -  Unassigned,  Open-Drain, Digital
    // P0.5  -  Unassigned,  Open-Drain, Digital
    // P0.6  -  Unassigned,  Open-Drain, Digital
    // P0.7  -  Unassigned,  Open-Drain, Digital

    // P1.0  -  Unassigned,  Open-Drain, Digital
    // P1.1  -  Skipped,     Push-Pull,  Digital    SRCLK for 74HC595
    // P1.2  -  Skipped,     Push-Pull,  Digital    RCLK for 74HC595
    // P1.3  -  Skipped,     Push-Pull,  Digital    SER for 74HC595
    // P1.4  -  Skipped,     Open-Drain, Digital    COM1 for LCD
    // P1.5  -  Skipped,     Open-Drain, Digital    COM2 for LCD
    // P1.6  -  Skipped,     Open-Drain, Digital    COM3 for LCD
    // P1.7  -  Skipped,     Open-Drain, Digital    COM4 for LCD

    P0MDOUT   = 0x80;
    P1MDOUT   = 0x0E;                   // configure above pins to Push-Pull
    P1MDIN    = 0x0F;                   // configure Pins 1.4 - 1.7 to analog in
    P1SKIP    = 0xFE;                   // skip pins 1.1 to 1.7
    XBR1      = 0x40;                   // enable crossbar

}

//-----------------------------------------------------------------------------
// Timer2_Init
//-----------------------------------------------------------------------------
//
// The timer overflows at a rate of TIMER2_RATE times a second
// The interrupt generated in handled by the LCD_refresh ISR
//
void Timer2_Init (int counts)
{
   TMR2CN = 0x00;                      // STOP Timer2; Clear TF2H and TF2L;
                                       // disable low-byte interrupt; disable
                                       // split mode; select internal timebase
   CKCON |= 0x10;                      // Timer2 uses SYSCLK as its timebase

   TMR2RL  = -counts;                  // Init reload values
   TMR2    = TMR2RL;                   // Init Timer2 with reload value
   ET2 = 1;                            // enable Timer2 interrupts
   TR2 = 1;                            // start Timer2
}

//------------------------------------------------------------------------------------
// Timer3_Init
//------------------------------------------------------------------------------------
//
// Configure the Timer to overflow without interrupts
// The overflow will be used in the wait function
//
void Timer3_Init (int count)
{
   TMR3CN = 0x00;                      // STOP Timer3; Clear TF3H and TF3L;
                                       // disable low-byte interrupt; disable
                                       // split mode; select internal timebase

   CKCON |= 0x40;                      // Timer3 uses SYSCLK as its timebase

   TMR3RL  =  -count;                  // Init reload values
   TMR3    =  TMR3RL;                  // Init Timer3 with reload value
   EIE1    &= 0x7F;                    // disable Timer3 interrupts

   TMR3CN  |= 0x01;                    // start Timer3
}

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

// LCDrefresh is triggered on a Timer2 Overflow

// Takes what is in the LCD bar bits and shift them into the two 74HC595
// shift registers depending on the COM cycle; The most signficant
// LCD pin (pin 16) gets shifted out first; Only 15 bits get shifted each
// COM cycle;

void LCDrefresh_ISR (void) interrupt 5
{
   int i = 0;

   if (com_cycle == 1)
   {
      SER = 1    ^ com_invert; Strobe();         // non-existent segment
      SER = D6A  ^ com_invert; Strobe();
      SER = 1    ^ com_invert; Strobe();         // non-existent segment
      SER = D5A  ^ com_invert; Strobe();
      SER = 1    ^ com_invert; Strobe();         // non-existent segment
      SER = D4A  ^ com_invert; Strobe();
      SER = 1    ^ com_invert; Strobe();         // non-existent segment
      SER = D3A  ^ com_invert; Strobe();
      SER = 1    ^ com_invert; Strobe();         // non-existent segment
      SER = D2A  ^ com_invert; Strobe();
      SER = 1    ^ com_invert; Strobe();         // non-existent segment
      SER = D1A  ^ com_invert; Strobe();
      SER = 1    ^ com_invert; Strobe();         // non-existent segment
      SER = 1    ^ com_invert; Strobe();         // non-existent segment
      SER = 1    ^ com_invert; Strobe();         // non-existent segment

      RCLK = 1;                        // put shifted data to LCD - rising edge
      for (i=0; i<PULSE_LENGTH; i++);  // keep clock high for a while
      RCLK = 0;                        // turn off clock

      P1MDIN  &= ~0x80;                // configure COM4 to ANALOG_IN;
      P1MDIN  |= 0x10;                 // and COM1 to digital

      P1MDOUT &= ~0x80;                // make COM4 an open-drain
      P1MDOUT |= 0x10;                 // make COM1 a push-pull
      COM4    = 1;                     // set COM4 to high impedance
      COM1    = 1 ^ com_invert;        // start the COM1 cycle
      com_cycle = 2;                   // next state
   }

   else if (com_cycle == 2)
   {
      SER = D6B  ^ com_invert;     Strobe();
      SER = D6F  ^ com_invert;     Strobe();
      SER = D5B  ^ com_invert;     Strobe();
      SER = D5F  ^ com_invert;     Strobe();
      SER = D4B  ^ com_invert;     Strobe();
      SER = D4F  ^ com_invert;     Strobe();
      SER = D3B  ^ com_invert;     Strobe();
      SER = D3F  ^ com_invert;     Strobe();
      SER = D2B  ^ com_invert;     Strobe();
      SER = D2F  ^ com_invert;     Strobe();
      SER = D1B  ^ com_invert;     Strobe();
      SER = D1F  ^ com_invert;     Strobe();
      SER = 1    ^ com_invert;     Strobe();      // non-existent segment
      SER = 1    ^ com_invert;     Strobe();      // non-existent segment
      SER = 1    ^ com_invert;     Strobe();      // non-existent segment

      RCLK = 1;                        // put shifted data to LCD - rising edge
      for (i=0; i<PULSE_LENGTH; i++);  // keep clock high for a while
      RCLK = 0;                        // turn off clock

      P1MDIN  &= ~0x10;                // configure COM1 to ANALOG_IN;
      P1MDIN  |= 0x20;                 // and COM2 to digital

      P1MDOUT &= ~0x10;                // make COM1 an open-drain
      P1MDOUT |= 0x20;                 // make COM2 a push-pull
      COM1    = 1;                     // set COM1 to high impedance
      COM2    = 1 ^ com_invert;        // start the COM2 cycle
      com_cycle = 3;                   // next state
   }

   else if (com_cycle == 3)
   {
      SER = D6C ^ com_invert;     Strobe();
      SER = D6G ^ com_invert;     Strobe();
      SER = D5C ^ com_invert;     Strobe();
      SER = D5G ^ com_invert;     Strobe();
      SER = D4C ^ com_invert;     Strobe();
      SER = D4G ^ com_invert;     Strobe();
      SER = D3C ^ com_invert;     Strobe();
      SER = D3G ^ com_invert;     Strobe();
      SER = D2C ^ com_invert;     Strobe();
      SER = D2G ^ com_invert;     Strobe();
      SER = D1C ^ com_invert;     Strobe();
      SER = D1G ^ com_invert;     Strobe();
      SER = 1   ^ com_invert;     Strobe();      // non-existent segment
      SER = 1   ^ com_invert;     Strobe();      // non-existent segment
      SER = 1   ^ com_invert;     Strobe();      // non-existent segment

      RCLK = 1;                        // put shifted data to LCD - rising edge
      for (i=0; i<PULSE_LENGTH; i++);  // keep clock high for a while
      RCLK = 0;                        // turn off clock

      P1MDIN  &= ~0x20;                // configure COM2 to ANALOG_IN;
      P1MDIN  |= 0x40;                 // and COM3 to digital

      P1MDOUT &= ~0x20;                // make COM2 an open-drain
      P1MDOUT |= 0x40;                 // make COM3 a push-pull
      COM2    = 1;                     // set COM2 to high impedance
      COM3    = 1 ^ com_invert;        // start the COM3 cycle
      com_cycle = 4;                   // next state
   }

   else if (com_cycle == 4)
   {
      SER = D6D ^ com_invert;     Strobe();
      SER = D6E ^ com_invert;     Strobe();
      SER = D5D ^ com_invert;     Strobe();
      SER = D5E ^ com_invert;     Strobe();
      SER = D4D ^ com_invert;     Strobe();
      SER = D4E ^ com_invert;     Strobe();
      SER = D3D ^ com_invert;     Strobe();
      SER = D3E ^ com_invert;     Strobe();
      SER = D2D ^ com_invert;     Strobe();
      SER = D2E ^ com_invert;     Strobe();
      SER = D1D ^ com_invert;     Strobe();
      SER = D1E ^ com_invert;     Strobe();
      SER = 1   ^ com_invert;     Strobe();         // non-existent segment
      SER = 1   ^ com_invert;     Strobe();         // non-existent segment
      SER = 1   ^ com_invert;     Strobe();         // non-existent segment

      RCLK = 1;                        // put shifted data to LCD - rising edge
      for (i=0; i<PULSE_LENGTH; i++);  // keep clock high for a while
      RCLK = 0;                        // turn off clock

      P1MDIN  &= ~0x40;                // configure COM3 to ANALOG_IN;
      P1MDIN  |= 0x80;                 // and COM4 to digital

      P1MDOUT &= ~0x40;                // make COM3 an open-drain
      P1MDOUT |= 0x80;                 // make COM4 a push-pull
      COM3    = 1;                     // set COM3 to high impedance
      COM4    = 1 ^ com_invert;        // start the COM4 cycle
      com_cycle = 1;                   // next state

      com_invert = com_invert ^ 1;     // toggle com_invert
   }

   TF2H = 0;                           // clear TF2

} // end LCDrefresh_ISR

//-----------------------------------------------------------------------------
// Strobe
//-----------------------------------------------------------------------------
//
// Strobe is used to clock the data into the 74HC595 shift registers
//
void Strobe()
{
   int i = 0;

   SRCLK = 1;
   for (i = 0; i < PULSE_LENGTH; i++);          // wait a few cycles
   SRCLK = 0;
   for (i = 0; i < PULSE_LENGTH; i++);          // wait a few cycles

}

//-----------------------------------------------------------------------------
// wait_one_msec
//-----------------------------------------------------------------------------
//
// Assumes Timer3 overflows once every 500 usec
//
void wait_one_ms(unsigned int count)
{
   count = count * 2;                  // overflows once every 500 usec
                                       // so double that is 1 ms

   TMR3CN &= ~0x80;                    // Clear Timer3 overflow flag
   TMR3 = TMR3RL;
   TMR3CN  = 0x04;                     // Start Timer3

   while (count--)
   {
      while (!(TMR3CN & 0x80)) {}      // wait for overflow
      TMR3CN &= ~0x80;                 // clear overflow indicator
   }

   TMR3CN &= ~0x04;                    // Stop Timer3
}

//-----------------------------------------------------------------------------
// LCD functions
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// putchar
//-----------------------------------------------------------------------------
//
// putchar only handles the digit components on the LCD screen.
// This functions shifts the digit values to the left, shifting out the
// left-most digit.  This function has 3 potential actions based on the input:
//
// 1. Any input whose ASCII code is between 0 and 127 gets translated
//    according to the translation table above
//
// 2. Any input whose ASCII code is between 128 and 255 is directly sent to
//    the LCD. The lower 7 bits indicate which of the seven segments are lit.
//
// 3. Passing a newline char '\n' to this function clears all 6 digits
//
// This function, unlike standard putchar, does not have any error return msgs.
//
// This function will not cause an interrupt to force output.  The input char
// will be displayed on the screen on the next refresh cycle

char putchar(char charIN)
{
   unsigned char iter = 0;

   if (charIN != '\n')                              // not a new line
   {
      if ((charIN & 0x80) == 0) {                   // translation necesssary
        charIN = translation_table [charIN]; }      // quick lookup

      EA = 0;                                       // prevent partial display

      for (iter = 0; iter < 5; iter++) {            // shift the digits left
         LCD_digits[iter] = LCD_digits[iter+1]; }

      LCD_digits[5] = charIN;                       // new digit is rightmost

      EA = 1;                                       // enable interrupts again
   }

   else                                             // input is a newline
   {
      EA = 0;                                       // disable interrupts
      for (iter = 0; iter < 6; iter++) {
         LCD_digits[iter] = 0xFF; }                 // clear all digits
      EA = 1;                                       // enable interrupts
   }

   if (charIN == 0xFF) {                            // couldn't interpret OR space
      charIN = ' '; }                               // return space

   return charIN;                                   // just like putchar
}