//-----------------------------------------------------------------------------
// Sensorless BLDC Motor Reference Design
//-----------------------------------------------------------------------------
// Copyright 2013 Silicon Laboratories Inc.
//
// AUTH: KAB
// DATE: 30 AUG 2006
//
// This program provides Sensorless BLDC motor control using the
// 'F310. This software is written specifically for the SBLDC
// reference design hardware. Please see the full application
// note for further information.
//
// Target: C8051F31x
//
// Tool chain: KEIL 'c' full-version required
//
// Revision 1.1 - KAB/TP (13 NOV 2013)
// - Updated Target to 'F31x
//
//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include <c8051f310.h>                 // SFR declarations
#include <slbdc.h>                     // macro constants

//-----------------------------------------------------------------------------
// commutation patterns
//-----------------------------------------------------------------------------

const unsigned char code skipPattern[6]=
    {071,055,047,066,036,033};         // code in octal

const unsigned char code openPhase[6]=
    {0x0C,0x0B,0x0A,0x0C,0x0B,0x0A};   // open phase mux value


const unsigned char code TRamp[192]=   // linear acceleration table
{
   0xF0, 0x63, 0x4C, 0x40, 0x38, 0x33, 0x2F, 0x2B,
   0x29, 0x26, 0x25, 0x23, 0x21, 0x20, 0x1F, 0x1E,
   0x1D, 0x1C, 0x1B, 0x1B, 0x1A, 0x19, 0x19, 0x18,
   0x18, 0x17, 0x17, 0x16, 0x16, 0x16, 0x15, 0x15,
   0x15, 0x14, 0x14, 0x14, 0x13, 0x13, 0x13, 0x13,
   0x12, 0x12, 0x12, 0x12, 0x11, 0x11, 0x11, 0x11,
   0x11, 0x11, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
   0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
   0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E,
   0x0E, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
   0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0C, 0x0C, 0x0C,
   0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
   0x0C, 0x0C, 0x0C, 0x0C, 0x0B, 0x0B, 0x0B, 0x0B,
   0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
   0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0A,
   0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
   0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
   0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
   0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
   0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
   0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
   0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
   0x09, 0x09, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
   0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08
};


//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void StartMotor(void);
void T0_Init(void);
void T0_ISR (void);
void stop(void);
void align(void);
void start(void);
void start0(void);
void start7(void);
void run(void);
void run0(void);
void run1(void);
void run2(void);
void run3(void);
unsigned int avgCurrent(void);
signed int avgVoltage(void);
signed int avgVmid(void);
signed int avgVdelta(void);
signed int flip(signed int);
void updateT(void);
void commutate();
void error(void);


//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------

// Public accessed by GUI
unsigned char Status;                  // motor state variable
unsigned int Imotor;                   // motor current measurement
unsigned int Vpot;                     // speed pot voltage measurement
signed int Vemf;
signed int Verror;                     // error voltage from min-point voltage
bit Stall;                             // stall flag bit
bit OverCurrent;                       // over-current flag bit

// Local global T0_ISR variables
unsigned char Tslot;
unsigned int NextT;
unsigned char AccIndex;
unsigned char MtrIndex;
signed int idata Vmid[6];              // used for avg Verror calc
signed int idata Vdelta[6];            // used for average Vdelta calc
signed int Vmin;
signed int Vmax;
bit Flip;
sbit AlignLED = P0^2;
sbit BlinkLED = P0^3;

// external T2_ISR variables
extern unsigned char Speed;

//-----------------------------------------------------------------------------
// StartMotor function - called from main
//-----------------------------------------------------------------------------

void StartMotor(void)
{
   TL0 = 0xF0;                         // T0 overflow in 16 ticks
   TH0 = 0xFF;                         // msb set to FF
   ET0 = 1;                            // enable T0 interrupt
   TR0 = 1;                            // start Timer0
   MtrIndex = 5;                       // MtrIndex will flip to 0
   Status = ALIGN;                     // set Status to ALIGN state
   AlignLED = ON;                      // turn on LED
   Tslot = 0;                          // commutate on next ISR
   Speed = 0;                          // reset speed ramp controller
}

//-----------------------------------------------------------------------------
// T0_Init - called from main
//-----------------------------------------------------------------------------

void T0_Init(void)
{
   TMOD    &= ~0x03;                   // clear T0 mode
   TMOD    |= 0x01;                    // T0 mode 1
   CKCON   &= ~0x07;                   // clear SCAx and T0M
   CKCON   |= 0x02;                    // T0 uses SYSCLK/48
   IP      |= 0x02;                    // T0 high priority
   CKCON   |= 0x10;                    // T2 uses SYSCLK
   TMR2RLL   = T2_RELOAD_L;            // Timer 2 Low Byte
   TMR2RLH   = T2_RELOAD_H;            // Timer 2 High Byte
}
//-----------------------------------------------------------------------------
// T0_ISR
//-----------------------------------------------------------------------------

void T0_ISR (void) interrupt 1

{
   if (Tslot==0) commutate();          // commutate first if time slot zero

   switch(Status)                      // implement state diagram
   {
      case STOP:
         stop();                       // stop motor
         break;

      case ALIGN:
         align();                      // align motor
         break;

     case START:
         start();                      // start motor
         break;

      case RUN:
         run();                        // run
         break;
   }
   updateT();                          // update T before returning
}

//-----------------------------------------------------------------------------
// stop() called from T0_ISR()
//-----------------------------------------------------------------------------

void stop (void)
{
      PCA0CPH0 = 0;                    // reset duty cycle
      PCA0CPH1 = 0;                    // reset duty cycle
      PCA0CPM0 = 0x00;                 // disable PWM
      PCA0CPM1 = 0x00;                 // disable PWM
      ET0 = 0;                         // disable T0 interrupt
      ET2 = 0;                         // disable T2 interrupt
      TR0 = 0;                         // stop Timer 0      
		TMR2CN &= ~0x04;                 // stop Timer 2 

}

//-----------------------------------------------------------------------------
// align() called from T0_ISR()
//-----------------------------------------------------------------------------

void align(void)
{
   static unsigned char v=0;
   static unsigned char d=0;
   Tslot = 1;                          // don't commutate
   NextT = TENMS;                      // execute align every 10ms
   if(v < (128 + VLIMIT))              // ramp voltage to 50% + VLIMIT
   {
      v++;                             // increment v
      PCA0CPH0 = v;                    // update PWM
      PCA0CPH1 = v;                    // update PWM
   }
   else if (d < TALIGN)                // align delay
   {
      d++;                             // increment d
   }
   else
   {
      Status = START;                  // go to next state
      AccIndex = 0;                    // reset table index
      Tslot = 0;                       // commutate on next interrupt
      AlignLED = OFF;                  // turn off LED
      v = 0;                           // clear v for next restart
      d = 0;                           // clear d for next restart
   }

}
//-----------------------------------------------------------------------------
// start() called from T0_ISR()
//-----------------------------------------------------------------------------
void  start(void)
{
      switch(Tslot)                    // implement time slot manager
      {
         case 0:
            start0();                  // do start0() one of eight times
            break;
         case 7:
            start7();                  // check if done
            break;
      }
      Tslot++;                         // increment time slot
      Tslot &= 0x07;                   // mask to 3 bits (modulus 8)
}
//-----------------------------------------------------------------------------
// start0()
//-----------------------------------------------------------------------------
void start0(void)
{
   unsigned int t,v;
   unsigned char d;

   t = TRamp[AccIndex];                // look-up table value
   v = VSTART / t;                     // divide to get v
   t *= TSCALE;                        // scale to get t
   v += 128;                           // add 50%
   v += VLIMIT;                        // add VLIMIT
   if(v > 255)                         // limit to unsigned char
   {
      v = 255;
   }
   d = v;                              // copy to unsigned char d
   PCA0CPH0 = d;                       // update  PWM
   PCA0CPH1 = d;                       // update PWM
   AccIndex++;                         // increment index
   NextT = t;                          // update next T
}

//-----------------------------------------------------------------------------
// start7()
//-----------------------------------------------------------------------------
void start7(void)
{
   if (AccIndex > 191)                 // check if done accelerating
   {
      Status = RUN;                    // go to next state
      NextT = 16 * TSCALE;             // update next T
      AMX0P = 0x09;                    // config for motor current
      AMX0N = 0xFF;                    // single ended
      ADC0CN = 0x81;                   // initiate on T0 overflow
      TMR2CN|= 0x04;                   // enabe Timer 2 
      ET2 = 1;                         // Enable T2 interrupt
   }
}
//-----------------------------------------------------------------------------
// run()
//-----------------------------------------------------------------------------

void run(void)
{
   Tslot &= 0x03;                      // mask Tslot to 2 bits (modulo 4)
   switch(Tslot)                       // implement time slot manager
   {
      case 0:
         run0();                       // run time slot 0
         break;
      case 1:
         run1();                       // run time slot 1
         break;
      case 2:
         run2();                       // run time slot 2
         break;
      case 3:
         run3();                       // run time slot 3
         break;
   }
   Tslot++;                            // increment time slot
   Tslot &= 0x03;                      // mask Tslot to 2 bits (modulo 4)
}
//-----------------------------------------------------------------------------
// run0()
//-----------------------------------------------------------------------------
void run0(void)
{
   Imotor = avgCurrent();              // avg motor current using ADC
   if (Imotor > ILIMIT)                //
   {
      OverCurrent = 1;                 //
      AlignLED = ON;                   //
      Status = STOP;
   }
  AMX0P = 0x0E;                        // config for pot
  AMX0N = 0xFF;                        // single ended
  ADC0CN = 0xC0;                       // initiate on AD0BUSY, LPT
  ADC0CF |= 0x04;                      // left justify
  AD0INT = 0;                          // clear ADC0 end-of-conversion
  AD0BUSY = 1;                         // initiate conversion
  while (!AD0INT);                     // wait for conversion to complete
  Vpot = ADC0H;                        // read speed pot
  AMX0P = openPhase[MtrIndex];         // config for open phase
  AMX0N = 0x0D;                        // differential measurement
  ADC0CF &= ~0x04;                     // right justify
  ADC0CN = 0x81;                       // initiate on T0 overflow
}
//-----------------------------------------------------------------------------
// run1()
//-----------------------------------------------------------------------------
void run1(void)
{
   signed int v;
   v = avgVoltage();                   // read back EMF
   v = flip(v);                        // flip every other cycle
   Vmin = v;                           // save min voltage
   ADC0CN = 0x81;                      // initiate on T0 overflow
}
//-----------------------------------------------------------------------------
// run2()
//-----------------------------------------------------------------------------
void run2(void)
{
   signed int v;
   Vmid[MtrIndex] = avgVoltage();      // read avg voltage and store in array
   v = avgVmid() - Vmid[MtrIndex];     // subtract from avg midpoint
   v = flip(v);                        // flip every other cycle
   Verror = v;                         // store in global Verror
   ADC0CN = 0x81;                      // initiate on T0 overflow
}

//-----------------------------------------------------------------------------
// run3()
//-----------------------------------------------------------------------------
void run3(void)
{
   static unsigned char stable = 12;
   signed int v;
   v = avgVoltage();                   // read avg voltage
   v = flip(v);                        // flip every other cycle
   Vmax = v;                           // store max voltage
   Vdelta[MtrIndex]= Vmax - Vmin;      // calculate delta
   if(stable==0)                       // stabilization delay
   {
      Vemf = avgVdelta();              // calculate back EMF magnitude
      if(Vemf<VSTALL)                  // check for stall
      {
         Stall = 1;                    // set stall flag
         AlignLED = ON;                // blink LED
      }
      else
      {
         Stall = 0;                    // clear stall flag
         AlignLED = OFF;               // turn off LED
      }

   }
   else
   {
      stable--;                        // decrement stable delay
   }
   AMX0P = 0x09;                       // config for motor current
   AMX0N = 0xFF;                       // single ended
   ADC0CN = 0x81;                      // initiate on T0 overflow
}
//-----------------------------------------------------------------------------
// run4()
//-----------------------------------------------------------------------------
unsigned int avgCurrent(void)
{
   unsigned int sum, result;
   udblbyte v;

   unsigned char i ;

   sum = 0;
   while (!AD0INT);                    // wait for conversion to complete
   v.b.lo = ADC0L;                     // read ADC
   v.b.hi = ADC0H;
   sum += v.w;
   ADC0CN = 0x80;                      // initiate on AD0BUSY
   for (i = 7; i != 0; i--)            // repeat 7 more times
   {
      AD0INT = 0;                      // clear ADC0 end-of-conversion
      AD0BUSY = 1;                     // initiate conversion
      while (!AD0INT);                 // wait for conversion to complete
      v.b.lo = ADC0L;                  // read ADC
      v.b.hi = ADC0H;
      sum += v.w;                      // add to sum
   }
   result = sum>>2;
   return result;                      // return average reading
}

//-----------------------------------------------------------------------------
// flip()
//-----------------------------------------------------------------------------
signed int flip (signed int v)
{
   if ((MtrIndex & 0x01) == 0x00)      // flip on 0, 2, and 4
   {
      v = -v;
   }
   return v;
}
//-----------------------------------------------------------------------------
// avgVoltage()
//-----------------------------------------------------------------------------

signed int avgVoltage(void)
{
   signed int sum, result;
   sdblbyte v;

   unsigned char i ;

   sum = 0;
   while (!AD0INT);                    // wait for conversion to complete
   v.b.lo = ADC0L;                     // read ADC
   v.b.hi = ADC0H;
   sum += v.w;                         // add to sum
   ADC0CN = 0x80;                      // initiate on AD0BUSY
   for (i = 7; i != 0; i--)            // repeat 7 more times
   {
      AD0INT = 0;                      // clear ADC0 end-of-conversion
      AD0BUSY = 1;                     // initiate conversion
      while (!AD0INT);                 // wait for conversion to complete
      v.b.lo = ADC0L;                  // read ADC
      v.b.hi = ADC0H;
      sum += v.w;
   }
   result = sum>>2;                    // divide by 4,  11-bit effective
   return result;                      // return average reading
}
//-----------------------------------------------------------------------------
// avgVmid()
//-----------------------------------------------------------------------------
signed int avgVmid(void)
{
   signed int sum;
   unsigned char i;
   sum = 0;
   for (i = 0; i < 6; i++)             // repeat 6  times
   {
         sum += Vmid[i];               // calculate sum of midpoints
   }
   sum /=6;                            // divide by 6
   return sum;                         // return average
}
//-----------------------------------------------------------------------------
// avgVdelta()
//-----------------------------------------------------------------------------

 signed int avgVdelta(void)
 {
    signed int sum;
    unsigned char i;
    sum = 0;
    for (i = 0; i < 6; i++)            // repeat 6  times
    {
       sum += Vdelta[i];               // calculate sum of Vdelta
    }
    sum /=6;                           // divide by 6
    return sum;
}

//-----------------------------------------------------------------------------
// updateT()
//-----------------------------------------------------------------------------
void updateT (void)
{
   udblbyte t0;
   t0.b.lo = TL0;                      // get current value
   t0.b.hi = TH0;
   t0.w -= NextT;                      // subtract period
   TL0 = t0.b.lo;                      // save new overflow value
   TH0 = t0.b.hi;
}
//-----------------------------------------------------------------------------
// commutate()
//-----------------------------------------------------------------------------
void commutate (void)
{
   MtrIndex++;                         // increment MtrIndex
   if(MtrIndex>5)                      // fix if greater than 5
   {
      MtrIndex = 0;
   }
   P1       = 0xff;                    // P1 all high
   PCA0CPM0 = 0x00;                    // disable PWM
   PCA0CPM1 = 0x00;                    // disable PWM
   XBR1 &= ~0x40;                      // disable crossbar
   P1SKIP = skipPattern[MtrIndex];
   XBR1 |= 0x40;                       // enable crossbar
   PCA0CPM0 = 0x42;                    // enable 8-bit PWM mode
   PCA0CPM1 = 0x42;                    // enable 8-bit PWM mode


   if(MtrIndex==0)                     // toggle LED on zero
   {
      BlinkLED = ON;
   }
   else
   {
      BlinkLED = OFF;
   }

}


