//-----------------------------------------------------------------------------
// Sensorless BLDC Motor Reference Design
//-----------------------------------------------------------------------------
// Copyright 2006 Silicon Laboratories Inc.
//
// AUTH: KAB
// DATE: 14 SEP 2006
//
// This program provides Sensorless BLDC motor control using the
// 'F310. This software is written specifically for the SBLDC
// reference design hardware. Please see the full application
// note for further information.
//
// Target: C8051F30x
//
// Tool chain: KEIL 'c' full-version required
//
// Rev History:
// 
// 30 AUG 2006 - Initial release (Pitmann  N2311A011 Motor)
//
// 14 SEP 2006 - Changed Motor Supplier (Anaheim Automation BLY171S-24V-4000)
//               Added switch for different motor types.
//               Modified Motor Parameters in this header file only.
//               Other source files unchanged.
//
//-----------------------------------------------------------------------------
// MACROS
//-----------------------------------------------------------------------------
// User modified Motor Parameters
//
// These parameters are related to thespecific motor used for the reference
// design. These parameters will need to be changed if using a different motor.
//
// define motor type - Pittman, Anaheim Automation, or custom motor 
//
//

#define ANAHEIM_BLY171S_24V_4000
//#define PITTMAN_N2311A011

// Motor parameters for Aneheim Automation BLYS171S_24V_4000 Motor
#ifdef ANAHEIM_BLY171S_24V_4000
#define MILLIOHMS 1800                 // motor resistance in milliohms
#define VMOTOR 12                      // motor voltage in volts
#define IMOTOR 500                     // maximum motor current in millamps
#define NUM_POLES 8                    // motor poles
#define RATED_RPM 4000                 // rated no load rpm at max voltage
#define MAX_RPM 2000                   // max desired speed
#define KPINIT 20                      // Kp initial value
#endif

// Motor parameters for Pittman Motor N2311A011 Motor
#ifdef PITTMAN_N2311A011 
#define MILLIOHMS 260                  // motor resistance in milliohms
#define VMOTOR 12                      // motor voltage in volts
#define IMOTOR 5                       // maximum motor current in amps
#define NUM_POLES 4                    // motor poles
#define RATED_RPM 7500                 // rated no load rpm at max voltage
#define MAX_RPM 5000                   // max desired speed
#define KPINIT 40                      // Kp initial value
#endif

//-----------------------------------------------------------------------------
// constants
#define SYSCLK 24500000L               // bus clock (long int)
#define ILIMIT 1000                    // current limit
#define VSTALL 2                       // minimum stall criteria
#define TALIGN 255                     // align time in 10ms increments
#define ON 0                           // push-button logic ON
#define OFF 1                          // push-button logic OFF
#define STOP 0                         // Motor State variable values
#define ALIGN 1          
#define START 2
#define RUN 3
//-----------------------------------------------------------------------------
// preprocessor calculated values
#define TABLE_MIN 8
#define TABLE_MAX 240
#define RAMP_MOD 8
#define RUN_MOD 4
#define T0_DIV 48
#define VLIMIT  ((128L * MILLIOHMS * IMOTOR )/VMOTOR /1000000)
#define T0_CLK (SYSCLK/T0_DIV)
#define TSCALE ((2*10*60/6)*T0_CLK/RATED_RPM/NUM_POLES/RAMP_MOD/TABLE_MIN)
#define VMIN (128/10)                  // (128 /10)
#define VSTART (128 * TABLE_MIN / 10)
#define TENMS (SYSCLK / T0_DIV / 100)      // timer count for 10ms delay
#define TRUN_TIMES_127 (((127L*2*60/6)*T0_CLK)/RATED_RPM/NUM_POLES/RUN_MOD)
#define RPM_SCALE (RATED_RPM/127)
#define T2_RELOAD (65536-SYSCLK/1000)
#define T2_RELOAD_L (T2_RELOAD & 0x00FF)
#define T2_RELOAD_H (T2_RELOAD/256)
#define OMEGA_LIMIT (127*MAX_RPM/RATED_RPM)

//-----------------------------------------------------------------------------
// preprocessor error checking
// These macros check for range errors on user modifiable parameters
#if ((TSCALE)>(255))
   #error "TSCALE requires 16 bit math"
#endif
#if ((TSCALE*TABLE_MAX)>65535)
   #error "TSCALE too large"
#endif

//-----------------------------------------------------------------------------
// typedefs
//-----------------------------------------------------------------------------
typedef union                          // unsigned union for accessing SFRs
{
  struct
  {
      unsigned char hi;
      unsigned char lo;
  } b;
  unsigned int w;
}udblbyte;

typedef union                          // signed union used for signed ADC
{
  struct
  {
      unsigned char hi;
      unsigned char lo;
  } b;
  signed int w;
}sdblbyte;