//-----------------------------------------------------------------------------
// Copyright (c) 2012 by Silicon Laboratories
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Silicon Laboratories End User
// License Agreement which accompanies this distribution, and is available at
// http://developer.silabs.com/legal/version/v10/License_Agreement_v10.htm
//
//
// Original content and implementation provided by Silicon Laboratories
//-----------------------------------------------------------------------------
//
// Program Description:
//-----------------------------------------------------------------------------
//
// This example code runs in Power Mode 8 and reproduces the power numbers
// provided in the SiM3L1xx data sheet.
//
// To reproduce a current measurement from the data sheet, uncomment the
// corresponding settings below, recompile, download, and run on an SiM3L1xx
// MCU card.
//
// Note: While the debug adatper is physically plugged into the board, there
// will be approximately 60 uA of leakage current. Remove the debug adapter
// from the 10-pin connector before taking current measurements.
//
// Note: Running the AHB/APB at 16 kHz will cause issues with debugging and
// downloading code.  Press and hold PB1.4 (switch A) after a reset to
// re-download code to the device.
//
// Resources Used:
//-----------------------------------------------------------------------------
// LDO0 for active voltage scaling
// DCDC0
// PLL0 for AHB/APB frequencies above 20 MHz
// LPOSC0 for AHB/APB frequencies 20 MHz and below
// LFOSC0 for AHB/APB 16 kHz frequency
//
// PB0.1 AHB/16 out, if enabled
// PB1.4 switch A - hold core out of reset
// PB0.2 tied to PB1.5 switch B - port match wake from PM8
// PB1.5 LED - turns on after a port match not from PM8
// PB1.7 LED - turns on after PM8 exits
//
// Notes On Example and Modes:
//-----------------------------------------------------------------------------
// Clock frequencies are dependent upon the mode selected below.
//
// NOTE: Remove all three jumpers on J7 and measure across the Imeasure jumper
// (JP2).
//
// How to Use:
//-----------------------------------------------------------------------------
// Using a fixed VBAT:
//
// 1) Connect a USB Debug Adapter to the 10-pin CoreSight connector (J14) on
//    the SiM3L1xx MCU card.
// 2) Remove the three shorting blocks from J7.
// 3) Remove the JP2 Imeasure shorting block and put a multimeter across
//    (positive side on the top pin).
// 4) Move the VBAT Sel switch (SW2) to the middle +3.3V_VREG position.
// 5) Move the VIO Sel (SW8) and VIORF Sel (SW9) switches to the bottom VBAT
//    position.
// 6) Connect the 9 V power adapter to POWER (J6).
// 7) Select the desired data sheet settings from the options in this file.
// 8) Compile and download the code to the board.
// 9) Remove the debug adapter connection.
// 10) Connect PB0.2 to PB1.5 using a short wire and the J21 and J22 headers.
// 11) Power down the board.
// 12) Power up the board.  This will cause a power-on reset that will clear
//     the settings of some modules (i.e. PMU).
// 13) Press PB1.7 to place the device in PM8.  The PB1.5 LED will turn off.
// 14) Measure the power of the device.
// 15) Press PB1.5 to wake the device from PM8. When this happens, the PB1.7
//     LED will turn on and the core will sit in an infinite while(1) loop.
//
// Using a varying VBAT:
//
// 1) Connect a USB Debug Adapter to the 10-pin CoreSight connector (J14) on
//    the SiM3L1xx MCU card.
// 2) Remove the three shorting blocks from J7.
// 3) Remove the JP2 Imeasure shorting block.
// 4) Move the VBAT Sel switch (SW2) to the bottom BAT position.
// 5) Move the VIO Sel (SW8) and VIORF Sel (SW9) switches to the bottom VBAT
//    position.
// 6) (Optional) Place a battery in the BT1 holder.
// 7) (Optional) Instead of a battery, connect the positive terminal of a bench
//    power supply to the positive terminal of the multimeter, the negative
//    terminal of the power supply to the MCU card ground, and the negative
//    terminal of the multimeter to the top pin of the Imeasure jumper (JP2).
//    This configuration will prevent any circuitry on the board from
//    interfering with the power measurement.
// 8) Select the desired data sheet settings from the options in this file.
// 9) Compile and download the code to the board.
// 10) Remove the debug adapter connection.
// 11) Connect PB0.2 to PB1.5 using a short wire and the J21 and J22 headers.
// 12) Power down the board.
// 13) Power up the board.  This will cause a power-on reset that will clear
//     the settings of some modules (i.e. PMU).
// 14) Press PB1.7 to place the device in PM8.  The PB1.5 LED will turn off.
// 15) Measure the power of the device.
// 16) Press PB1.5 to wake the device from PM8. When this happens, the PB1.7
//     LED will turn on and the core will sit in an infinite while(1) loop.
//
// Revision History:
//-----------------------------------------------------------------------------
//
// Release 1.0
//    -Initial Revision (BD and TP)
//    -6 AUG 12
//
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

// si32 HAL
#include <si32_device.h>
#include <SI32_PBSTD_A_Type.h>
#include <SI32_RSTSRC_A_Type.h>
#include <SI32_CLKCTRL_A_Type.h>
#include <SI32_EXTOSC_A_Type.h>
#include <SI32_FLASHCTRL_A_Type.h>
#include <SI32_RTC_B_Type.h>
#include <SI32_PMU_A_Type.h>
#include <SI32_LDO_A_Type.h>
#include <SI32_PLL_A_Type.h>
#include <SI32_DCDC_A_Type.h>
#include <SI32_LCD_A_Type.h>
#include <SI32_PBCFG_A_Type.h>
#include <SI32_WDTIMER_A_Type.h>
#include <SI32_VMON_B_Type.h>

//-----------------------------------------------------------------------------
// Power Measurement Configurations
//-----------------------------------------------------------------------------
// Uncomment the defines for the desired power measurement or use the defines
// to measure a desired configuration.

//=============================================================================
// Data Sheet Specifications
//=============================================================================

// Power Mode 8 (PM8)
// Low Power Sleep, powered through VBAT, VIO, and VIORF at 2.4 V (external
// supply needed), 32 kB of retention RAM
// RTC Disabled, TA = 25 deg C

#define LPOSC20
#define RTC_DISABLED
#define AVS_ON

// Power Mode 8 (PM8)
// Low Power Sleep, powered through VBAT, VIO, and VIORF at 2.4 V (external
// supply needed), 32 kB of retention RAM
// RTC w/ 16.4 kHz LFO, TA = 25 deg C
/*
#define LPOSC20
#define RTC_USING_LFOSC
#define AVS_ON
*/
// Power Mode 8 (PM8)
// Low Power Sleep, powered through VBAT, VIO, and VIORF at 2.4 V (external
// supply needed), 32 kB of retention RAM
// RTC with 32.768 kHz Crystal, TA = 25 deg C
/*
#define LPOSC20
#define RTC_USING_CRYSTAL
#define AVS_ON
*/
// Power Mode 8 (PM8)
// Low Power Sleep, powered by the low power mode charge pump, 32 kB of
// retention RAM
// RTC w/ 16.4 kHz LFO, TA = 25 deg C, VBAT = 2.4 V or 3.6 V (external supply
// needed)
/*
#define LPOSC20
#define RTC_USING_LFOSC
#define CHARGE_PUMP
#define AVS_ON
*/
// Power Mode 8 (PM8)
// Low Power Sleep, powered by the low power mode charge pump, 32 kB of
// retention RAM
// RTC with 32.768 kHz Crystal, TA = 25 deg C, VBAT = 2.4 V or 3.6 V (external
// supply needed)
/*
#define LPOSC20
#define RTC_USING_CRYSTAL
#define CHARGE_PUMP
#define AVS_ON
*/

//=============================================================================
// Configurable Defines
//=============================================================================

// AHB/APB Clock Settings
//#define PLL49
//#define LPOSC20
//#define LPOSC5
//#define LPOSC2p5
//#define LPOSC1p25
//#define LPOSC0p625
//#define LFOSC16k
//#define EXTOSC0to5

//#define APBDIV2

//#define PERIPH_CLOCKS_OFF

// LDO0 Adaptive Voltage Scaling (AVS)
//#define AVS_ON

// DCDC0 Settings
// DCDC_ON must be defined in conjunction with a range
/*
#define DCDC_ON
#define DCDC_gt15mA
#define DCDC_5to15mA
#define DCDC_lt5mA
*/

// RTC Settings
//#define RTC_DISABLED
//#define RTC_USING_LFOSC
//#define RTC_USING_CRYSTAL
//#define ENABLE_RTC_INTERRUPTS_AND_WAKE

// Low Power Mode Charge Pump Settings
//#define CHARGE_PUMP

// AHB Output Setting for observation
// When enabled, AHB/16 is output on PB0.1
//#define AHB_OUT_ENABLE

//-----------------------------------------------------------------------------
// Global Defines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------

volatile uint32_t msTicks;

//-----------------------------------------------------------------------------
// Function Prototypes
//-----------------------------------------------------------------------------

//{{Config
void Clocks_Configure(void);
void PLL_lock_freq_to_hz(uint32_t freq);
void DCDC0_Configure(void);
void RTC0_Configure(void);
void PB_Configure(void);
void LDO0_Configure(void);
//}}Config

//{{App
void PowerMode_8(void);
//}}App

//{{Loop
//}}Loop

//-----------------------------------------------------------------------------
// main()
//-----------------------------------------------------------------------------
int main()
{
   // Initialization code
   //{{Config
   while (!SI32_PBSTD_A_read_pin(SI32_PBSTD_1, 4));

   Clocks_Configure();
   PB_Configure();

   // Press PB1.7 to enter PM8
   // Turn off the red PB1.5 LED.
   while (SI32_PBSTD_A_read_pin(SI32_PBSTD_1, 7));


   RTC0_Configure();

   #ifdef AVS_ON
      LDO0_Configure();
   #endif

   #ifdef DCDC_ON
      DCDC0_Configure();
   #endif
   //}}Config

   // Application code
   //{{App
   PowerMode_8();
   //}}App

   while(1)
   {
      // Loop forever
      //{{Loop
      //}}Loop
   }

}

//-----------------------------------------------------------------------------
// Initialization Code
//-----------------------------------------------------------------------------

//{{Configure

//-----------------------------------------------------------------------------
// This function is invoked by the CMSIS-required SystemInit() function in
// system_<device>.c.  SystemInit() is invoked by Reset_Handler() when the
// CPU boots.
void mySystemInit()
{
   // Disable the watchdog timer to prevent device resets
   // Any lines added to this function should be added below this line of code
   SI32_WDTIMER_A_stop_counter(SI32_WDTIMER_0);
}

void Clocks_Configure(void)
{
   SI32_CLKCTRL_A_enable_apb_to_all_modules(SI32_CLKCTRL_0);

   #ifdef APBDIV2
      SI32_CLKCTRL_A_select_apb_divider_2(SI32_CLKCTRL_0);
   #endif

   #ifdef PLL49
      SI32_PLL_A_select_dco_free_running_mode(SI32_PLL_0);
      SI32_FLASHCTRL_A_enter_read_store_mode(SI32_FLASHCTRL_0);
      SI32_FLASHCTRL_A_select_flash_speed_mode(SI32_FLASHCTRL_0, 2);
      SI32_FLASHCTRL_A_select_flash_read_time_fast(SI32_FLASHCTRL_0);
      SI32_CLKCTRL_A_select_ahb_source_pll(SI32_CLKCTRL_0);

      SystemCoreClock = 49000000;
   #endif

   #ifdef LPOSC20
      SI32_CLKCTRL_A_select_ahb_source_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_FLASHCTRL_A_select_flash_read_time_fast(SI32_FLASHCTRL_0);

      SystemCoreClock = 20000000;
   #endif

   #ifdef LPOSC5
      SI32_CLKCTRL_A_select_ahb_source_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_divider(SI32_CLKCTRL_0, 2);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);

      SystemCoreClock = 5000000;
   #endif

   #ifdef LPOSC2p5
      SI32_CLKCTRL_A_select_ahb_source_divided_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);

      SystemCoreClock = 2500000;
   #endif

   #ifdef LPOSC1p25
      SI32_CLKCTRL_A_select_ahb_source_divided_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_divider(SI32_CLKCTRL_0, 1);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);

      SystemCoreClock = 1250000;
   #endif

   #ifdef LPOSC0p625
      SI32_CLKCTRL_A_select_ahb_source_divided_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_divider(SI32_CLKCTRL_0, 2);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);

      SystemCoreClock = 625000;
   #endif

   #ifdef LFOSC16k
      SI32_CLKCTRL_A_select_ahb_source_low_frequency_oscillator(SI32_CLKCTRL_0);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);

      SystemCoreClock = 16400;
   #endif

   #ifdef EXTOSC0to5
      SI32_PBCFG_A_enable_crossbar_0(SI32_PBCFG_0);
      SI32_EXTOSC_A_select_oscillator_mode_cmos(SI32_EXTOSC_0);
      SI32_CLKCTRL_A_enable_extosc_in(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_source_external_oscillator(SI32_CLKCTRL_0);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);

      SystemCoreClock = 5000000;
   #endif

   // Update SysTick
   SysTick_Config(SystemCoreClock / 1000);

   // Update ITM Divider
   *((uint32_t *) 0xE0040010) = ((50 * SystemCoreClock) / 20000000) - 1;
}

void LDO0_Configure(void)
{
   // From the Reference Manual, LDO chapter:
   //
   // The LDO outputs are adjustable, with a valid range of 0.8 to 1.9 V.
   // Additional power savings for the device may be realized by lowering the
   // LDO output voltage of the memory and digital regulators under appropriate
   // conditions, such as lower clock speeds. The regulators are adjusted using
   // the ALDOOVAL, DLDOOVAL and MLDOOVAL bit fields in the CONTROL register.
   // The adjustment fields are linear, with each LSB representing
   // approximately 50 mV. To adjust the memory LDO to 1.6 V, for example,
   // the MLDOOVAL field can be set to 0x10 (0.8 V + 50 mV x 16). The LDO
   // outputs are also available to the SARADC and Comparators for monitoring
   // purposes.
   //
   // The optimal LDO settings for different operational speeds are to be
   // determined (TBD), pending full characterization of silicon over process,
   // temperature, and voltage corners. Preliminary silicon characterization
   // suggests that the safe minimum voltage for the memory LDO at all speeds
   // is 1.6 V. The digital LDO may be safely set to 1.0 V at speeds of 20 MHz
   // or less, and 1.3 V between 20 and 50 MHz. The analog LDO should normally
   // be left at 1.8 V output during active operation. All LDOs may be adjusted
   // prior to entering PM8, to extend the external supply operating range. See
   // the electrical specification tables in the datasheet for specified output
   // settings.

   #ifdef PLL49
      // DLDO = 1.3 V, MLDO = 1.6 V, ALDO = 1.8 V, high bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x002A3034);
   #endif

   #ifdef LPOSC20
      // DLDO = 1.3 V, MLDO = 1.6 V, ALDO = 1.8 V, high bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x002A3034);
   #endif

   #ifdef EXTOSC0to5
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC5
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC2p5
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC1p25
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC0p625
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LFOSC16k
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif
}

void DCDC0_Configure(void)
{
   // Enable the clock to the DCDC0 module
   SI32_CLKCTRL_A_enable_apb_to_modules_0(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG0_DCDC0CEN_ENABLED_U32
      | SI32_CLKCTRL_A_APBCLKG0_LCD0CEN_ENABLED_U32);
   SI32_CLKCTRL_A_enable_apb_to_modules_1(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG1_MISC0CEN_ENABLED_U32);

   // Configure the DCDC0 module
   SI32_LCD_A_enable_dcdc_bias(SI32_LCD_0);
   SI32_DCDC_A_enable_module(SI32_DCDC_0);

   #ifdef DCDC_gt15mA
      SI32_DCDC_A_select_power_switch_size(SI32_DCDC_0, 3);
      SI32_DCDC_A_enable_sync_mode(SI32_DCDC_0);
      SI32_DCDC_A_select_minimum_pulse_width(SI32_DCDC_0, 0);
   #endif

   #ifdef DCDC_5to15mA
      SI32_DCDC_A_select_power_switch_size(SI32_DCDC_0, 0);
      SI32_DCDC_A_enable_sync_mode(SI32_DCDC_0);
      SI32_DCDC_A_select_minimum_pulse_width(SI32_DCDC_0, 0);
   #endif

   #ifdef DCDC_lt5mA
      SI32_DCDC_A_select_power_switch_size(SI32_DCDC_0, 0);
      SI32_DCDC_A_enable_async_mode(SI32_DCDC_0);
      SI32_DCDC_A_select_minimum_pulse_width(SI32_DCDC_0, 3);
   #endif

   SI32_DCDC_A_select_output_voltage(SI32_DCDC_0, 0x0C);
   SI32_DCDC_A_select_clock_source_local(SI32_DCDC_0);

   // Wait until output voltage reached programmed value
   while (!SI32_DCDC_A_is_output_low_flag_set(SI32_DCDC_0)) {};

   // Check that the output voltage hasn't exceed the programmed value
   while (SI32_DCDC_A_is_output_high_flag_set(SI32_DCDC_0)) {};

   SI32_LDO_A_select_digital_source_dcdc_output(SI32_LDO_0);
   SI32_LDO_A_select_memory_source_dcdc_output(SI32_LDO_0);
   SI32_LDO_A_select_analog_source_dcdc_output(SI32_LDO_0);
}

void RTC0_Configure(void)
{
   SI32_CLKCTRL_A_enable_apb_to_modules_1(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG1_MISC0CEN_ENABLED_U32);

   #ifdef RTC_DISABLED
       SI32_RTC_B_set_clock_source_lfo(SI32_RTC_0);

       SI32_RTC_B_disable_crystal_oscillator(SI32_RTC_0);
       SI32_RTC_B_disable_bias_doubler(SI32_RTC_0);
       SI32_RTC_B_disable_auto_gain_control(SI32_RTC_0) ;

       SI32_RTC_B_disable_low_frequency_oscillator_output(SI32_RTC_0);
       SI32_RTC_B_disable_low_frequency_oscillator(SI32_RTC_0);

       SI32_RTC_B_disable_clock_out(SI32_RTC_0);

       SI32_RTC_B_disable_rtc_timer(SI32_RTC_0);
   #endif

   #ifdef RTC_USING_LFOSC
      SI32_RTC_B_enable_low_frequency_oscillator(SI32_RTC_0);
      SI32_RTC_B_set_clock_source_lfo(SI32_RTC_0);
      SI32_RTC_B_disable_crystal_oscillator(SI32_RTC_0);
      SI32_RTC_B_enable_rtc_timer(SI32_RTC_0);
      SI32_RTC_B_enable_clock_out(SI32_RTC_0);

      // Set ALARM0 with ~1 second
      SI32_RTC_B_write_alarm0(SI32_RTC_0, 0x4000);
      SI32_RTC_B_enable_alarm0_auto_reset(SI32_RTC_0);

      #ifdef ENABLE_RTC_INTERRUPTS_AND_WAKE
         NVIC_ClearPendingIRQ(RTC0ALRM_IRQn);
         NVIC_EnableIRQ(RTC0ALRM_IRQn);

         SI32_RTC_B_enable_alarm0_interrupt(SI32_RTC_0);
      #endif

      SI32_RTC_B_start_timer(SI32_RTC_0);
   #endif

   #ifdef RTC_USING_CRYSTAL
      // Crystal mode
      SI32_RTC_B_disable_auto_gain_control(SI32_RTC_0);
      SI32_RTC_B_enable_bias_doubler(SI32_RTC_0);
      SI32_RTC_B_enable_autostep(SI32_RTC_0);
      SI32_RTC_B_set_rtc_load_capacitance(SI32_RTC_0,7);
      SI32_RTC_B_set_clock_source_rtc(SI32_RTC_0);
      SI32_RTC_B_enable_crystal_oscillator(SI32_RTC_0);
      SI32_RTC_B_enable_rtc_timer(SI32_RTC_0);
      SI32_RTC_B_enable_clock_out(SI32_RTC_0);

      // Wait at least 20 ms
      uint32_t now = msTicks;
      while (msTicks <= now + 20);

      // Poll clock until stable
      while (!((SI32_RTC_B_is_external_oscillator_valid(SI32_RTC_0) == true)));

      // Poll load capacitance ready
      while (!((SI32_RTC_B_is_load_capacitance_ready(SI32_RTC_0) == true)));

      // (Recommended) Enable automatic gain control and disable bias doubling for maximum power savings
      // This is done after enabling the charge pump to ensure the CP stabilizes

      // (Recommended) Enable missing clock detector

      // Wait at least 2 ms
      now = msTicks;
      while (msTicks <= now + 2);

        // Set ALARM0 with 1 second
      SI32_RTC_B_write_alarm0(SI32_RTC_0, 0x8000);
      SI32_RTC_B_enable_alarm0_auto_reset(SI32_RTC_0);

      SI32_RTC_B_disable_low_frequency_oscillator_output(SI32_RTC_0);
      SI32_RTC_B_disable_low_frequency_oscillator(SI32_RTC_0);

      #ifdef ENABLE_RTC_INTERRUPTS_AND_WAKE
         NVIC_ClearPendingIRQ(RTC0ALRM_IRQn);
         NVIC_EnableIRQ(RTC0ALRM_IRQn);

         SI32_RTC_B_enable_alarm0_interrupt(SI32_RTC_0);
      #endif

      SI32_RTC_B_start_timer(SI32_RTC_0);
   #endif
}

void PB_Configure(void)
{
   SI32_CLKCTRL_A_enable_apb_to_modules_0(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG0_PB0CEN_ENABLED_U32);

   // Enable the LED drivers (PB1.4, PB1.5, PB1.6, PB1.7)
   SI32_PBSTD_A_set_pins_push_pull_output(SI32_PBSTD_1, 0x00F0);
   SI32_PBSTD_A_write_pins_high(SI32_PBSTD_1,0x00D0);

   // Turn on the PB1.5 LED
   SI32_PBSTD_A_write_pins_low(SI32_PBSTD_1,0x0020);

   // PB1.0 and PB1.1 are connected to the EBID processor and are being pulled
   // low when the MCU Card is disconnected from the UDP Motherboard. Setting
   // these pins to analog mode disconnects the pins from the weak pull-ups,
   // which draw current.
   SI32_PBSTD_A_set_pins_analog(SI32_PBSTD_1, 0x00000003);

   // Leave the LED pins in open-drain mode so the switches can pull the
   // pins low.

   #ifdef AHB_OUT_ENABLE
      SI32_PBSTD_A_set_pins_push_pull_output(SI32_PBSTD_0, 0x00000002);
      SI32_PBSTD_A_write_pbskipen(SI32_PBSTD_0, 0x00000001);
      SI32_PBCFG_A_enable_xbar0_peripherals(SI32_PBCFG_0,
         SI32_PBCFG_A_XBAR0_AHBEN);
   #endif

   SI32_PBCFG_A_enable_crossbar_0(SI32_PBCFG_0);
}

//}}Config

//-----------------------------------------------------------------------------
// Subroutines
//-----------------------------------------------------------------------------

//{{App

void PowerMode_8(void)
{
   // 1. Enable the clocks to peripherals that will be configured by firmware.
   SI32_CLKCTRL_A_enable_apb_to_all_modules(SI32_CLKCTRL_0);

   // 2. Enable PM8 mode by setting the PMSEL bit to 1 in the CLKCTRL CONFIG
   // register.

   SI32_CLKCTRL_A_enable_power_mode_8(SI32_CLKCTRL_0);

   // 3. Set up the desired PMU wakeup source.

   // Explicitly disable all wakeup sources and re-enable the ones needed
   SI32_PMU_A_write_wakeen(SI32_PMU_0, 0x0000);

   // Device will wake when PB0.2 goes low
   SI32_PMU_A_set_pin_wake_events(SI32_PMU_0, 0x04, 0x04);

   SI32_PMU_A_enable_pin_wake(SI32_PMU_0);
   SI32_PMU_A_enable_pin_wake_event(SI32_PMU_0);

   #ifdef ENABLE_RTC_INTERRUPTS_AND_WAKE
      SI32_PMU_A_enable_rtc0_alarm_wake_event(SI32_PMU_0);
      SI32_RSTSRC_A_disable_rtc0_reset_source(SI32_RSTSRC_0);
   #endif

   // 4. Switch the AHB clock to the 20 MHz Low Power Oscillator and adjust the
   // Flash read timing as necessary.

   SI32_CLKCTRL_A_select_ahb_source_low_power_oscillator(SI32_CLKCTRL_0);
   SI32_FLASHCTRL_A_select_flash_read_time_fast(SI32_FLASHCTRL_0);
   SI32_LDO_A_select_digital_bias_high(SI32_LDO_0);
   SI32_LDO_A_select_memory_bias_high(SI32_LDO_0);
   SI32_LDO_A_select_analog_bias_high(SI32_LDO_0);

   // 5. Enable the retention mode of RAM banks needed by the application (all
   // eight banks recommended for most applications).
   // Note: To properly execute code after exiting PM8, the stack pointer must
   // be located in a RAM bank with retention enabled.

   // Enable retention on all 32 kB of RAM
   SI32_PMU_A_set_ram_retention_enable_mask(SI32_PMU_0, 0xFF);

   // 6. Set the pins in the lowest power configuration for this mode.

   SI32_PBSTD_A_write_pins_high(SI32_PBSTD_1, 0x00F0);

   // 7. Set all LDOs to 1.5 V for VBAT between 1.8 to 2.9V and to 1.9 V for
   // VBAT between 2.0 to 3.8 V output.

   // Note: It is necessary to use LDO settings appropriate to the VBAT supply
   // range to ensure proper PM8 sleep functionality.

   // SI32_LDO_A_write_control(SI32_LDO_0, 0x002E2E2E); // 1.5 V
   SI32_LDO_A_write_control(SI32_LDO_0, 0x00363636); // 1.9 V

   // 8. (Optional) If VBAT is between 2.0 to 3.8 V, set the VBAT Monitor
   // threshold to the high setting.

   SI32_VMON_B_select_vbat_high_threshold(SI32_VMON_0);

   // 9. Set up the low power peripherals as desired (RTC0, ACCTR0, LCD0,
   // etc.).
   // (see RTC0_Configure())

   // 10. (Optional) If using the RTC in low-frequency oscillator mode, enable
   // the RTC0 module using the steps in the Reference Manual. If using the RTC
   // in crystal mode, the recommended initialization sequence is:
   // a. Disable RTC0 automatic gain control (AGC) and enable the bias doubler.
   // b. Set up the RTC0 in crystal mode using the list of steps in the
   // Reference Manual.
   // c. (Optional) Enable the charge pump (see Step 10).
   // d. Enable RTC0 automatic gain control and disable the bias doubler.
   // (see RTC0_Configure() and code below)

   #ifdef CHARGE_PUMP
      SI32_CLKCTRL_A_enable_apb_to_modules_1(SI32_CLKCTRL_0,
         SI32_CLKCTRL_A_APBCLKG1_MISC0CEN_ENABLED_U32);

      // 11. (Optional) Configure and enable the charge pump:
      // a. Enable the low power charge pump monitor in the PMU.

      SI32_PMU_A_enable_low_power_charge_pump_monitor(SI32_PMU_0);

      // b. Enable the RTC0 clock to other modules (CLKOEN = 1 in the RTC0
      // module).

      SI32_RTC_B_enable_clock_out(SI32_RTC_0);

      // c. Enable the low power charge pump monitor as a reset source.

      SI32_PMU_A_enable_low_power_charge_pump_monitor_interrupt(SI32_PMU_0);
      SI32_RSTSRC_A_enable_low_power_mode_charge_pump_reset_source(SI32_RSTSRC_0);

      // d. Set the low power charge pump load to the appropriate value for the
      // application. See AN725 for more information.

      SI32_PMU_A_set_low_power_charge_pump_load(SI32_PMU_0, 0);

      // e. Enable the low power charge pump and enable it as a PMU wake up source.

      SI32_PMU_A_enable_low_power_charge_pump(SI32_PMU_0);
      SI32_PMU_A_enable_low_power_charge_pump_wake_event(SI32_PMU_0);

      // f. Enable the low power charge pump interrupt.

      NVIC_ClearPendingIRQ(CPFAIL_IRQn);
      NVIC_EnableIRQ(CPFAIL_IRQn);
   #endif

   #ifdef RTC_USING_CRYSTAL
      // 10d. Enable RTC0 automatic gain control and disable the bias doubler.
      // (Recommended) Enable automatic gain control and disable bias doubling
      // for maximum power savings
      SI32_RTC_B_enable_auto_gain_control(SI32_RTC_0);
      SI32_RTC_B_disable_bias_doubler(SI32_RTC_0);
   #endif

   // 12. Disable all undesired reset sources.
   // Note: The Watchdog Timer (WDTIMER0) and PVTOSC0 modules will behave as
   // if a power-on reset occurred after exiting PM8, so disabling the Watchdog
   // Timer as a reset source will prevent unwanted resets.

   // The LDOs are disabled in PM8, so the LDO bias doesn't affect power
   // consumption

   // 13. Disable the SysTick timer.

   SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;

   // 14. Disable all unused peripherals.

   // 15. (Optional) Disable the clocks to all unused peripherals. Entering PM8
   // stops all clocks, so this step is not necessary for this mode.

   // 16. Clear the PMU wakeup flags.

   SI32_PMU_A_clear_wakeup_flags(SI32_PMU_0);

   // 17. Enable the interrupts for the PMU wakeup source (enabled in Step 3).

   NVIC_ClearPendingIRQ(PMATCH_IRQn);
   NVIC_EnableIRQ(PMATCH_IRQn);

   // 18. Set the SLEEPDEEP bit in the core.

   SCB->SCR = SCB_SCR_SLEEPDEEP_Msk; // set SLEEPDEEP

   // 19. Execute the DSB (Data Synchronization Barrier), ISB (Instruction
   // Synchronization Barrier), and WFI (Wait for Interrupt) or WFE (Wait for
   // Event) instructions. For SiM3L1xx devices, WFI and WFE have the same
   // behavior

   __DSB();
   __ISB();
   __WFI();

   // Wake from PM8

   // The watchdog timer is re-enabled after a PM8, so disable it again after
   // exiting PM8
   SI32_WDTIMER_A_stop_counter(SI32_WDTIMER_0);

   SI32_PMU_A_clear_wakeup_flags(SI32_PMU_0);

   #ifdef AVS_ON
      LDO0_Configure();
   #endif

   SI32_PBSTD_A_write_pins_low(SI32_PBSTD_1, 0x0010);
}

//}}App

//-----------------------------------------------------------------------------
// Interrupt Service Routines (ISRs)
//-----------------------------------------------------------------------------

// SysTick is used to enable the RTC crystal, but is otherwise disabled in
// this example.
void SysTick_Handler(void)
{
   msTicks++;
}

//{{ISR

void RTC0ALRM_IRQHandler(void)
{
   SI32_RTC_B_clear_alarm0_interrupt(SI32_RTC_0);

   if(SI32_PMU_A_is_power_mode_8_exit_flag_set(SI32_PMU_0))
   {
      if(SI32_PMU_A_is_rtc0_alarm_wakeup_flag_set(SI32_PMU_0))
      {
            // Do wakeup activities
      }
   }

//   SI32_PBSTD_A_write_pins_low(SI32_PBSTD_1, 0x0020);
}

void VCPFAIL_IRQHandler(void)
{
   if(SI32_PMU_A_is_power_mode_8_exit_flag_set(SI32_PMU_0))
   {
      if(SI32_PMU_A_is_low_power_charge_pump_wakeup_flag_set(SI32_PMU_0))
      {
            // Do wakeup activities
      }
   }

   NVIC_ClearPendingIRQ(CPFAIL_IRQn);
   NVIC_DisableIRQ(CPFAIL_IRQn);

//   SI32_PBSTD_A_write_pins_low(SI32_PBSTD_1, 0x0020);
}

void PMATCH_IRQHandler(void)
{
   // The watchdog timer is re-enabled after a PM8, so disable it again after
   // exiting PM8
   SI32_WDTIMER_A_stop_counter(SI32_WDTIMER_0);

   if(SI32_PMU_A_is_power_mode_8_exit_flag_set(SI32_PMU_0))
   {
      if(SI32_PMU_A_is_pin_wakeup_event_flag_set(SI32_PMU_0))
      {
         switch(SI32_PMU_A_get_last_wakeup_source(SI32_PMU_0))
         {
            case SI32_PIN_WAKEUP:
                  SI32_PBSTD_A_write_pins_low(SI32_PBSTD_1, 0x0080);
               break;
            default:
               break;
         }
      }
   }

   NVIC_DisableIRQ(PMATCH_IRQn);
   NVIC_ClearPendingIRQ(PMATCH_IRQn);
}

//}}ISR

//-eof-------------------------------------------------------------------------
