//-----------------------------------------------------------------------------
// Copyright (c) 2012 by Silicon Laboratories
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Silicon Laboratories End User
// License Agreement which accompanies this distribution, and is available at
// http://developer.silabs.com/legal/version/v10/License_Agreement_v10.htm
//
//
// Original content and implementation provided by Silicon Laboratories
//-----------------------------------------------------------------------------
//
// Program Description:
//-----------------------------------------------------------------------------
//
// This example code runs in Normal Mode (Power Mode 0) or Power Mode 4 and
// runs a string of NOPs to provide the baseline power consumption numbers at
// the settings provided in the SiM3L1xx data sheet.
//
// To reproduce a current measurement from the data sheet, uncomment the
// corresponding settings below, recompile, download, and run on an SiM3L1xx
// MCU card.
//
// Note: While the debug adatper is physically plugged into the board, there
// will be approximately 60 uA of leakage current. Remove the debug adapter
// from the 10-pin connector before taking current measurements.
//
// Note: Running the AHB/APB at 16 kHz will cause issues with debugging and
// downloading code.  Press and hold PB1.4 (switch A) after a reset to
// re-download code to the device.
//
// Resources Used:
//-----------------------------------------------------------------------------
// LDO0 for active voltage scaling
// DCDC0
// PLL0 for AHB/APB frequencies above 20 MHz
// LPOSC0 for AHB/APB frequencies 20 MHz and below
// LFOSC0 for AHB/APB 16 kHz frequency
//
// PB0.1 AHB/16 out, if enabled
// PB1.4 switch A - hold core out of reset
//
// Notes On Example and Modes:
//-----------------------------------------------------------------------------
// Clock frequencies are dependent upon the mode selected below.
//
// NOTE: Remove all three jumpers on J7 and measure across the Imeasure jumper
// (JP2).
//
// How to Use:
//-----------------------------------------------------------------------------
// Using a fixed VBAT:
//
// 1) Connect a USB Debug Adapter to the 10-pin CoreSight connector (J14) on
//    the SiM3L1xx MCU card.
// 2) Remove the three shorting blocks from J7.
// 3) Remove the JP2 Imeasure shorting block and put a multimeter across
//    (positive side on the top pin).
// 4) Move the VBAT Sel switch (SW2) to the middle +3.3V_VREG position.
// 5) Move the VIO Sel (SW8) and VIORF Sel (SW9) switches to the bottom VBAT
//    position.
// 6) Connect the 9 V power adapter to POWER (J6).
// 7) Select the desired data sheet settings from the options in this file.
// 8) Compile and download the code to the board.
// 9) Remove the debug adapter connection.
// 10) Reset the device.
// 11) Measure the power of the device.
//
// Using a varying VBAT:
//
// 1) Connect a USB Debug Adapter to the 10-pin CoreSight connector (J14) on
//    the SiM3L1xx MCU card.
// 2) Remove the three shorting blocks from J7.
// 3) Remove the JP2 Imeasure shorting block.
// 4) Move the VBAT Sel switch (SW2) to the bottom BAT position.
// 5) Move the VIO Sel (SW8) and VIORF Sel (SW9) switches to the bottom VBAT
//    position.
// 6) (Optional) Place a battery in the BT1 holder.
// 7) (Optional) Instead of a battery, connect the positive terminal of a bench
//    power supply to the positive terminal of the multimeter, the negative
//    terminal of the power supply to the MCU card ground, and the negative
//    terminal of the multimeter to the top pin of the Imeasure jumper (JP2).
//    This configuration will prevent any circuitry on the board from
//    interfering with the power measurement.
// 8) Select the desired data sheet settings from the options in this file.
// 9) Compile and download the code to the board.
// 10) Remove the debug adapter connection.
// 11) Reset the device.
// 12) Measure the power of the device.
//
//
// Revision History:
//-----------------------------------------------------------------------------
//
// Release 1.0
//    -Initial Revision (BD and TP)
//    -2 AUG 12
//
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

// si32 HAL
#include <si32_device.h>
#include <SI32_PBSTD_A_Type.h>
#include <SI32_RSTSRC_A_Type.h>
#include <SI32_CLKCTRL_A_Type.h>
#include <SI32_EXTOSC_A_Type.h>
#include <SI32_FLASHCTRL_A_Type.h>
#include <SI32_RTC_B_Type.h>
#include <SI32_PMU_A_Type.h>
#include <SI32_LDO_A_Type.h>
#include <SI32_PLL_A_Type.h>
#include <SI32_DCDC_A_Type.h>
#include <SI32_LCD_A_Type.h>
#include <SI32_PBCFG_A_Type.h>
#include <SI32_WDTIMER_A_Type.h>

//-----------------------------------------------------------------------------
// Power Measurement Configurations
//-----------------------------------------------------------------------------
// Uncomment the defines for the desired power measurement or use the defines
// to measure a desired configuration.

//=============================================================================
// Data Sheet Specifications
//=============================================================================

// Normal Mode (PM0)
// Full speed with code executing from Flash, peripheral clocks ON
// FAHB = 49 MHz, FAPB = 24.5 MHz

#define PLL49
#define APBDIV2
#define AVS_ON

// Normal Mode (PM0)
// Full speed with code executing from Flash, peripheral clocks ON
// FAHB = 20 MHz, FAPB = 10 MHz
/*
#define LPOSC20
#define APBDIV2
#define AVS_ON
*/
// Normal Mode (PM0)
// Full speed with code executing from Flash, peripheral clocks ON
// FAHB = 2.5 MHz, FAPB = 1.25 MHz
/*
#define LPOSC2p5
#define APBDIV2
#define AVS_ON
*/
// Normal Mode (PM0)
// Full speed with code executing from Flash, peripheral clocks OFF
// FAHB = 49 MHz, FAPB = 24.5 MHz
/*
#define PLL49
#define APBDIV2
#define PERIPH_CLOCKS_OFF
#define AVS_ON
*/
// Normal Mode (PM0)
// Full speed with code executing from Flash, peripheral clocks OFF
// FAHB = 20 MHz, FAPB = 10 MHz
/*
#define LPOSC20
#define APBDIV2
#define PERIPH_CLOCKS_OFF
#define AVS_ON
*/
// Normal Mode (PM0)
// Full speed with code executing from Flash, peripheral clocks OFF
// FAHB = 2.5 MHz, FAPB = 1.25 MHz
/*
#define LPOSC2p5
#define APBDIV2
#define PERIPH_CLOCKS_OFF
#define AVS_ON
*/
// Normal Mode (PM0)
// Full speed with code executing from Flash, LDOs powered by dc-dc,
// peripheral clocks OFF
// FAHB = 49 MHz, FAPB = 24.5 MHz, VBAT = 3.3 V or
// VBAT = 3.8 V (external supply needed)
/*
#define PLL49
#define APBDIV2
#define PERIPH_CLOCKS_OFF
#define AVS_ON
#define DCDC_ON
#define DCDC_5to15mA
*/
// Normal Mode (PM0)
// Full speed with code executing from Flash, LDOs powered by dc-dc,
// peripheral clocks OFF
// FAHB = 20 MHz, FAPB = 10 MHz, VBAT = 3.3 V or
// VBAT = 3.8 V (external supply needed)
/*
#define LPOSC20
#define APBDIV2
#define PERIPH_CLOCKS_OFF
#define AVS_ON
#define DCDC_ON
#define DCDC_lt5mA
*/
// Power Mode 4 (PM4)
// Slower clock speed with code executing from Flash, peripheral clocks ON
// FAHB = FAPB = 16 kHz, VBAT = 1.8 V or VBAT = 3.8 V (external supply needed)
// Note: Running the AHB/APB at 16 kHz will cause issues with debugging and
// downloading code.  Press and hold PB1.4 (switch A) after a reset to
// re-download code to the device.
/*
#define LFOSC16k
#define AVS_ON
*/

//=============================================================================
// Code Dependency Settings
//=============================================================================

//#define PLLVARY
//#define NOP_CODE
//#define BRANCH_CODE
/*
// 50 MHz
#define PLL_FREQUENCY    50000000
#define PLL_NUMERATOR        1600
#define PLL_DENOMINATOR        80
#define FLASH_SPMD              2
*/
/*
// 45 MHz
#define PLL_FREQUENCY    45000000
#define PLL_NUMERATOR        1800
#define PLL_DENOMINATOR       100
#define FLASH_SPMD              2
*/
/*
// 40 MHz
#define PLL_FREQUENCY    40000000
#define PLL_NUMERATOR        1600
#define PLL_DENOMINATOR       100
#define FLASH_SPMD              1
*/
/*
// 35 MHz
#define PLL_FREQUENCY    35000000
#define PLL_NUMERATOR        1400
#define PLL_DENOMINATOR       100
#define FLASH_SPMD              1
*/
/*
// 30 MHz
#define PLL_FREQUENCY    30000000
#define PLL_NUMERATOR        1200
#define PLL_DENOMINATOR       100
#define FLASH_SPMD              1
*/
/*
// 25 MHz
#define PLL_FREQUENCY    25000000
#define PLL_NUMERATOR        1000
#define PLL_DENOMINATOR       100
#define FLASH_SPMD              1
*/

// Check the AHB frequency using this
//#define AHB_OUT_ENABLE

//=============================================================================
// Configurable Defines
//=============================================================================

// AHB/APB Clock Settings
//#define PLL49
// PLLVARY must be defined in conjunction with a frequency, numerator,
// denominator, and the Flash speed mode
/*
#define PLLVARY
#define PLL_FREQUENCY    50000000
#define PLL_NUMERATOR         800
#define PLL_DENOMINATOR        40
#define FLASH_SPMD              2
*/
//#define LPOSC20
//#define LPOSC5
//#define LPOSC2p5
//#define LPOSC1p25
//#define LPOSC0p625
//#define LFOSC16k
//#define EXTOSC0to5

//#define APBDIV2

//#define PERIPH_CLOCKS_OFF

// LDO0 Adaptive Voltage Scaling (AVS)
//#define AVS_ON

// DCDC0 Settings
// DCDC_ON must be defined in conjunction with a range
/*
#define DCDC_ON
#define DCDC_gt15mA
#define DCDC_5to15mA
#define DCDC_lt5mA
*/

// AHB Output Setting for observation
// When enabled, AHB/16 is output on PB0.1
//#define AHB_OUT_ENABLE

//-----------------------------------------------------------------------------
// Global Defines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Function Prototypes
//-----------------------------------------------------------------------------

//{{Config
void Clocks_Configure(void);
void PLL_lock_freq_to_hz(uint32_t freq);
void DCDC0_Configure(void);
void RTC0_Configure(void);
void PB_Configure(void);
void PMU0_Configure(void);
void LDO0_Configure(void);
//}}Config

//{{App
void PowerModes_0_or_4(void);
void nop_code_loop(void);
//}}App

//{{Loop
//}}Loop

//-----------------------------------------------------------------------------
// main()
//-----------------------------------------------------------------------------
int main()
{
   // Initialization code
   //{{Config
   while (!SI32_PBSTD_A_read_pin(SI32_PBSTD_1, 4));

   Clocks_Configure();

   // Disable the SysTick timer to prevent these interrupts
   // from waking the core.
   SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;

   RTC0_Configure();
   PB_Configure();
   PMU0_Configure();

   //}}Config

   // Application code
   //{{App
   PowerModes_0_or_4();
   //}}App

   while(1)
   {
      // Loop forever
      //{{Loop
      //}}Loop
   }

}

//-----------------------------------------------------------------------------
// Initialization Code
//-----------------------------------------------------------------------------

//{{Configure

//-----------------------------------------------------------------------------
// This function is invoked by the CMSIS-required SystemInit() function in
// system_<device>.c.  SystemInit() is invoked by Reset_Handler() when the
// CPU boots.
void mySystemInit()
{
   // Disable the watchdog timer to prevent device resets
   // Any lines added to this function should be added below this line of code
   SI32_WDTIMER_A_stop_counter(SI32_WDTIMER_0);
}

void Clocks_Configure(void)
{
   SI32_CLKCTRL_A_enable_apb_to_all_modules(SI32_CLKCTRL_0);

   #ifdef APBDIV2
      SI32_CLKCTRL_A_select_apb_divider_2(SI32_CLKCTRL_0);
   #endif

   #ifdef PLL49
      SI32_PLL_A_select_dco_free_running_mode(SI32_PLL_0);
      SI32_FLASHCTRL_A_enter_read_store_mode(SI32_FLASHCTRL_0);
      SI32_FLASHCTRL_A_select_flash_speed_mode(SI32_FLASHCTRL_0, 2);
      SI32_FLASHCTRL_A_select_flash_read_time_fast(SI32_FLASHCTRL_0);
      SI32_CLKCTRL_A_select_ahb_source_pll(SI32_CLKCTRL_0);
   #endif

   #ifdef PLLVARY
      SI32_PLL_A_select_reference_clock_source_lp0osc(SI32_PLL_0);
      SI32_PLL_A_set_numerator(SI32_PLL_0, PLL_NUMERATOR);
      SI32_PLL_A_set_denominator(SI32_PLL_0, PLL_DENOMINATOR);
      PLL_lock_freq_to_hz(PLL_FREQUENCY);
      SI32_PLL_A_select_dco_free_running_mode(SI32_PLL_0);
      SI32_FLASHCTRL_A_enter_read_store_mode(SI32_FLASHCTRL_0);
      SI32_FLASHCTRL_A_select_flash_speed_mode(SI32_FLASHCTRL_0, FLASH_SPMD);
      SI32_FLASHCTRL_A_select_flash_read_time_fast(SI32_FLASHCTRL_0);
      SI32_CLKCTRL_A_select_ahb_source_pll(SI32_CLKCTRL_0);
   #endif

   #ifdef LPOSC20
      SI32_CLKCTRL_A_select_ahb_source_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_FLASHCTRL_A_select_flash_read_time_fast(SI32_FLASHCTRL_0);
   #endif

   #ifdef LPOSC5
      SI32_CLKCTRL_A_select_ahb_source_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_divider(SI32_CLKCTRL_0, 2);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif

   #ifdef LPOSC2p5
      SI32_CLKCTRL_A_select_ahb_source_divided_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif

   #ifdef LPOSC1p25
      SI32_CLKCTRL_A_select_ahb_source_divided_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_divider(SI32_CLKCTRL_0, 1);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif

   #ifdef LPOSC0p625
      SI32_CLKCTRL_A_select_ahb_source_divided_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_divider(SI32_CLKCTRL_0, 2);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif

   #ifdef LFOSC16k
      SI32_CLKCTRL_A_select_ahb_source_low_frequency_oscillator(SI32_CLKCTRL_0);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif

   #ifdef EXTOSC0to5
      SI32_PBCFG_A_enable_crossbar_0(SI32_PBCFG_0);
      SI32_EXTOSC_A_select_oscillator_mode_cmos(SI32_EXTOSC_0);
      SI32_CLKCTRL_A_enable_extosc_in(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_source_external_oscillator(SI32_CLKCTRL_0);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif
}

void PLL_lock_freq_to_hz(uint32_t freq)
{
  // Calculate initial mode guess
  uint8_t mode = 4;
  if (freq < 35000000)
  {
    mode = 0;
  }
  else if (freq < 45000000)
  {
    mode = 1;
  }
  else if (freq < 55000000)
  {
    mode = 2;
  }
  else if (freq < 70000000)
  {
    mode = 3;
  }

  // Set range
  SI32_PLL_A_select_disable_dco_output(SI32_PLL_0);
  SI32_PLL_A_set_frequency_adjuster_value(SI32_PLL_0, 0xFFF);
  SI32_PLL_A_set_output_frequency_range(SI32_PLL_0, mode);

  // Lock and block for result
  SI32_PLL_A_select_dco_frequency_lock_mode(SI32_PLL_0);

  while (!((SI32_PLL_A_is_locked(SI32_PLL_0) == true)
           || (SI32_PLL_A_is_saturation_low_interrupt_pending(SI32_PLL_0) == true)
           || (SI32_PLL_A_is_saturation_high_interrupt_pending(SI32_PLL_0) == true)));

  if (!SI32_PLL_A_is_locked(SI32_PLL_0))
  {
    SI32_PLL_A_select_disable_dco_output(SI32_PLL_0);

    // Adjust the range, if necessary
    if (SI32_PLL_A_is_saturation_low_interrupt_pending(SI32_PLL_0) && (mode < 4))
    {
      ++mode;
    }
    else if (mode < 4)
    {
      --mode;
    }
    SI32_PLL_A_set_output_frequency_range(SI32_PLL_0, mode);

    // Lock and block for result
    SI32_PLL_A_select_dco_frequency_lock_mode(SI32_PLL_0);

    while (!((SI32_PLL_A_is_locked(SI32_PLL_0) == true)
             || (SI32_PLL_A_is_saturation_low_interrupt_pending(SI32_PLL_0) == true)
             || (SI32_PLL_A_is_saturation_high_interrupt_pending(SI32_PLL_0) == true)));
  }
}

void LDO0_Configure(void)
{
   // From the Reference Manual, LDO chapter:
   //
   // The LDO outputs are adjustable, with a valid range of 0.8 to 1.9 V.
   // Additional power savings for the device may be realized by lowering the
   // LDO output voltage of the memory and digital regulators under appropriate
   // conditions, such as lower clock speeds. The regulators are adjusted using
   // the ALDOOVAL, DLDOOVAL and MLDOOVAL bit fields in the CONTROL register.
   // The adjustment fields are linear, with each LSB representing
   // approximately 50 mV. To adjust the memory LDO to 1.6 V, for example,
   // the MLDOOVAL field can be set to 0x10 (0.8 V + 50 mV x 16). The LDO
   // outputs are also available to the SARADC and Comparators for monitoring
   // purposes.
   //
   // The optimal LDO settings for different operational speeds are to be
   // determined (TBD), pending full characterization of silicon over process,
   // temperature, and voltage corners. Preliminary silicon characterization
   // suggests that the safe minimum voltage for the memory LDO at all speeds
   // is 1.6 V. The digital LDO may be safely set to 1.0 V at speeds of 20 MHz
   // or less, and 1.3 V between 20 and 50 MHz. The analog LDO should normally
   // be left at 1.8 V output during active operation. All LDOs may be adjusted
   // prior to entering PM8, to extend the external supply operating range. See
   // the electrical specification tables in the datasheet for specified output
   // settings.

   #ifdef PLL49
      // DLDO = 1.3 V, MLDO = 1.6 V, ALDO = 1.8 V, high bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x002A3034);
   #endif

   #ifdef PLLVARY
      // DLDO = 1.3 V, MLDO = 1.6 V, ALDO = 1.8 V, high bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x002A3034);
   #endif

   #ifdef LPOSC20
      // DLDO = 1.3 V, MLDO = 1.6 V, ALDO = 1.8 V, high bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x002A3034);
   #endif

   #ifdef EXTOSC0to5
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC5
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC2p5
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC1p25
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC0p625
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LFOSC16k
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif
}

void DCDC0_Configure(void)
{
   // Enable the clock to the DCDC0 module
   SI32_CLKCTRL_A_enable_apb_to_modules_0(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG0_DCDC0CEN_ENABLED_U32
      | SI32_CLKCTRL_A_APBCLKG0_LCD0CEN_ENABLED_U32);
   SI32_CLKCTRL_A_enable_apb_to_modules_1(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG1_MISC0CEN_ENABLED_U32);

   // Configure the DCDC0 module
   SI32_LCD_A_enable_dcdc_bias(SI32_LCD_0);
   SI32_DCDC_A_enable_module(SI32_DCDC_0);

   #ifdef DCDC_gt15mA
      SI32_DCDC_A_select_power_switch_size(SI32_DCDC_0, 3);
      SI32_DCDC_A_enable_sync_mode(SI32_DCDC_0);
      SI32_DCDC_A_select_minimum_pulse_width(SI32_DCDC_0, 0);
   #endif

   #ifdef DCDC_5to15mA
      SI32_DCDC_A_select_power_switch_size(SI32_DCDC_0, 0);
      SI32_DCDC_A_enable_sync_mode(SI32_DCDC_0);
      SI32_DCDC_A_select_minimum_pulse_width(SI32_DCDC_0, 0);
   #endif

   #ifdef DCDC_lt5mA
      SI32_DCDC_A_select_power_switch_size(SI32_DCDC_0, 0);
      SI32_DCDC_A_enable_async_mode(SI32_DCDC_0);
      SI32_DCDC_A_select_minimum_pulse_width(SI32_DCDC_0, 3);
   #endif

   SI32_DCDC_A_select_output_voltage(SI32_DCDC_0, 0x0C);
   SI32_DCDC_A_select_clock_source_local(SI32_DCDC_0);

   // Wait until output voltage reached programmed value
   while (!SI32_DCDC_A_is_output_low_flag_set(SI32_DCDC_0)) {};

   // Check that the output voltage hasn't exceed the programmed value
   while (SI32_DCDC_A_is_output_high_flag_set(SI32_DCDC_0)) {};

   SI32_LDO_A_select_digital_source_dcdc_output(SI32_LDO_0);
   SI32_LDO_A_select_memory_source_dcdc_output(SI32_LDO_0);
   SI32_LDO_A_select_analog_source_dcdc_output(SI32_LDO_0);
}

void RTC0_Configure(void)
{
   SI32_CLKCTRL_A_enable_apb_to_modules_1(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG1_MISC0CEN_ENABLED_U32);

   SI32_RTC_B_set_clock_source_lfo(SI32_RTC_0);

   #ifdef LFOSC16k
   #else
      SI32_RTC_B_disable_low_frequency_oscillator_output(SI32_RTC_0);
      SI32_RTC_B_disable_low_frequency_oscillator(SI32_RTC_0);
   #endif

   SI32_RTC_B_disable_clock_out(SI32_RTC_0);

   // Disable the RTC timer
   SI32_RTC_B_disable_rtc_timer(SI32_RTC_0);

   // Disable the clock to the RTC0 module
   SI32_CLKCTRL_A_disable_apb_to_modules_1(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG1_MISC0CEN_ENABLED_U32);
}

void PB_Configure(void)
{
   SI32_CLKCTRL_A_enable_apb_to_modules_0(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG0_PB0CEN_ENABLED_U32);

   // Enable the LED drivers (PB1.4, PB1.5, PB1.6, PB1.7)
   SI32_PBSTD_A_set_pins_push_pull_output(SI32_PBSTD_1, 0x000000F0);
   SI32_PBSTD_A_write_pins_low(SI32_PBSTD_1,0x00F0);

   // PB1.0 and PB1.1 are connected to the EBID processor and are being pulled
   // low when the MCU Card is disconnected from the UDP Motherboard. Setting
   // these pins to analog mode disconnects the pins from the weak pull-ups,
   // which draw current.
   SI32_PBSTD_A_set_pins_analog(SI32_PBSTD_1, 0x00000003);

   #ifdef AHB_OUT_ENABLE
      SI32_PBSTD_A_set_pins_push_pull_output(SI32_PBSTD_0, 0x00000002);
      SI32_PBSTD_A_write_pbskipen(SI32_PBSTD_0, 0x00000001);
      SI32_PBCFG_A_enable_xbar0_peripherals(SI32_PBCFG_0,
         SI32_PBCFG_A_XBAR0_AHBEN);
   #endif

   SI32_PBCFG_A_enable_crossbar_0(SI32_PBCFG_0);
}

void PMU0_Configure(void)
{
   // Enable the clock to the PMU
   SI32_CLKCTRL_A_enable_apb_to_modules_1(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG1_MISC0CEN_ENABLED_U32);

   SI32_CLKCTRL_A_disable_power_mode_8(SI32_CLKCTRL_0);

   SI32_PMU_A_clear_wakeup_flags(SI32_PMU_0);

   // Disable all resets
   SI32_RSTSRC_A_initialize(SI32_RSTSRC_0, 0x00000000);
}

//}}Config

//-----------------------------------------------------------------------------
// Subroutines
//-----------------------------------------------------------------------------

//{{App

void PowerModes_0_or_4(void)
{
   // 1. Enable the clocks to peripherals that will be configured by firmware.
   // (see Clocks_Configure())

   // 2. Select the desired clock source (LPOSC0, PLL0, etc.) and speed for
   // both AHB and APB clocks.
   // (see Clocks_Configure())

   // 3. Select the desired adaptive voltage scaling settings using the LDO
   // module.

   #ifdef AVS_ON
      LDO0_Configure();
   #endif

   #ifdef DCDC_ON
      DCDC0_Configure();
   #endif

   // 4. (Optional) Disable the retention mode of any enabled RAM banks.

   // 5. Set the pins in the lowest power configuration for this mode.

   SI32_PBSTD_A_write_pins_high(SI32_PBSTD_1,0x00F0);

   // 6. Disable all unused peripherals.

   // 7. Disable the clocks to all unused peripherals.

   #ifdef PERIPH_CLOCKS_OFF
      SI32_CLKCTRL_A_disable_apb_to_all_modules(SI32_CLKCTRL_0);
   #endif

   // 8. Jump to code in Flash.

   nop_code_loop();
}

void nop_code_loop(void)
{
   while(1)
   {
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
      __NOP();
   }
}

//}}App

//-----------------------------------------------------------------------------
// Interrupt Service Routines (ISRs)
//-----------------------------------------------------------------------------

// SysTick is disabled in this example, so this handler doesn't need to do
// anything.
void SysTick_Handler(void)
{
}

//{{ISR
//}}ISR

//-eof-------------------------------------------------------------------------
