//-----------------------------------------------------------------------------
// Copyright (c) 2012 by Silicon Laboratories
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Silicon Laboratories End User
// License Agreement which accompanies this distribution, and is available at
// http://developer.silabs.com/legal/version/v10/License_Agreement_v10.htm
//
//
// Original content and implementation provided by Silicon Laboratories
//-----------------------------------------------------------------------------
//
// Program Description:
//-----------------------------------------------------------------------------
//
// This example code runs in Power Mode 2 or Power Mode 6 to reproduce the
// power numbers provided in the SiM3L1xx data sheet.
//
// To reproduce a current measurement from the data sheet, uncomment the
// corresponding settings below, recompile, download, and run on an SiM3L1xx
// MCU card.
//
// Note: While the debug adatper is physically plugged into the board, there
// will be approximately 60 uA of leakage current. Remove the debug adapter
// from the 10-pin connector before taking current measurements.
//
// Note: Running the AHB/APB at 16 kHz will cause issues with debugging and
// downloading code.  Press and hold PB1.4 (switch A) after a reset to
// re-download code to the device.
//
// Resources Used:
//-----------------------------------------------------------------------------
// LDO0 for active voltage scaling
// DCDC0
// PLL0 for AHB/APB frequencies above 20 MHz
// LPOSC0 for AHB/APB frequencies 20 MHz and below
// LFOSC0 for AHB/APB 16 kHz frequency
//
// PB0.1 AHB/16 out, if enabled
// PB1.4 switch A - hold core out of reset
// PB1.5 switch B - port match wake from PM2/6
// PB1.7 LED - turns on after PM2/6 exits
//
// Notes On Example and Modes:
//-----------------------------------------------------------------------------
// Clock frequencies are dependent upon the mode selected below.
//
// NOTE: Remove all three jumpers on J7 and measure across the Imeasure jumper
// (JP2).
//
// How to Use:
//-----------------------------------------------------------------------------
// Using a fixed VBAT:
//
// 1) Connect a USB Debug Adapter to the 10-pin CoreSight connector (J14) on
//    the SiM3L1xx MCU card.
// 2) Remove the three shorting blocks from J7.
// 3) Remove the JP2 Imeasure shorting block and put a multimeter across
//    (positive side on the top pin).
// 4) Move the VBAT Sel switch (SW2) to the middle +3.3V_VREG position.
// 5) Move the VIO Sel (SW8) and VIORF Sel (SW9) switches to the bottom VBAT
//    position.
// 6) Connect the 9 V power adapter to POWER (J6).
// 7) Select the desired data sheet settings from the options in this file.
// 8) Compile and download the code to the board.
// 9) Remove the debug adapter connection.
// 10) Reset the device.
// 11) Press PB1.7 to place the device in PM2 or PM6.  The PB1.5 LED will turn
//     off.
// 12) Measure the power of the device.
// 13) Press PB1.5 to wake the device from PM2 or PM6.  When this happens,
//     the PB1.7 LED will turn on and the core will sit in an infinite while(1)
//     loop.
//
// Using a varying VBAT:
//
// 1) Connect a USB Debug Adapter to the 10-pin CoreSight connector (J14) on
//    the SiM3L1xx MCU card.
// 2) Remove the three shorting blocks from J7.
// 3) Remove the JP2 Imeasure shorting block.
// 4) Move the VBAT Sel switch (SW2) to the bottom BAT position.
// 5) Move the VIO Sel (SW8) and VIORF Sel (SW9) switches to the bottom VBAT
//    position.
// 6) (Optional) Place a battery in the BT1 holder.
// 7) (Optional) Instead of a battery, connect the positive terminal of a bench
//    power supply to the positive terminal of the multimeter, the negative
//    terminal of the power supply to the MCU card ground, and the negative
//    terminal of the multimeter to the top pin of the Imeasure jumper (JP2).
//    This configuration will prevent any circuitry on the board from
//    interfering with the power measurement.
// 8) Select the desired data sheet settings from the options in this file.
// 9) Compile and download the code to the board.
// 10) Remove the debug adapter connection.
// 11) Reset the device.
// 12) Press PB1.7 to place the device in PM2 or PM6.  The PB1.5 LED will turn
//     off.
// 13) Measure the power of the device.
// 14) Press PB1.5 to wake the device from PM2 or PM6. When this happens,
//     the PB1.7 LED will turn on and the core will sit in an infinite while(1)
//     loop.
//
//
// Revision History:
//-----------------------------------------------------------------------------
//
// Release 1.0
//    -Initial Revision (BD and TP)
//    -3 AUG 12
//
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

// si32 HAL
#include <si32_device.h>
#include <SI32_PBSTD_A_Type.h>
#include <SI32_RSTSRC_A_Type.h>
#include <SI32_CLKCTRL_A_Type.h>
#include <SI32_EXTOSC_A_Type.h>
#include <SI32_FLASHCTRL_A_Type.h>
#include <SI32_RTC_B_Type.h>
#include <SI32_PMU_A_Type.h>
#include <SI32_LDO_A_Type.h>
#include <SI32_PLL_A_Type.h>
#include <SI32_DCDC_A_Type.h>
#include <SI32_LCD_A_Type.h>
#include <SI32_PBCFG_A_Type.h>
#include <SI32_WDTIMER_A_Type.h>

//-----------------------------------------------------------------------------
// Power Measurement Configurations
//-----------------------------------------------------------------------------
// Uncomment the defines for the desired power measurement or use the defines
// to measure a desired configuration.

//=============================================================================
// Data Sheet Specifications
//=============================================================================

// Power Mode 2 (PM2)
// Core halted with peripheral clocks ON
// FAHB = 49 MHz, FAPB = 24.5 MHz

#define PLL49
#define APBDIV2
#define AVS_ON

// Power Mode 2 (PM2)
// Core halted with peripheral clocks ON
// FAHB = 20 MHz, FAPB = 10 MHz
/*
#define LPOSC20
#define APBDIV2
#define AVS_ON
*/
// Power Mode 2 (PM2)
// Core halted with peripheral clocks ON
// FAHB = 2.5 MHz, FAPB = 1.25 MHz
/*
#define LPOSC2p5
#define APBDIV2
#define AVS_ON
*/
// Power Mode 2 (PM2)
// Core halted with only Port I/O clocks on (wake from pin).
// Press the PB1.5 pin to wake the device from PM2
// FAHB = 49 MHz, FAPB = 24.5 MHz
/*
#define PLL49
#define APBDIV2
#define PERIPH_CLOCKS_OFF
#define AVS_ON
*/
// Power Mode 2 (PM2)
// Core halted with only Port I/O clocks on (wake from pin).
// Press the PB1.5 pin to wake the device from PM2
// FAHB = 20 MHz, FAPB = 10 MHz
/*
#define LPOSC20
#define APBDIV2
#define PERIPH_CLOCKS_OFF
#define AVS_ON
*/
// Power Mode 2 (PM2)
// Core halted with only Port I/O clocks on (wake from pin).
// Press the PB1.5 pin to wake the device from PM2
// FAHB = 2.5 MHz, FAPB = 1.25 MHz
/*
#define LPOSC2p5
#define APBDIV2
#define PERIPH_CLOCKS_OFF
#define AVS_ON
*/
// Power Mode 6 (PM6)
// Core halted with peripheral clocks ON
// FAHB = FAPB = 16 kHz, VBAT = 1.8 V or VBAT = 3.8 V (external supply needed)
// Note: Running the AHB/APB this slowly will cause issues with debugging and
// downloading code.  Press and hold PB1.4 (switch A) after a reset to
// re-download code to the device.
/*
#define LFOSC16k
#define AVS_ON
*/

//=============================================================================
// Configurable Defines
//=============================================================================

// AHB/APB Clock Settings
//#define PLL49
//#define LPOSC20
//#define LPOSC5
//#define LPOSC2p5
//#define LPOSC1p25
//#define LPOSC0p625
//#define LFOSC16k
//#define EXTOSC0to5

//#define APBDIV2

//#define PERIPH_CLOCKS_OFF

// LDO0 Adaptive Voltage Scaling (AVS)
//#define AVS_ON

// DCDC0 Settings
// DCDC_ON must be defined in conjunction with a range
/*
#define DCDC_ON
#define DCDC_gt15mA
#define DCDC_5to15mA
#define DCDC_lt5mA
*/

// AHB Output Setting for observation
// When enabled, AHB/16 is output on PB0.1
//#define AHB_OUT_ENABLE

//-----------------------------------------------------------------------------
// Global Defines
//-----------------------------------------------------------------------------

#if   defined ( __CC_ARM ) // Keil/ARM

// This define allows code to be placed in RAM using uVision
// In order for this define to work, the project must link to the customized
// linker file in the PowerModes_1_and_5/src directory.
// To ensure the functions are in the RAM region of Flash, open the project
// map file and search for "PowerModes_2_or_6".  This function should be
// located above address 0x2000_0000.
#define __SI32_RRAM __attribute__ ((section("MCU_RAM_CODE")))

#endif

//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Function Prototypes
//-----------------------------------------------------------------------------

//{{Config
void Clocks_Configure(void);
void DCDC0_Configure(void);
void RTC0_Configure(void);
void PB_Configure(void);
void PMU0_Configure(void);
void LDO0_Configure(void);
//}}Config

//{{App
void PowerModes_2_or_6(void);
//}}App

//{{Loop
//}}Loop

//-----------------------------------------------------------------------------
// main()
//-----------------------------------------------------------------------------
int main()
{
   // Initialization code
   //{{Config
   while (!SI32_PBSTD_A_read_pin(SI32_PBSTD_1, 4));

   Clocks_Configure();
   PB_Configure();

   // Press PB1.7 to enter PM2 or PM6
   // Turn off the red PB1.5 LED.
   while (SI32_PBSTD_A_read_pin(SI32_PBSTD_1, 7));
   SI32_PBSTD_A_write_pins_high(SI32_PBSTD_1,0x0020);

   RTC0_Configure();
   PMU0_Configure();
   //}}Config

   // Application code
   //{{App
   PowerModes_2_or_6();
   //}}App

   while(1)
   {
      // Loop forever
      //{{Loop
      //}}Loop
   }

}

//-----------------------------------------------------------------------------
// Initialization Code
//-----------------------------------------------------------------------------

//{{Configure

//-----------------------------------------------------------------------------
// This function is invoked by the CMSIS-required SystemInit() function in
// system_<device>.c.  SystemInit() is invoked by Reset_Handler() when the
// CPU boots.
void mySystemInit()
{
   // Disable the watchdog timer to prevent device resets
   // Any lines added to this function should be added below this line of code
   SI32_WDTIMER_A_stop_counter(SI32_WDTIMER_0);
}

void Clocks_Configure(void)
{
   SI32_CLKCTRL_A_enable_apb_to_all_modules(SI32_CLKCTRL_0);

   #ifdef APBDIV2
      SI32_CLKCTRL_A_select_apb_divider_2(SI32_CLKCTRL_0);
   #endif

   #ifdef PLL49
      SI32_PLL_A_select_dco_free_running_mode(SI32_PLL_0);
      SI32_FLASHCTRL_A_enter_read_store_mode(SI32_FLASHCTRL_0);
      SI32_FLASHCTRL_A_select_flash_speed_mode(SI32_FLASHCTRL_0, 2);
      SI32_FLASHCTRL_A_select_flash_read_time_fast(SI32_FLASHCTRL_0);
      SI32_CLKCTRL_A_select_ahb_source_pll(SI32_CLKCTRL_0);
   #endif

   #ifdef LPOSC20
      SI32_CLKCTRL_A_select_ahb_source_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_FLASHCTRL_A_select_flash_read_time_fast(SI32_FLASHCTRL_0);
   #endif

   #ifdef LPOSC5
      SI32_CLKCTRL_A_select_ahb_source_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_divider(SI32_CLKCTRL_0, 2);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif

   #ifdef LPOSC2p5
      SI32_CLKCTRL_A_select_ahb_source_divided_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif

   #ifdef LPOSC1p25
      SI32_CLKCTRL_A_select_ahb_source_divided_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_divider(SI32_CLKCTRL_0, 1);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif

   #ifdef LPOSC0p625
      SI32_CLKCTRL_A_select_ahb_source_divided_low_power_oscillator(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_divider(SI32_CLKCTRL_0, 2);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif

   #ifdef LFOSC16k
      SI32_CLKCTRL_A_select_ahb_source_low_frequency_oscillator(SI32_CLKCTRL_0);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif

   #ifdef EXTOSC0to5
      SI32_PBCFG_A_enable_crossbar_0(SI32_PBCFG_0);
      SI32_EXTOSC_A_select_oscillator_mode_cmos(SI32_EXTOSC_0);
      SI32_CLKCTRL_A_enable_extosc_in(SI32_CLKCTRL_0);
      SI32_CLKCTRL_A_select_ahb_source_external_oscillator(SI32_CLKCTRL_0);
      SI32_LDO_A_select_digital_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_memory_bias_low(SI32_LDO_0);
      SI32_LDO_A_select_analog_bias_low(SI32_LDO_0);
      SI32_FLASHCTRL_A_select_flash_read_time_slow(SI32_FLASHCTRL_0);
   #endif
}

void LDO0_Configure(void)
{
   // From the Reference Manual, LDO chapter:
   //
   // The LDO outputs are adjustable, with a valid range of 0.8 to 1.9 V.
   // Additional power savings for the device may be realized by lowering the
   // LDO output voltage of the memory and digital regulators under appropriate
   // conditions, such as lower clock speeds. The regulators are adjusted using
   // the ALDOOVAL, DLDOOVAL and MLDOOVAL bit fields in the CONTROL register.
   // The adjustment fields are linear, with each LSB representing
   // approximately 50 mV. To adjust the memory LDO to 1.6 V, for example,
   // the MLDOOVAL field can be set to 0x10 (0.8 V + 50 mV x 16). The LDO
   // outputs are also available to the SARADC and Comparators for monitoring
   // purposes.
   //
   // The optimal LDO settings for different operational speeds are to be
   // determined (TBD), pending full characterization of silicon over process,
   // temperature, and voltage corners. Preliminary silicon characterization
   // suggests that the safe minimum voltage for the memory LDO at all speeds
   // is 1.6 V. The digital LDO may be safely set to 1.0 V at speeds of 20 MHz
   // or less, and 1.3 V between 20 and 50 MHz. The analog LDO should normally
   // be left at 1.8 V output during active operation. All LDOs may be adjusted
   // prior to entering PM8, to extend the external supply operating range. See
   // the electrical specification tables in the datasheet for specified output
   // settings.

   #ifdef PLL49
      // DLDO = 1.3 V, MLDO = 1.6 V, ALDO = 1.8 V, high bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x002A3034);
   #endif

   #ifdef LPOSC20
      // DLDO = 1.3 V, MLDO = 1.6 V, ALDO = 1.8 V, high bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x002A3034);
   #endif

   #ifdef EXTOSC0to5
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC5
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC2p5
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC1p25
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LPOSC0p625
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif

   #ifdef LFOSC16k
      // DLDO = 1.0 V, MLDO = 1.6 V, ALDO = 1.8 V, low bias
      SI32_LDO_A_write_control(SI32_LDO_0, 0x00041014);
   #endif
}

void DCDC0_Configure(void)
{
   // Enable the clock to the DCDC0 module
   SI32_CLKCTRL_A_enable_apb_to_modules_0(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG0_DCDC0CEN_ENABLED_U32
      | SI32_CLKCTRL_A_APBCLKG0_LCD0CEN_ENABLED_U32);
   SI32_CLKCTRL_A_enable_apb_to_modules_1(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG1_MISC0CEN_ENABLED_U32);

   // Configure the DCDC0 module
   SI32_LCD_A_enable_dcdc_bias(SI32_LCD_0);
   SI32_DCDC_A_enable_module(SI32_DCDC_0);

   #ifdef DCDC_gt15mA
      SI32_DCDC_A_select_power_switch_size(SI32_DCDC_0, 3);
      SI32_DCDC_A_enable_sync_mode(SI32_DCDC_0);
      SI32_DCDC_A_select_minimum_pulse_width(SI32_DCDC_0, 0);
   #endif

   #ifdef DCDC_5to15mA
      SI32_DCDC_A_select_power_switch_size(SI32_DCDC_0, 0);
      SI32_DCDC_A_enable_sync_mode(SI32_DCDC_0);
      SI32_DCDC_A_select_minimum_pulse_width(SI32_DCDC_0, 0);
   #endif

   #ifdef DCDC_lt5mA
      SI32_DCDC_A_select_power_switch_size(SI32_DCDC_0, 0);
      SI32_DCDC_A_enable_async_mode(SI32_DCDC_0);
      SI32_DCDC_A_select_minimum_pulse_width(SI32_DCDC_0, 3);
   #endif

   SI32_DCDC_A_select_output_voltage(SI32_DCDC_0, 0x0C);
   SI32_DCDC_A_select_clock_source_local(SI32_DCDC_0);

   // Wait until output voltage reached programmed value
   while (!SI32_DCDC_A_is_output_low_flag_set(SI32_DCDC_0)) {};

   // Check that the output voltage hasn't exceed the programmed value
   while (SI32_DCDC_A_is_output_high_flag_set(SI32_DCDC_0)) {};

   SI32_LDO_A_select_digital_source_dcdc_output(SI32_LDO_0);
   SI32_LDO_A_select_memory_source_dcdc_output(SI32_LDO_0);
   SI32_LDO_A_select_analog_source_dcdc_output(SI32_LDO_0);
}

void RTC0_Configure(void)
{
   SI32_CLKCTRL_A_enable_apb_to_modules_1(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG1_MISC0CEN_ENABLED_U32);

   SI32_RTC_B_set_clock_source_lfo(SI32_RTC_0);

   #ifdef LFOSC16k
   #else
      SI32_RTC_B_disable_low_frequency_oscillator_output(SI32_RTC_0);
      SI32_RTC_B_disable_low_frequency_oscillator(SI32_RTC_0);
   #endif

   SI32_RTC_B_disable_clock_out(SI32_RTC_0);

   // Disable the RTC timer
   SI32_RTC_B_disable_rtc_timer(SI32_RTC_0);

   // Disable the clock to the RTC0 module
   SI32_CLKCTRL_A_disable_apb_to_modules_1(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG1_MISC0CEN_ENABLED_U32);
}

void PB_Configure(void)
{
   SI32_CLKCTRL_A_enable_apb_to_modules_0(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG0_PB0CEN_ENABLED_U32);

   // Enable the LED drivers (PB1.4, PB1.5, PB1.6, PB1.7)
   SI32_PBSTD_A_set_pins_push_pull_output(SI32_PBSTD_1, 0x000000F0);
   SI32_PBSTD_A_write_pins_high(SI32_PBSTD_1,0x00D0);

   // Turn on the PB1.5 LED
   SI32_PBSTD_A_write_pins_low(SI32_PBSTD_1,0x0020);

   // PB1.0 and PB1.1 are connected to the EBID processor and are being pulled
   // low when the MCU Card is disconnected from the UDP Motherboard. Setting
   // these pins to analog mode disconnects the pins from the weak pull-ups,
   // which draw current.
   SI32_PBSTD_A_set_pins_analog(SI32_PBSTD_1, 0x00000003);

   #ifdef AHB_OUT_ENABLE
      SI32_PBSTD_A_set_pins_push_pull_output(SI32_PBSTD_0, 0x00000002);
      SI32_PBSTD_A_write_pbskipen(SI32_PBSTD_0, 0x00000001);
      SI32_PBCFG_A_enable_xbar0_peripherals(SI32_PBCFG_0,
         SI32_PBCFG_A_XBAR0_AHBEN);
   #endif

   SI32_PBCFG_A_enable_crossbar_0(SI32_PBCFG_0);
}

void PMU0_Configure(void)
{
   // Enable the clock to the PMU
   SI32_CLKCTRL_A_enable_apb_to_modules_1(SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG1_MISC0CEN_ENABLED_U32);

   SI32_CLKCTRL_A_disable_power_mode_8(SI32_CLKCTRL_0);

   SI32_PMU_A_clear_wakeup_flags(SI32_PMU_0);

   // Disable all resets
   SI32_RSTSRC_A_initialize(SI32_RSTSRC_0, 0x00000000);
}

//}}Config

//-----------------------------------------------------------------------------
// Subroutines
//-----------------------------------------------------------------------------

//{{App

#if   defined ( __CC_ARM ) // Keil/ARM

__SI32_RRAM void PowerModes_2_or_6(void)

#elif defined ( __GNUC__ ) // GCC

__attribute__ ((__section__(".data.pm2_or_6")))
void PowerModes_2_or_6(void)

#elif defined (__ICCARM__) // IAR

#endif
{

   // 1. Enable the clocks to peripherals that will be configured by firmware.

   SI32_CLKCTRL_A_enable_apb_to_all_modules(SI32_CLKCTRL_0);

   // 2. Select the desired clock source (LPOSC0, PLL0, etc.) and speed for
   // both AHB and APB clocks.
   // (see Clocks_Configure())

   // 3. Select the desired adaptive voltage scaling settings using the LDO
   // module.

   #ifdef AVS_ON
      LDO0_Configure();
   #endif

   #ifdef DCDC_ON
      DCDC0_Configure();
   #endif

   // 4. Disable Power Mode 8 mode in the Clock Control module.

   SI32_CLKCTRL_A_disable_power_mode_8(SI32_CLKCTRL_0);

   // 5. (Optional) Disable the retention mode of any enabled RAM banks.

   // 6. Set the pins in the lowest power configuration for this mode.
   SI32_PBSTD_A_write_pins_high(SI32_PBSTD_1, 0x00F0);

   // 7. Set up the wakeup interrupt source (including priority).

   // Enable PB1.5 as low port match source
   SI32_PBSTD_A_write_pm(SI32_PBSTD_0, 0x00000FFF);
   SI32_PBSTD_A_write_pmen(SI32_PBSTD_0, 0x00000000);
   SI32_PBSTD_A_write_pm(SI32_PBSTD_1, 0x00000FDF);
   SI32_PBSTD_A_write_pmen(SI32_PBSTD_1, 0x00000020);
   SI32_PBSTD_A_write_pm(SI32_PBSTD_2, 0x000000FF);
   SI32_PBSTD_A_write_pmen(SI32_PBSTD_2, 0x00000000);
   SI32_PBSTD_A_write_pm(SI32_PBSTD_3, 0x0000FFFF);
   SI32_PBSTD_A_write_pmen(SI32_PBSTD_3, 0x00000000);

   SI32_PBCFG_A_enable_port_match_interrupt(SI32_PBCFG_0);

   // Enable Interrupts
   NVIC_ClearPendingIRQ(PMATCH_IRQn);
   NVIC_EnableIRQ(PMATCH_IRQn);

   // 8. Disable the SysTick timer to prevent these interrupts from waking the core.

   SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;

   // 9. Disable all unused peripherals.

   // 10. Disable the clocks to all unused peripherals.

   #ifdef PERIPH_CLOCKS_OFF
      SI32_CLKCTRL_A_disable_apb_to_all_modules(SI32_CLKCTRL_0);

      // The PB module is re-enabled here so that port match
      // can wake the device
      SI32_CLKCTRL_A_enable_apb_to_modules_0(SI32_CLKCTRL_0,
         SI32_CLKCTRL_A_APBCLKG0_PB0);
   #endif

   // 11. Jump to code in RAM.
   // 12. Execute the DSB (Data Synchronization Barrier), ISB (Instruction
   // Synchronization Barrier), and WFI (Wait for Interrupt) or WFE (Wait for
   // Event) instructions. For SiM3L1xx devices, WFI and WFE have the same
   // behavior.

   __DSB();
   __ISB();
   __WFI();

   // Wake from PM2 or PM6

   #ifdef PERIPH_CLOCKS_OFF
      SI32_CLKCTRL_A_enable_apb_to_all_modules(SI32_CLKCTRL_0);
   #endif

   SI32_PBSTD_A_write_pins_low(SI32_PBSTD_1, 0x0080);
}

//}}App

//-----------------------------------------------------------------------------
// Interrupt Service Routines (ISRs)
//-----------------------------------------------------------------------------

// SysTick is disabled in this example, so this handler doesn't need to do
// anything.
void SysTick_Handler(void)
{
}

//{{ISR

void PMATCH_IRQHandler(void)
{
   // Enable APB clocks to everything
   SI32_CLKCTRL_A_enable_apb_to_all_modules(SI32_CLKCTRL_0);

   NVIC_DisableIRQ(PMATCH_IRQn);
}

//}}ISR

//-eof-------------------------------------------------------------------------
