// Copyright (c) 2012

//------------------------------------------------------------------------------
// Include Files
//------------------------------------------------------------------------------

#include <si32_device.h>
#include <SI32_PCA_A_Type.h>
#include <SI32_PCACH_A_Type.h>
#include <SI32_PBSTD_A_Type.h>
#include <SI32_PBCFG_A_Type.h>
#include <SI32_TIMER_A_Type.h>

#include "LED_control.h"

//------------------------------------------------------------------------------
// Definitions
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Global Variables
//------------------------------------------------------------------------------

Class_D_LED_Actions_Enum_Type LED_direction = BRIGHTEN;
int32_t LED_index = 0;
Class_D_LED_Enum_Type LED_to_use;

// The equation for Edge Aligned PWM Duty Cycle is:
// ((LIMIT * 2) - CCAPV)/(LIMIT * 2)
// These defines set the CCAPV values

uint32_t percent_array[NUM_LED_STEPS] = {      LED_OFF,
                                        LED_10_PERCENT,
                                        LED_20_PERCENT,
                                        LED_30_PERCENT,
                                        LED_40_PERCENT,
                                        LED_50_PERCENT,
                                        LED_60_PERCENT,
                                        LED_70_PERCENT,
                                        LED_80_PERCENT,
                                        LED_90_PERCENT,
                                        LED_100_PERCENT};

uint32_t LED_masks[6] = {0x00000001,  // LED on PB0.1
                         0x00000003,  // LED on PB0.2
                         0x0000000F,  // LED on PB0.4
                         0x0000001F,  // LED on PB0.5
                         0x000003FF,  // LED on PB0.10
                         0x000007FF}; // LED on PB0.11

#define LED0_PIN    0x00000002  // LED on PB0.1
#define LED1_PIN    0x00000004  // LED on PB0.2
#define LED2_PIN    0x00000010  // LED on PB0.4
#define LED3_PIN    0x00000020  // LED on PB0.5
#define LED4_PIN    0x00000400  // LED on PB0.10
#define LED5_PIN    0x00000800  // LED on PB0.11

uint32_t LED_pins[6] = {LED0_PIN,
                        LED1_PIN,
                        LED2_PIN,
                        LED3_PIN,
                        LED4_PIN,
                        LED5_PIN};

bool turn_on_red_led = 0;

//------------------------------------------------------------------------------
// Support Functions
//------------------------------------------------------------------------------

// This function is used to keep an LED on or off once it's conrolled by the PCA
// so the PCA can be multiplexed to another pin.  This is useful for things like
// the volume control, when multiple LEDs are on at once.
void LED_Manual_Control(Class_D_LED_Enum_Type LED, bool status)
{
   // Disable the PCA on the crossbar
   SI32_PBCFG_A_disable_xbar0_signal(SI32_PBCFG_0, SI32_XBAR0_PCA0_CEX0);

   if (status == ON)
   {
      // Turn on the LED
      SI32_PBSTD_A_write_pins_low(SI32_PBSTD_0, LED_pins[LED]);
   }
   else
   {
      // Turn off the LED
      SI32_PBSTD_A_write_pins_high(SI32_PBSTD_0, LED_pins[LED]);
   }
}

void LED_All_Off(void)
{
   // Disable the PCA on the crossbar
   SI32_PBCFG_A_disable_xbar0_signal(SI32_PBCFG_0, SI32_XBAR0_PCA0_CEX0);

   SI32_PBSTD_A_write_pins_high(SI32_PBSTD_0,
        LED0_PIN
        | LED1_PIN
        | LED2_PIN
        | LED3_PIN
        | LED4_PIN
        | LED5_PIN);
}

//------------------------------------------------------------------------------
//
//
//------------------------------------------------------------------------------
void LED_PCA_Control(Class_D_LED_Enum_Type LED, uint32_t brightness)
{
   // Set PBSTD_0 skip values for the LEDs
   SI32_PBSTD_A_write_pbskipen(SI32_PBSTD_0, LED_masks[LED]);
   SI32_PBCFG_A_enable_xbar0_signal(SI32_PBCFG_0, SI32_XBAR0_PCA0_CEX0);

   // Set the LED brightness
   SI32_PCACH_A_write_ccapvupd(SI32_PCA_0_CH0, brightness);
}

void LED_On(Class_D_LED_Enum_Type LED)
{
   LED_index = 0;
   LED_to_use = LED;
   LED_direction = BRIGHTEN;

   SI32_TIMER_A_clear_high_overflow_interrupt(SI32_TIMER_1);
   SI32_TIMER_A_start_high_timer(SI32_TIMER_1);
}

void LED_Off(Class_D_LED_Enum_Type LED)
{
   LED_index = NUM_LED_STEPS - 1;
   LED_to_use = LED;
   LED_direction = DARKEN;

   SI32_TIMER_A_clear_high_overflow_interrupt(SI32_TIMER_1);
   SI32_TIMER_A_start_high_timer(SI32_TIMER_1);
}

void Switch_LEDs(Class_D_LED_Enum_Type starting_LED)
{
   LED_index = NUM_LED_STEPS - 1;
   LED_to_use = starting_LED;
   LED_direction = DARKEN_THEN_BRIGHTEN_NEXT;

   SI32_TIMER_A_clear_high_overflow_interrupt(SI32_TIMER_1);
   SI32_TIMER_A_start_high_timer(SI32_TIMER_1);
}

//-eof--------------------------------------------------------------------------
