// Copyright (c) 2012

//------------------------------------------------------------------------------
// Include Files
//------------------------------------------------------------------------------

#include <si32_device.h>
#include <SI32_SARADC_A_Type.h>
#include <SI32_EPCA_A_Type.h>
#include <SI32_EPCACH_A_Type.h>
#include <SI32_FLASHCTRL_A_Type.h>
#include <SI32_PBHD_A_Type.h>
#include <SI32_PBSTD_A_Type.h>
#include <SI32_TIMER_A_Type.h>

#include "class_d.h"
#include "myCAPSENSE0.h"
#include "LED_Control.h"
#include "mulaw.h"
#include "myDataPlane.h"
#include "volume.h"
#include "pre_recorded_array.h"

#include <si32McuComponent.h>

//------------------------------------------------------------------------------
// Definitions
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Global Variables
//------------------------------------------------------------------------------

Class_D_Mode_Enum_Type system_mode = PLAY_JACK_INPUT;

uint32_t num_samples_recorded = 0;
uint32_t num_samples_played = 0;
bool playing = 0;
bool recording = 0;
bool erase_flag = 0;
bool clear_class_d_variables = 0;
uint32_t address;
uint8_t const recorded_array[RECORD_ARRAY_SIZE] __attribute__ ((aligned (1024))) = { 0xFF };

int32_t  rch_pv;            // right channel present value
int32_t  lch_pv;            // left channel present value
int16_t  rch_pv_epca;       // right channel present value
int16_t  lch_pv_epca;       // left channel present value
uint32_t adc0_data;         // adc0 output
uint32_t adc1_data;         // adc1 output
int32_t  rch_last = 0;      // previous value of right channel PDM
int32_t  lch_last = 0;      // previous value of left channel PDM
int32_t  rch_dcoffset = 0;  // right channel DC offset
int32_t  lch_dcoffset = 0;  // left channel DC offset
int32_t  rch_dcoff_err = 0; // right channel DC offset error (running total)
int32_t  lch_dcoff_err = 0; // left channel DC offset error (running total)
int16_t  tmp_int16;
int8_t   compressed_data;
bool flash_toggle = 0;
uint16_t flash_data;

// USB variables
int16_t rch_usb = 0;
int16_t lch_usb = 0;
int16_t rch_usb_epca = 0;
int16_t lch_usb_epca = 0;
int16_t mic_adc_input;
bool usb_buffer_ready = 0;
uint32_t num_usb_bytes = 0;
uint32_t usb_out_data_counter = 0;
uint32_t usb_in_data_counter = 0;
uint32_t sent_in_data_counter = 0;
bool usb_recording = 0;
bool recording_started = 0;
bool usb_transmitting = 0;

// Used by the dithering macro
int16_t input_lsb;
uint16_t random_counter = 0;
uint32_t random_value;
uint32_t random_reference = 2463534242; // using Marsaglia's original 'magic'
                                        // numbers

//------------------------------------------------------------------------------
// Global Non-Extern Variables
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Data Section
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Local Prototypes
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// ADC_Reset
//------------------------------------------------------------------------------
// This function resets the ADC and clears out the FIFOs when switching between
// modes.
void ADC_Reset(void)
{
   // Wait until any in-progress conversions complete
   if (SI32_SARADC_A_is_busy(SI32_SARADC_0))
   {
     while (!SI32_SARADC_A_is_single_conversion_complete_interrupt_pending(SI32_SARADC_0));
   }
   if (SI32_SARADC_A_is_busy(SI32_SARADC_1))
   {
     while (!SI32_SARADC_A_is_single_conversion_complete_interrupt_pending(SI32_SARADC_1));
   }

   // Clear the conversion complete flags
   SI32_SARADC_A_clear_single_conversion_complete_interrupt(SI32_SARADC_0);
   SI32_SARADC_A_clear_single_conversion_complete_interrupt(SI32_SARADC_1);

   // Clear the ADC0 data output if previous data exists
   while (SI32_SARADC_A_get_num_words_in_fifo(SI32_SARADC_0) > 0)
   {
      SI32_SARADC_A_read_data(SI32_SARADC_0); // Dummy fetch from ADC0
   }

   // Clear the ADC1 data output if previous data exists
   while (SI32_SARADC_A_get_num_words_in_fifo(SI32_SARADC_1) > 0)
   {
      SI32_SARADC_A_read_data(SI32_SARADC_1); // Dummy fetch from ADC1
   }
}

//------------------------------------------------------------------------------
// myClassD_updateMode
//------------------------------------------------------------------------------
// This function handles any configuration changes when switching between modes.
void myClassD_updateMode(void)
{
   // Reset all the variables keeping track of previous data
   rch_last = 0;
   lch_last = 0;
   rch_dcoffset = 0;
   lch_dcoffset = 0;
   rch_dcoff_err = 0;
   lch_dcoff_err = 0;
   rch_pv = 0;
   lch_pv = 0;

   switch (system_mode)
   {
      case PLAY_JACK_INPUT:
        system_mode = PLAY_USB_INPUT;
        break;
      case PLAY_USB_INPUT:
        system_mode = PLAY_PRERECORDED_FLASH;
        break;
      case PLAY_PRERECORDED_FLASH:
        system_mode = RECORD_PLAY_MIC_INPUT;
        break;
      case RECORD_PLAY_MIC_INPUT:
        system_mode = PLAY_JACK_INPUT;
        break;
      default:
        system_mode = PLAY_JACK_INPUT;
        break;
   }

   // Stopping, resetting, and restarting the timer in each mode gives the ADC time
   // to track a new input, if the input switched.
   //
   // Clearing and re-enabling the interrupts ensures each mode switches cleanly.
   //
   // These mode switches don't touch the USB interrupt, since that should always
   // trigger as fast as possible to respond to the host.
   if (system_mode == PLAY_JACK_INPUT)
   {
      SI32_SARADC_A_select_timeslot0_channel(SI32_SARADC_0, 14);

      // Set the overflow rate for 46.875 kHz
      SI32_TIMER_A_set_low_count(SI32_TIMER_1, 64512);
      SI32_TIMER_A_set_low_reload(SI32_TIMER_1, 64512);

      ADC_Reset();

      SI32_TIMER_A_start_low_timer(SI32_TIMER_1);

      NVIC_ClearPendingIRQ(CAPSENSE0_IRQn);
      NVIC_ClearPendingIRQ(TIMER0H_IRQn);
      NVIC_ClearPendingIRQ(TIMER1L_IRQn);

      NVIC_EnableIRQ(CAPSENSE0_IRQn);
      NVIC_EnableIRQ(TIMER0H_IRQn);
      NVIC_EnableIRQ(TIMER1L_IRQn);
   }
   else if (system_mode == PLAY_USB_INPUT)
   {
      SI32_SARADC_A_select_timeslot0_channel(SI32_SARADC_0, 8);

      // Set TIMER1 up for 48 kHz sampling
      SI32_TIMER_A_set_low_reload(SI32_TIMER_1, 64536);
      SI32_TIMER_A_set_low_count(SI32_TIMER_1, 64536);

      ADC_Reset();

      // Reset all the mode flags
      _receive0 = NO;
      _receive1 = NO;
      _receive2 = NO;

      _transmit0 = NO;
      _transmit1 = NO;
      _transmit2 = NO;

      usb_out_data_counter = 0;
      usb_in_data_counter = 0;

      SI32_TIMER_A_start_low_timer(SI32_TIMER_1);

      NVIC_ClearPendingIRQ(CAPSENSE0_IRQn);
      NVIC_ClearPendingIRQ(TIMER0H_IRQn);
      NVIC_ClearPendingIRQ(TIMER1L_IRQn);

      NVIC_EnableIRQ(CAPSENSE0_IRQn);
      NVIC_EnableIRQ(TIMER0H_IRQn);
      NVIC_EnableIRQ(TIMER1L_IRQn);
   }
   else if (system_mode == PLAY_PRERECORDED_FLASH)
   {
      // No ADC data needed in this mode, since all the values
      // are pulled from Flash.

      // Set TIMER1 up for 9600 Hz sampling
      SI32_TIMER_A_set_low_reload(SI32_TIMER_1, 60536);
      SI32_TIMER_A_set_low_count(SI32_TIMER_1, 60536);

      // Reset all the mode flags
      num_samples_played = 0;
      playing = 0;

      // TIMER1 is started in this mode when the PLAY button
      // is pressed

      NVIC_ClearPendingIRQ(CAPSENSE0_IRQn);
      NVIC_ClearPendingIRQ(TIMER0H_IRQn);
      NVIC_ClearPendingIRQ(TIMER1L_IRQn);

      NVIC_EnableIRQ(CAPSENSE0_IRQn);
      NVIC_EnableIRQ(TIMER0H_IRQn);
      NVIC_EnableIRQ(TIMER1L_IRQn);
   }
   else if (system_mode == RECORD_PLAY_MIC_INPUT)
   {
      SI32_SARADC_A_select_timeslot0_channel(SI32_SARADC_0, 8);

      // Set TIMER1 up for 9600 Hz sampling
      SI32_TIMER_A_set_low_reload(SI32_TIMER_1, 60536);
      SI32_TIMER_A_set_low_count(SI32_TIMER_1, 60536);

      ADC_Reset();

      // Reset all the mode flags
      recording = 0;
      num_samples_played = 0;
      playing = 0;
      flash_toggle = 0;

      // TIMER1 is started in this mode when the RECORD/PLAY buttons
      // are pressed

      NVIC_ClearPendingIRQ(CAPSENSE0_IRQn);
      NVIC_ClearPendingIRQ(TIMER0H_IRQn);
      NVIC_ClearPendingIRQ(TIMER1L_IRQn);

      NVIC_EnableIRQ(CAPSENSE0_IRQn);
      NVIC_EnableIRQ(TIMER0H_IRQn);
      NVIC_EnableIRQ(TIMER1L_IRQn);
   }
}

//------------------------------------------------------------------------------
// class_d
//------------------------------------------------------------------------------
// This is the main class_d loop that runs when no system interrupts are
// executing.  This main loop handles a few small functions, from kicking off the
// USB transmission during USB recording and erasing the Flash during Play/Record
// Flash Mode.
void class_d (void)
{
  uint32_t address;
  si32UsbEndpointObject *ep;

  // Start the EPCA timer/counter
  SI32_EPCA_A_start_counter_timer(SI32_EPCA_0);

  while(1)
  {
      // PLAY_JACK_INPUT mode -------------------------------------------------
      if (system_mode == PLAY_JACK_INPUT)
      {

      }
      // PLAY_USB_INPUT mode --------------------------------------------------
      if (system_mode == PLAY_USB_INPUT)
      {
         // Start up the transmission process when two buffers are full and
         // ready to go to the host.  Once this process starts, the DataPlaneIn
         // si32Library callback functions handle the rest.
         if ((_transmit0 == YES)
             && (recording_started == 1))
         {
            recording_started = 0;
            usb_transmitting = 1;

            ep = myDataPlaneInEp;

            if (ep != NULL)
            {
               si32UsbEndpointObject_transmit(ep,
                                              &usb_in_buffer[0],
                                              USB_IN_BUFFER_SIZE,
                                              tx0_callback,
                                              NULL,
                                              NULL);
            }
         }
      }
      // PLAY_PRERECORDED_FLASH mode ------------------------------------------
      if (system_mode == PLAY_PRERECORDED_FLASH)
      {

      }
      // RECORD_PLAY_MIC_INPUT mode -------------------------------------------
      if (system_mode == RECORD_PLAY_MIC_INPUT)
      {
         // Erase the Flash if the button is pressed.
         // This function should occur here since no interrupts should be going
         // on at the time.
         if (erase_flag == 1)
         {
            NVIC_DisableIRQ(TIMER0H_IRQn);
            NVIC_DisableIRQ(CAPSENSE0_IRQn);
            NVIC_DisableIRQ(TIMER1L_IRQn);

            erase_Flash();

            erase_flag = 0;
            recording = 0;
            playing = 0;

            LED_Off(LED4);

            NVIC_ClearPendingIRQ(TIMER0H_IRQn);
            NVIC_EnableIRQ(TIMER0H_IRQn);

            NVIC_ClearPendingIRQ(CAPSENSE0_IRQn);
            NVIC_EnableIRQ(CAPSENSE0_IRQn);

            NVIC_ClearPendingIRQ(TIMER1L_IRQn);
            NVIC_EnableIRQ(TIMER1L_IRQn);
         }
      }
      // ----------------------------------------------------------------------
   }
}

//-eof-------------------------------------------------------------------------
