// Copyright (c) 2012

#include <stdbool.h>
#include <si32WideTypes.h>
#include <si32McuComponent.h>

//------------------------------------------------------------------------------
// Global Defines
//------------------------------------------------------------------------------

#if (P32_RELEASE == 1)

// Approximate Code size (bytes): 47200
// Pre-Recorded Flash array size (bytes): 66508
// Flash array size (bytes): 128000
//
// The Flash array size must be specified in terms of 1024-byte pages

#define RECORD_ARRAY_SIZE           125*1024

#elif (P32_DEBUG == 1)

// Approximate Code size (bytes): 68200
// Pre-Recorded Flash array size (bytes): 66508
// Flash array size (bytes): 81920
//
// The Flash array size must be specified in terms of 1024-byte pages

#define RECORD_ARRAY_SIZE           80*1024

#elif (UVISION_RELEASE == 1)

// Approximate Code size (bytes): 34700
// Pre-Recorded Flash array size (bytes): 66508
// Flash array size (bytes): 143360
//
// The Flash array size must be specified in terms of 1024-byte pages

#define RECORD_ARRAY_SIZE           140*1024

#elif (UVISION_DEBUG == 1)

// Approximate Code size (bytes): 40200
// Pre-Recorded Flash array size (bytes): 66508
// Flash array size (bytes): 81920
//
// The Flash array size must be specified in terms of 1024-byte pages

#define RECORD_ARRAY_SIZE           80*1024

#endif

#define RECORD_ARRAY_BYTE_SIZE      RECORD_ARRAY_SIZE

#define ACZERO_12B                  2048         // 4096/2 is best-case ac zero
#define THHOLD_DCOFF_FIX            400000       // DC offset rate of adaptation

//------------------------------------------------------------------------------
// Global Variables
//------------------------------------------------------------------------------

typedef enum Class_D_Mode_Enum
{
    PLAY_JACK_INPUT = 0x00,
    PLAY_USB_INPUT = 0x01,
    PLAY_PRERECORDED_FLASH = 0x02,
    RECORD_PLAY_MIC_INPUT = 0x03
}
Class_D_Mode_Enum_Type;

extern bool system_mode_updated;
extern Class_D_Mode_Enum_Type system_mode;

extern bool playing;
extern bool recording;
extern bool erase_flag;
extern uint32_t num_samples_played;
extern uint32_t num_pr_samples_played;
extern uint32_t num_samples_recorded;

extern bool clear_class_d_variables;
extern int32_t rch_pv;
extern int32_t lch_pv;
extern int16_t rch_pv_epca;        // right channel present value
extern int16_t lch_pv_epca;        // left channel present value
extern uint32_t adc0_data;         // adc0 output
extern uint32_t adc1_data;         // adc1 output
extern uint32_t tmp_uint32;        // temporary var type uint32
extern uint32_t random;            // random bits for dither
extern uint32_t random_cnt;        // usage of random bits
extern uint32_t rch_random;        // random bits for right dither
extern uint32_t lch_random;        // random bits for left dither
extern int32_t  rch_last;          // previous value of right channel PDM
extern int32_t  lch_last;          // previous value of left channel PDM
extern int32_t  rch_dcoffset;      // right channel DC offset
extern int32_t  lch_dcoffset;      // left channel DC offset
extern int32_t  rch_dcoff_err;     // right channel DC offset error (running total)
extern int32_t  lch_dcoff_err;     // left channel DC offset error (running total)
extern int16_t  tmp_int16;
extern int8_t   compressed_data;
extern bool flash_toggle;
extern uint16_t flash_data;

// USB variables
extern int16_t rch_usb;
extern int16_t lch_usb;
extern int16_t rch_usb_epca;
extern int16_t lch_usb_epca;
extern int16_t mic_adc_input;
extern uint32_t usb_out_data_counter;
extern uint32_t usb_in_data_counter;
extern uint32_t sent_in_data_counter;
extern bool usb_recording;
extern bool recording_started;
extern bool usb_transmitting;

// Used by the dithering macro
extern int16_t input_lsb;
extern uint16_t random_counter;
extern uint32_t random_value;
extern uint32_t random_reference;

extern uint32_t address;
extern uint8_t const recorded_array[RECORD_ARRAY_SIZE];

//------------------------------------------------------------------------------
// Prototypes
//------------------------------------------------------------------------------

void myClassD_initialize(void);
void class_d (void);
void myClassD_updateMode(void);
uint32_t xorshift32rng(void);

//------------------------------------------------------------------------------
// Macros
//------------------------------------------------------------------------------

// Erase the recording Flash
#define erase_Flash() \
   if ((recorded_array[1] != 0xFFFF) || (recorded_array[100] != 0xFFFF)) \
   { \
      SI32_FLASHCTRL_A_enter_flash_erase_mode(SI32_FLASHCTRL_0); \
      SI32_FLASHCTRL_A_write_flash_key(SI32_FLASHCTRL_0, 0xA5); \
      SI32_FLASHCTRL_A_write_flash_key(SI32_FLASHCTRL_0, 0xF2); \
      for (address = (uint32_t) &recorded_array[0]; \
           address < ((uint32_t) &recorded_array[0] + RECORD_ARRAY_BYTE_SIZE); \
           address += 1024) \
       { \
        SI32_FLASHCTRL_A_write_wraddr(SI32_FLASHCTRL_0, address); \
        SI32_FLASHCTRL_A_write_wrdata(SI32_FLASHCTRL_0, (uint16_t) address); \
       } \
       SI32_FLASHCTRL_A_write_flash_key(SI32_FLASHCTRL_0, 0x5A); \
       SI32_FLASHCTRL_A_exit_flash_erase_mode(SI32_FLASHCTRL_0); \
       SI32_FLASHCTRL_A_write_wraddr(SI32_FLASHCTRL_0, (uint32_t) &recorded_array[0]); \
       num_samples_recorded = 0; \
   }

// Dither macro for Class-D output data
//
// This algorithm recaptures some of the energy lost when translating to 9 bits from
// 12 (ADC) or 16 (USB), but in a random way to ensure it's inaudible.
#define dither(input_data, shift_bits, mask, output_data) \
   if (random_counter == 0) \
   { \
      create_random_value(random_value); \
      random_counter = 4; \
   } \
   else \
   { \
      random_counter--; \
   } \
   random_value = random_value & mask; \
   random_value = random_value >> shift_bits; \
   input_lsb = input_data & mask; \
   output_data = (int16_t) input_data >> shift_bits; \
   if (output_data >= 0) \
   { \
      if (input_lsb > random_value) \
      { \
         output_data++; \
         \
         if (output_data > 255) \
         { \
            output_data = 255; \
         } \
      } \
   } \
   else \
   { \
      if (input_lsb < random_value) \
      { \
         output_data--; \
         \
         if (output_data < -256) \
         { \
            output_data = -256; \
         } \
      } \
   }

//------------------------------------------------------------------------------
// create_random_value
//
// G. Marsaglia's "xorshift RNG" algorithm, used to generate
// "good enough" random numbers for dithering audio LSBs
// (c.f. http://www.jstatsoft.org/v08/i14/paper)
//------------------------------------------------------------------------------
#define create_random_value(input) \
   random_reference ^= (random_reference << 13); \
   random_reference ^= (random_reference >> 17); \
   random_reference ^= (random_reference << 5); \
   input = random_reference;

//------------------------------------------------------------------------------
// End of File
//------------------------------------------------------------------------------
