// Copyright (c) 2012

// library
#include <stdio.h>
// hal
#include <si32_device.h>
#include <SI32_CAPSENSE_A_Type.h>
#include <SI32_TIMER_A_Type.h>
#include <SI32_PBSTD_A_Type.h>
// application
#include "myCAPSENSE0.h"
#include "class_d.h"
#include "LED_control.h"
#include "volume.h"

//------------------------------------------------------------------------------
// Global Defines
//------------------------------------------------------------------------------

#define NUM_CHANNELS 6
#define THRESHOLD    0x1500  // This is an arbitrary threshold that can be
                             // adjusted based on the desired performance
                             // The higher the threshold, the harder the press
                             // required to trigger a touch

//------------------------------------------------------------------------------
// Global Variables
//------------------------------------------------------------------------------

uint8_t CapsenseTouchPos;
bool CapsenseTouch = 0;
bool movement = 0;
bool touch_lost = 0;
uint32_t touch_lost_hysteresis = 0;

// CAPSENSE Calculation and Information Struct
struct capsensTracking_struct
{
  bool scanning;
  uint32_t chan;
  uint32_t channels[NUM_CHANNELS];
  uint32_t readings[NUM_CHANNELS];
  uint32_t baseline[NUM_CHANNELS];
  uint32_t values[NUM_CHANNELS];
} CS_info;

void calculate_position(uint8_t); //defined below

//==============================================================================
// 2nd Level Interrupt Handlers (Called from generated code)
//==============================================================================

//------------------------------------------------------------------------------
// This function is called by the TIMER0H ISR and kicks off a scan of the slider
// This function starts the conversion on the first channel.  The conversion and
// all other channels are measured using the CAPSENSE0_conv_complete_handler
void scan_slider(void)
{
   // Reset the channel pointer
   CS_info.chan = 0;

   // Connect to the first channel
   SI32_CAPSENSE_A_write_mux(SI32_CAPSENSE_0, CS_info.channels[CS_info.chan]);
   SI32_CAPSENSE_A_connect_capsense_channel(SI32_CAPSENSE_0);

   // Start a conversion
   SI32_CAPSENSE_A_enable_module(SI32_CAPSENSE_0);
   SI32_CAPSENSE_A_start_manual_conversion(SI32_CAPSENSE_0);
}

// This ISR scans through and measures each channel
void CAPSENSE0_conv_complete_handler(void)
{
   // Save the reading
   CS_info.readings[CS_info.chan] = SI32_CAPSENSE_A_read_data(SI32_CAPSENSE_0);

   // Clear the interrupt and disable the module.
   // This disable action is required between each Capacitive Sensing
   // measurement.
   SI32_CAPSENSE_A_clear_conversion_complete_interrupt(SI32_CAPSENSE_0);
   SI32_CAPSENSE_A_disable_module(SI32_CAPSENSE_0);

   CS_info.chan++;

   // If there are remaining channels to scan...
   if (CS_info.chan < NUM_CHANNELS)
   {
       // Point to the next channel...
       SI32_CAPSENSE_A_write_mux(SI32_CAPSENSE_0, CS_info.channels[CS_info.chan]);
       SI32_CAPSENSE_A_connect_capsense_channel(SI32_CAPSENSE_0);

       // ...and start a conversion
       SI32_CAPSENSE_A_enable_module(SI32_CAPSENSE_0);
       SI32_CAPSENSE_A_start_manual_conversion(SI32_CAPSENSE_0);

   }
   else
   {
      // Disconnect the pins
      SI32_CAPSENSE_A_disconnect_capsense_channel(SI32_CAPSENSE_0);

      // Calculate the position on the slider
      // The 4 is the scaling factor on the calculation: (0-19) positions
      calculate_position(4);
   }
}

// Once a scan is complete, this function determines if a touch was
// detected on the slider.
void calculate_position(uint8_t scale)
{
   uint32_t num_channels = sizeof(CS_info.readings) / 4;
   uint32_t channel;
   uint32_t position = 0;
   uint32_t accumulator = 0;
   uint32_t value;
   bool touch_detect = 0;
   uint32_t max_channel_value = 0;
   uint32_t max_channel = 0;
   static uint32_t starting_position;

   // For each CAPSENSE channel
   for (channel = 0; channel < num_channels; channel++)
   {
      if (CS_info.readings[channel] > CS_info.baseline[channel])
      {
         value = CS_info.readings[channel] - CS_info.baseline[channel];
         CS_info.values[channel] = value;

         // If the measured value is above the threshold, indicate
         // that a touch was detected and find the maximum channel
         if (value > THRESHOLD)
         {
            touch_detect = 1;

            if (value > max_channel_value)
            {
               max_channel = channel;
               max_channel_value = value;
            }
         }
      }
      else
      {
         CS_info.values[channel] = 0;
      }
   }

   // If a touch was detected, use the maximum channel and the two adjacent
   // channels to calculate the position on the slider.  All other channels
   // are ignored.
   // This prevents any channels from pulling the touch position down or up
   // because of a slightly higher than normal measurement.
   if (touch_detect == 1)
   {
      if (max_channel == 0)
      {
         position += CS_info.values[0];
         position += CS_info.values[1]*2;

         accumulator = CS_info.values[0] + CS_info.values[1];
      }
      else if (max_channel == (NUM_CHANNELS - 1))
      {
         position += CS_info.values[NUM_CHANNELS-1]*(NUM_CHANNELS);
         position += CS_info.values[NUM_CHANNELS-2]*(NUM_CHANNELS-1);

         accumulator = CS_info.values[NUM_CHANNELS-1] + CS_info.values[NUM_CHANNELS-2];
      }
      else
      {
         position += CS_info.values[max_channel]*(max_channel+1);
         position += CS_info.values[max_channel-1]*(max_channel);
         position += CS_info.values[max_channel+1]*(max_channel+2);

         accumulator = CS_info.values[max_channel]
                      + CS_info.values[max_channel-1]
                      + CS_info.values[max_channel+1];
      }

      // The scale factor is a parameter to the function and determines
      // how many virtual positions the physical slider channels equate to.
      CapsenseTouchPos = ((position*scale)/accumulator) - scale;

      // If we previously didn't have a touch, record the starting position.
      // This is used to determine if the touch is a button press or a slider
      // movement.
      if (CapsenseTouch == 0)
      {
         starting_position = CapsenseTouchPos;
      }

      // Allow a little bit of leeway on button presses
      //
      // The movement flag is only set during a touch to prevent
      // the algorithm from not detecting a slide when the finger slides
      // up and then back to the same starting position
      //
      // The movement flag is only reset when no touch is detected
      if ((starting_position != CapsenseTouchPos)
          && ((starting_position + 1) != CapsenseTouchPos)
          && ((starting_position - 1) != CapsenseTouchPos)
          && ((starting_position + 2) != CapsenseTouchPos)
          && ((starting_position - 2) != CapsenseTouchPos))
      {
         movement = 1;
      }

      CapsenseTouch = 1;

      // If a slide on the slider is detected, modify the volume
      // Otherwise, the touch is a button press and the volume isn't changed
      if (movement == 1)
      {
         // Wait for the USB to stop updating the volume control before
         // updating it here.
         while (volume_updating);

         volume_updating = 1;

         VolumeControl[0] = CapsenseTouchPos;
         VolumeControl[1] = CapsenseTouchPos;
         volume_control();

         volume_updating = 0;
      }

      // Reset these "no touch" variables
      touch_lost = 0;
      touch_lost_hysteresis = 0;
  }
  else
  {
     // We only analyze the touch once the touch is no longer detected
     // This means someone can keep their finger on a button for as long as
     // they like.  When they remove the finger, that's when the action
     // will take place.
     if (touch_lost == 1)
     {
         // If a touch was detected and it wasn't a volume update (i.e. slide)
         if ((CapsenseTouch == 1) && (volume_update == 0))
         {
            // Mode button pressed
            if ((CapsenseTouchPos == 0)
                || (CapsenseTouchPos == 1)
                || (CapsenseTouchPos == 2))
            {
               // Prevent the mode from switching when the USB recording interface
               // is active.
               if (usb_transmitting == 0)
               {
                  LED_All_Off();

                  // Stop all non-USB interrupts while the system mode updates
                  SI32_TIMER_A_stop_low_timer(SI32_TIMER_1);
                  NVIC_DisableIRQ(TIMER1L_IRQn);
                  NVIC_DisableIRQ(CAPSENSE0_IRQn);
                  NVIC_DisableIRQ(TIMER0H_IRQn);

                  switch (system_mode)
                  {
                    case PLAY_JACK_INPUT:
                       Switch_LEDs(LED0);
                       break;
                    case PLAY_USB_INPUT:
                       Switch_LEDs(LED1);
                       break;
                    case PLAY_PRERECORDED_FLASH:
                       Switch_LEDs(LED2);
                       break;
                    case RECORD_PLAY_MIC_INPUT:
                       Switch_LEDs(LED3);
                       break;
                    default:
                       LED_On(LED0);
                       break;
                  }

                  myClassD_updateMode();
               }
            }
            // Record button pressed
            else if ((CapsenseTouchPos == 15)
                     || (CapsenseTouchPos == 16))
            {
               // Only detect a Record press during a mode that supports
               // the record feature
               if (system_mode == RECORD_PLAY_MIC_INPUT)
               {
                  if (recording == 1)
                  {
                      SI32_TIMER_A_stop_low_timer(SI32_TIMER_1);
                      LED_Manual_Control(LED3, ON);
                      LED_Off(LED4);
                      recording = 0;
                  }
                  else if (playing == 0)
                  {
                      SI32_TIMER_A_set_low_count(SI32_TIMER_1, 60536);
                      SI32_TIMER_A_start_low_timer(SI32_TIMER_1);
                      LED_Manual_Control(LED3, ON);
                      LED_On(LED4);
                      recording = 1;
                  }
               }
            }
            // Play button pressed
            else if ((CapsenseTouchPos == 18)
                     || (CapsenseTouchPos == 19))
            {
               // Only detect a Play press during a mode that supports
               // the play feature
               if (system_mode == PLAY_PRERECORDED_FLASH)
               {
                  num_samples_played = 0;

                  if (playing == 1)
                  {
                     SI32_TIMER_A_stop_low_timer(SI32_TIMER_1);
                     LED_Manual_Control(LED2, ON);
                     LED_Off(LED5);
                     playing = 0;
                  }
                  else
                  {
                     SI32_TIMER_A_set_low_count(SI32_TIMER_1, 60536);
                     SI32_TIMER_A_start_low_timer(SI32_TIMER_1);
                     clear_class_d_variables = 1;
                     LED_Manual_Control(LED2, ON);
                     LED_On(LED5);
                     playing = 1;
                  }
               }
               else if (system_mode == RECORD_PLAY_MIC_INPUT)
               {
                  num_samples_played = 0;

                  if (playing == 1)
                  {
                     SI32_TIMER_A_stop_low_timer(SI32_TIMER_1);
                     LED_Manual_Control(LED3, ON);
                     LED_Off(LED5);
                     playing = 0;
                  }
                  else if (recording == 0)
                  {
                     SI32_TIMER_A_set_low_count(SI32_TIMER_1, 60536);
                     SI32_TIMER_A_start_low_timer(SI32_TIMER_1);
                     clear_class_d_variables = 1;
                     LED_Manual_Control(LED3, ON);
                     LED_On(LED5);
                     playing = 1;
                  }
               }
            }
            // Erase button pressed
            else if ((CapsenseTouchPos == 10)
                     || (CapsenseTouchPos == 9))
            {
               // Only detect an Erase press during a mode that supports
               // the play and record feature
               // The actual erase occurs in the class_d main code
               // outside of the ISR
               if (system_mode == RECORD_PLAY_MIC_INPUT)
               {
                  LED_Manual_Control(LED3, ON);
                  LED_On(LED4);
                  erase_flag = 1;
               }
            }
         }
         // If the volume was updated...
         // This can occur both from the USB interface and from the
         // CAPSENSE slider, which is why we're not checking if a touch
         // was detected here
         else if (volume_update == 1)
         {
            LED_All_Off();

            // After the volume update, turn on the appropriate mode
            // LEDs
            // During the record/playback modes, two LEDs must be
            // enabled while recording or playing
            switch (system_mode)
            {
               case PLAY_JACK_INPUT:
                  LED_On(LED0);
                  break;
               case PLAY_USB_INPUT:
                  LED_On(LED1);
                  break;
               case PLAY_PRERECORDED_FLASH:

                  if ((recording == 1)
                      || (playing == 1))
                  {
                     LED_Manual_Control(LED2, ON);

                     if (recording == 1)
                     {
                        LED_On(LED4);
                     }
                     else if (playing == 1)
                     {
                        LED_On(LED5);
                     }
                  }
                  else
                  {
                     LED_On(LED2);
                  }
                  break;
               case RECORD_PLAY_MIC_INPUT:
                  if ((recording == 1)
                      || (playing == 1))
                  {
                     LED_Manual_Control(LED3, ON);

                     if (recording == 1)
                     {
                        LED_On(LED4);
                     }
                     else if (playing == 1)
                     {
                        LED_On(LED5);
                     }
                  }
                  else
                  {
                     LED_On(LED3);
                  }
                  break;
               default:
                  LED_On(LED0);
                  break;
            }

            volume_update = 0;
         }

         touch_lost = 0;
         movement = 0;
         CapsenseTouch = 0;
      }
      else
      {
         if ((CapsenseTouch == 1)
             || (volume_update == 1))
         {
            touch_lost_hysteresis++;

            // Add extra touch loss hysteresis while in volume mode.  This is
            // because a loss of touch detect is really obvious since the LEDs
            // flicker constantly.  This hysteresis also adds an extra
            // feature of pausing on the volume level momentarily before
            // switching back to the mode LED.
            //
            // We don't want this extra hysteresis when using the buttons
            // since it adds noticeable lag to responsiveness of the board.
            if (volume_update == 1)
            {
               if (touch_lost_hysteresis == 30)
               {
                  touch_lost = 1;

                  touch_lost_hysteresis = 0;
               }
            }
            else
            {
               if (touch_lost_hysteresis == 2)
               {
                  touch_lost = 1;

                  touch_lost_hysteresis = 0;
               }
            }
         }
      }
   }
}


//------------------------------------------------------------------------------
void calibrate_capsense(void)
{
   uint32_t i;

   // Disable the CAPSENSE0 interrupts
   NVIC_DisableIRQ(CAPSENSE0_IRQn);

   // Initialize the channels in the appropriate order
   CS_info.chan = 0;
   CS_info.channels[0] = 15;
   CS_info.channels[1] = 14;
   CS_info.channels[2] = 13;
   CS_info.channels[3] = 12;
   CS_info.channels[4] = 11;
   CS_info.channels[5] = 10;

   // Create the initial baseline for all of the channels
   for (i=0; i < NUM_CHANNELS; i++)
   {
      // Connect to the channel and start the conversion
      SI32_CAPSENSE_A_write_mux(SI32_CAPSENSE_0, CS_info.channels[i]);
      SI32_CAPSENSE_A_connect_capsense_channel(SI32_CAPSENSE_0);
      SI32_CAPSENSE_A_enable_module(SI32_CAPSENSE_0);
      SI32_CAPSENSE_A_start_manual_conversion(SI32_CAPSENSE_0);

      // Wait until the conversion is done
      while(SI32_CAPSENSE_A_is_conversion_in_progress(SI32_CAPSENSE_0));
      SI32_CAPSENSE_A_disable_module(SI32_CAPSENSE_0);

      // Set the baseline for the channel with 0x100 as extra margin
      CS_info.baseline[i] = SI32_CAPSENSE_A_read_data(SI32_CAPSENSE_0) - 0x100;
   }
}

//-eof-------------------------------------------------------------------------
