// Copyright (c) 2012

// Title:
//
// Stereo Class-D Amplification Using SiM3U1xx HD I/O + HS differential PWM (EPCA)
//
// Introduction (by BM and TP 2012):
//
// Using the Si3MU1xx Class-D RD board, two pairs of EPCA channels are used
// to generate PWM waveforms for high-drive I/O connected as differential drivers
// to a pair of audio speakers. The SARADCs operate in 12-bit mode collecting stereo
// audio samples from a music player's headphone amplifer, USB, or data from Flash
// as source data for the amplifiers.
//
// The process uses frequency-dependent signal processing methods to deliver
// greater apparent signal resolution than seemingly possible given the specs for
// ADC and amplifier used in the system. The techniques are ideal for embedded
// applications where perceived sound quality is more important than performance to
// technical specifications.
//
// Operating at 50 MHz, the EPCA provides 10 ns timing resolution. At 48 MHz, 10.4 ns.
//
// 50 MHz / 4 = 12.5 MHz SAR clk, 12-bit samples @ 195 ksps.
//   Oversample at 4:1 to get 13 bit resolution samples @ 48.8 ksps.
// 48 MHz / 3 = 16 MHz SAR clk, 12-bit samples @ 250 ksps.
//
// Set PWM frequency high to simplify filtering, but low enough that the process
// is not a computational burden:
// 48 MHz / 256 counts/cycle = 187.5 kHz PWM cycle
// PWM symbol magnitude = 256 * 2 = 512 levels; 9 bits
// This has the advantage of being close to the maximum 12-bit ADC conversion
// rate.
//
// Because we can generate samples faster than they can be clocked out at full
// resolution, a variation on pulse-density modulation (PDM) methods are used
// to post-process the ADC data. As with traditional PDM, we decompose output
// data into smaller blocks of high-frequency PWM data. This pushes switching
// noise far outside of the audio pass band to simplify output filtering. The
// method used here performs exponential weighting of most-recent-arrived data
// to give better tracking of fast-changing inputs while allowing accumulation
// to higher resolutions in the heart of the pass-band. Resulting distortion in
// the frequency domain is an acceptable trade-off in the target applications.
//
// Each speaker is driven by a pair of EPCA channels. Each channel is connected to
// a different high-drive pad, both operating in push-pull mode. The two pads are
// connected to the two sides of an audio speaker through a pair of coils and caps
// creating low-pass filters for suppression of switching noise at-and-above the
// PWM frequency of ~190 kHz (corner frequency approx 10 kHz).
//
// The board's mode of operations are:
// 1) Stereo Jack Input Mode
// 2) USB Input/Output Mode
// 3) Play Pre-recorded Flash Mode
// 4) Play/Record Flash Mode
// See the Class-D RD User's Guide for a detailed explanation of each mode.
//
// Volume is set by changing power settings for the high-drive pins. This exploits a
// novel feature of SiM3U1xx high-drive pins to provide attenuation while maintaining
// full PWM fidelity. Unlike most Class-D systems, fidelity remains constant with
// decreasing output power. As the noise floor drops to sub-audible levels, 'perceived'
// fidelity actually increases at lower volume levels.
//
// Power settings for the pin drivers are calculated based on the true output strength
// of the drivers. The source/sink drive strength ratio is asymmetic because, even
// though source and sink drivers have similar range settings, the maximum source
// drive current is half of the maximum available sink drive current.
// At lower volumes, symmetric drive strength is provided by selecting source drive
// settings that are double the sink drive settings.
// For higher volumes (above mid range), the source drive strength is set to maximum
// and volume is controlled by adjustment of the sink drive strength. For the intended
// application, this asymmetric drive strength is mostly inauduble.
//
// At zero volume, the output drivers are turned off.
//
// See the User's Guide for more information on how to use the Class-D RD board
//
//

#include "myTIMER1.h"
#include <si32_device.h>
#include <SI32_TIMER_A_Type.h>
#include <SI32_PBSTD_A_Type.h>
#include <SI32_EPCA_A_Type.h>
#include <SI32_EPCACH_A_Type.h>
#include <SI32_FLASHCTRL_A_Type.h>
#include <SI32_SARADC_A_Type.h>

#include "class_d.h"
#include "mulaw.h"
#include "myCAPSENSE0.h"
#include "LED_Control.h"
#include "myDataPlane.h"
#include "pre_recorded_array.h"
#include <si32McuComponent.h>

//==============================================================================
// 2nd Level Interrupt Handlers (Called from generated code)
//==============================================================================

// This ISR is the "heartbeat" of the system and handles almost all of the
// Class-D algorithms and functionality.  This Timer will overflow at different
// rates depending on the mode of the system.
void TIMER1L_low_overflow_handler(void)
{
   SI32_TIMER_A_clear_low_overflow_interrupt(SI32_TIMER_1);

   #if ((P32_DEBUG == 1) || (UVISION_DEBUG == 1))

   SI32_PBSTD_A_write_pins_high(SI32_PBSTD_3, 0x0001);

   #endif

   if (system_mode == PLAY_JACK_INPUT)
   {
      // Get right channel ADC output

      // Wait for conversion to complete
      while (!SI32_SARADC_A_is_single_conversion_complete_interrupt_pending(SI32_SARADC_0));

      // Clear the conversion complete interrupt flag
      SI32_SARADC_A_clear_single_conversion_complete_interrupt(SI32_SARADC_0);

      // Fetch the ADC0 results
      adc0_data = SI32_SARADC_A_read_data(SI32_SARADC_0);

      // Then get left channel ADC output

      // Wait for conversion to complete
      while (!SI32_SARADC_A_is_single_conversion_complete_interrupt_pending(SI32_SARADC_1));

      // Clear the conversion complete interrupt flag
      SI32_SARADC_A_clear_single_conversion_complete_interrupt(SI32_SARADC_1);

      // Fetch the ADC0 results
      adc1_data = SI32_SARADC_A_read_data(SI32_SARADC_1);

      // Turn on the red LED (DS8) if an ADC saturation is close on either channel
      // The LED is turned on and off by the Capsense scan interrupt (TIMER0H)
      if ((adc0_data >= 0x0FFA)
          || (adc0_data <= 0x002)
          || (adc1_data >= 0x0FFA)
          || (adc1_data <= 0x0002))
      {
         turn_on_red_led = 1;
      }

      // Convert to signed numbers in the +/- 12-bit range
      rch_pv = (int32_t) (adc0_data - ACZERO_12B);
      lch_pv = (int32_t) (adc1_data - ACZERO_12B);

      // DC Offset algorithm
      // This offset algorithm corrects the DC offset of the input to VREF/2, since the output of
      // an mp3 player or phone is guaranteed power, but not guaranteed voltage or offset. However,
      // the Class-D board biases all input signals to VREF/2, so this algorithm is no longer needed.

      /* // Apply dc offsets
      rch_pv = rch_pv - rch_dcoffset;
      lch_pv = lch_pv - lch_dcoffset;

      // Sum DC offset error (running total)
      rch_dcoff_err = rch_dcoff_err + rch_pv;
      lch_dcoff_err = lch_dcoff_err + lch_pv;

      // Adjust dc offset if necessary
      if (rch_dcoff_err > THHOLD_DCOFF_FIX)
      {
        rch_dcoffset++;
        rch_dcoff_err = 0;
      }
      else
      {
         if (rch_dcoff_err < -THHOLD_DCOFF_FIX)
         {
            rch_dcoffset--;
            rch_dcoff_err = 0;
         }
      }

      if (lch_dcoff_err > THHOLD_DCOFF_FIX)
      {
         lch_dcoffset++;
         lch_dcoff_err = 0;
      }
      else
      {
         if (lch_dcoff_err < -THHOLD_DCOFF_FIX)
         {
            lch_dcoffset--;
            lch_dcoff_err = 0;
         }
      }*/

      // Apply an exponentially weighted moving average to the new sample to
      // get next output
      rch_pv = (rch_pv + rch_last) >> 1; // EWMA, alpha = 0.5
      lch_pv = (lch_pv + lch_last) >> 1;
      rch_last = rch_pv;
      lch_last = lch_pv;

      // Perform a remainder-weighted dither and shift to 9 bits
      //
      // This algorithm is optional and can provide benefit.
      // However, there is not enough discernible difference with the selected
      // output LC values to warrant the extra CPU cycles.
      /*
      dither(rch_pv, 3, 0x7, rch_pv_epca);
      dither(lch_pv, 3, 0x7, lch_pv_epca);
      */

      // Shift to 9 bits
      rch_pv_epca = rch_pv >> 3;
      lch_pv_epca = lch_pv >> 3;

      // Update the EPCA output PWM

      // Right side

      SI32_EPCA_A_disable_internal_register_update_on_overflow(SI32_EPCA_0);

      // Adjust back to unsigned magnitude - R+
      SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH0, rch_pv_epca + 256);

      // Invert magnitude for 3-level modulation - R-
      SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH1, -(rch_pv_epca - 255));

      // Left side

      // Adjust back to unsigned magnitude - L+
      SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH2, lch_pv_epca + 256);

      // Invert magnitude for 3-level modulation - L-
      SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH3, -(lch_pv_epca - 255));

      SI32_EPCA_A_enable_internal_register_update_on_overflow(SI32_EPCA_0);
   }
   else if (system_mode == PLAY_USB_INPUT)
   {
      // The USB data is received in the DataPlaneOut file using si32Library
      // This portion of the code outputs the samples to the EPCA at a 48 kHz
      // rate
      //
      // The usb_out_buffer is 3 x USB_OUT_PACKET_SIZE
      // This buffer is really three buffers that are cycled between
      //
      if (_receive0
          && (usb_out_data_counter < USB_OUT_BUFFER_SIZE))
      {
         // Read the buffer (bytes) and create each 16-bit signed sample
         rch_usb = (int16_t) usb_out_buffer[usb_out_data_counter];
         usb_out_data_counter++;
         rch_usb |= ((int16_t) usb_out_buffer [usb_out_data_counter]) << 8;
         usb_out_data_counter++;

         lch_usb = (int16_t) usb_out_buffer[usb_out_data_counter];
         usb_out_data_counter++;
         lch_usb |= ((int16_t) usb_out_buffer [usb_out_data_counter]) << 8;
         usb_out_data_counter++;

         if (usb_out_data_counter == USB_OUT_BUFFER_SIZE)
         {
            _receive0 = NO;
         }
      }
      else if (_receive1
               && (usb_out_data_counter < (USB_OUT_BUFFER_SIZE*2)))
      {
         // Read the buffer (bytes) and create each 16-bit signed sample
         rch_usb = (int16_t) usb_out_buffer[usb_out_data_counter];
         usb_out_data_counter++;
         rch_usb |= ((int16_t) usb_out_buffer [usb_out_data_counter]) << 8;
         usb_out_data_counter++;

         lch_usb = (int16_t) usb_out_buffer[usb_out_data_counter];
         usb_out_data_counter++;
         lch_usb |= ((int16_t) usb_out_buffer [usb_out_data_counter]) << 8;
         usb_out_data_counter++;

         if (usb_out_data_counter == (USB_OUT_BUFFER_SIZE*2))
         {
            _receive1 = NO;
         }
      }
      else if (_receive2
               && (usb_out_data_counter < (USB_OUT_BUFFER_SIZE*3)))
      {
         // Read the buffer (bytes) and create each 16-bit signed sample
         rch_usb = (int16_t) usb_out_buffer[usb_out_data_counter];
         usb_out_data_counter++;
         rch_usb |= ((int16_t) usb_out_buffer [usb_out_data_counter]) << 8;
         usb_out_data_counter++;

         lch_usb = (int16_t) usb_out_buffer[usb_out_data_counter];
         usb_out_data_counter++;
         lch_usb |= ((int16_t) usb_out_buffer [usb_out_data_counter]) << 8;
         usb_out_data_counter++;

         if (usb_out_data_counter == (USB_OUT_BUFFER_SIZE*3))
         {
            _receive2 = NO;

            usb_out_data_counter = 0;
         }
      }

      // Perform a remainder-weighted dither and shift to 9 bits
      //
      // This algorithm is optional and can provide benefit.
      // However, there is not enough discernible difference with the selected
      // output LC values to warrant the extra CPU cycles.
      /*
      dither(rch_usb, 7, 0x7F, rch_usb_epca);
      dither(lch_usb, 7, 0x7F, lch_usb_epca);
      */

      // Shift to 9 bits
      rch_usb_epca = rch_usb >> 7;
      lch_usb_epca = lch_usb >> 7;

      // Update the EPCA output PWM

      // Right side

      SI32_EPCA_A_disable_internal_register_update_on_overflow(SI32_EPCA_0);

      // Adjust back to unsigned magnitude - R+
      SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH0, rch_usb_epca + 256);

      // Invert magnitude for 3-level modulation - R-
      SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH1, -(rch_usb_epca - 255));

      // Left side

      // Adjust back to unsigned magnitude - L+
      SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH2, lch_usb_epca + 256);

      // Invert magnitude for 3-level modulation - L-
      SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH3, -(lch_usb_epca - 255));

      SI32_EPCA_A_enable_internal_register_update_on_overflow(SI32_EPCA_0);

      // If a recording interface is open, perform the recording (upstream)
      // USB operations
      if (usb_recording == 1)
      {
         // The USB data is sent in the DataPlaneIn and class_d.c files using
         // si32Library. This portion of the code inputs the samples to the buffer
         // at a 48 kHz rate
         //
         // The usb_in_buffer is 3 x USB_IN_PACKET_SIZE
         // This buffer is really three buffers that are cycled between
         //
         if ((_transmit0 == NO)
             && (usb_in_data_counter < USB_IN_BUFFER_SIZE))
         {
            // Wait for conversion to complete
            while(!SI32_SARADC_A_is_single_conversion_complete_interrupt_pending(SI32_SARADC_0));

            // Clear the conversion complete interrupt flag
            SI32_SARADC_A_clear_single_conversion_complete_interrupt(SI32_SARADC_0);

            // Fetch the ADC0 results
            mic_adc_input = SI32_SARADC_A_read_data(SI32_SARADC_0);

            // Turn on the red LED (DS8) if an ADC saturation occurs
            // The LED is turned on and off by the Capsense scan interrupt (TIMER0H)
            if ((mic_adc_input >= 0x0FFA)
                  || (mic_adc_input <= 0x0002))
            {
               turn_on_red_led = 1;
            }

            // Convert to a 12-bit signed value
            mic_adc_input -= ACZERO_12B;

            // Convert to a 16-bit signed value
            mic_adc_input <<= 4;

            // Record the ADC0 sample in the buffer
            usb_in_buffer[usb_in_data_counter] = mic_adc_input & 0xFF;
            usb_in_data_counter++;
            usb_in_buffer[usb_in_data_counter] = (mic_adc_input & 0xFF00) >> 8;
            usb_in_data_counter++;

            if (usb_in_data_counter == USB_IN_BUFFER_SIZE)
            {
               if (recording_started == 0)
               {
                  _transmit2 = YES;
               }
            }
         }
         else if ((_transmit1 == NO)
               && (usb_in_data_counter < (USB_IN_BUFFER_SIZE*2)))
         {
            // Wait for conversion to complete
            while(!SI32_SARADC_A_is_single_conversion_complete_interrupt_pending(SI32_SARADC_0));

            // Clear the conversion complete interrupt flag
            SI32_SARADC_A_clear_single_conversion_complete_interrupt(SI32_SARADC_0);

            // Fetch the ADC0 results
            mic_adc_input = SI32_SARADC_A_read_data(SI32_SARADC_0);

            // Turn on the red LED (DS8) if an ADC saturation occurs
            // The LED is turned on and off by the Capsense scan interrupt (TIMER0H)
            if ((mic_adc_input >= 0x0FFA)
                  || (mic_adc_input <= 0x0002))
            {
               turn_on_red_led = 1;
            }

            // Convert to a 12-bit signed value
            mic_adc_input -= ACZERO_12B;

            // Convert to a 16-bit signed value
            mic_adc_input <<= 4;

            // Record the ADC0 sample in the buffer
            usb_in_buffer[usb_in_data_counter] = mic_adc_input & 0xFF;
            usb_in_data_counter++;
            usb_in_buffer[usb_in_data_counter] = (mic_adc_input & 0xFF00) >> 8;
            usb_in_data_counter++;

            if (usb_in_data_counter == (USB_IN_BUFFER_SIZE*2))
            {
               _transmit0 = YES;
            }
         }
         else if ((_transmit2 == NO)
               && (usb_in_data_counter < (USB_IN_BUFFER_SIZE*3)))
         {
            // Wait for conversion to complete
            while(!SI32_SARADC_A_is_single_conversion_complete_interrupt_pending(SI32_SARADC_0));

            // Clear the conversion complete interrupt flag
            SI32_SARADC_A_clear_single_conversion_complete_interrupt(SI32_SARADC_0);

            // Fetch the ADC0 results
            mic_adc_input = SI32_SARADC_A_read_data(SI32_SARADC_0);

            // Turn on the red LED (DS8) if an ADC saturation occurs
            // The LED is turned on and off by the Capsense scan interrupt (TIMER0H)
            if ((mic_adc_input >= 0x0FFA)
                  || (mic_adc_input <= 0x0002))
            {
               turn_on_red_led = 1;
            }

            // Convert to a 12-bit signed value
            mic_adc_input -= ACZERO_12B;

            // Convert to a 16-bit signed value
            mic_adc_input <<= 4;

            // Record the ADC0 sample in the buffer
            usb_in_buffer[usb_in_data_counter] = mic_adc_input & 0xFF;
            usb_in_data_counter++;
            usb_in_buffer[usb_in_data_counter] = (mic_adc_input & 0xFF00) >> 8;
            usb_in_data_counter++;

            if (usb_in_data_counter == (USB_IN_BUFFER_SIZE*3))
            {
               _transmit1 = YES;

               usb_in_data_counter = 0;
            }
         }
      }
   }
   else if (system_mode == PLAY_PRERECORDED_FLASH)
   {
      // These variables are cleared every time a playback session starts to
      // ensure the recording always starts in the same state
      if (clear_class_d_variables == 1)
      {
         clear_class_d_variables = 0;

         rch_pv = 0;
         lch_pv = 0;
         rch_last = 0;
         lch_last = 0;
         flash_toggle = 0;
      }

      if (playing == 1)
      {
         if (num_samples_played < NUM_PR_SAMPLES_RECORDED)
         {
            // Read the data from the array in Flash
            compressed_data = pre_recorded_array[num_samples_played];

            num_samples_played++;

            // Convert the data to 12-bit signed from 14-bit signed
            tmp_int16 = (muLaw_Decode(compressed_data)) >> 2;

            rch_pv = (int32_t) tmp_int16;

            // Apply moving average to new sample to get next output
            rch_pv = (rch_pv + rch_last) >> 1; // EWMA, alpha = 0.5
            rch_last = rch_pv;

            // Remove LSBs from sample
            // We don't dither here because the LSBs will often be 0's
            rch_pv = rch_pv >> 3;

            // Update the EPCA output PWM
            // Since the recording is mono, send the same output to both right
            // and left channels

            // Right side

            SI32_EPCA_A_disable_internal_register_update_on_overflow(SI32_EPCA_0);

            // Adjust back to unsigned magnitude - R+
            SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH0, rch_pv + 256);

            // Invert magnitude for 3-level modulation - R-
            SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH1, -(rch_pv - 255));

            // Left side

            // Adjust back to unsigned magnitude - L+
            SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH2, rch_pv + 256);

            // Invert magnitude for 3-level modulation - L-
            SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH3, -(rch_pv - 255));

            SI32_EPCA_A_enable_internal_register_update_on_overflow(SI32_EPCA_0);
         }
         else
         {
            SI32_TIMER_A_stop_low_timer(SI32_TIMER_1);
            num_samples_played = 0;
            playing = 0;
            LED_Off(LED5);
         }
      }
   }
   else if (system_mode == RECORD_PLAY_MIC_INPUT)
   {
      // These variables are cleared every time a record or playback session
      // starts to ensure the recording always starts in the same state
      if (clear_class_d_variables == 1)
      {
         clear_class_d_variables = 0;

         rch_pv = 0;
         lch_pv = 0;
         rch_last = 0;
         lch_last = 0;
         flash_toggle = 0;
      }

      if (playing == 1)
      {
         if (num_samples_played < num_samples_recorded)
         {
            // Read the data from the array in Flash
            compressed_data = recorded_array[num_samples_played];

            num_samples_played++;

            // Convert the data to 12-bit signed from 14-bit signed
            tmp_int16 = (muLaw_Decode(compressed_data)) >> 2;

            rch_pv = (int32_t) tmp_int16;

            // Apply moving average to new sample to get next output
            rch_pv = (rch_pv + rch_last) >> 1; // EWMA, alpha = 0.5
            rch_last = rch_pv;

            // Remove LSBs from sample
            // We don't dither here because the LSBs will often be 0's
            rch_pv = rch_pv >> 3;

            // Update the EPCA output PWM
            // Since the recording is mono, send the same output to both right
            // and left channels

            // Right side

            SI32_EPCA_A_disable_internal_register_update_on_overflow(SI32_EPCA_0);

            // Adjust back to unsigned magnitude - R+
            SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH0, rch_pv + 256);

            // Invert magnitude for 3-level modulation - R-
            SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH1, -(rch_pv - 255));

            // Left side

            // Adjust back to unsigned magnitude - L+
            SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH2, rch_pv + 256);

            // Invert magnitude for 3-level modulation - L-
            SI32_EPCACH_A_write_ccapvupd(SI32_EPCA_0_CH3, -(rch_pv - 255));

            SI32_EPCA_A_enable_internal_register_update_on_overflow(SI32_EPCA_0);
         }
         else
         {
            // When the recording runs out, end the playback session
            SI32_TIMER_A_stop_low_timer(SI32_TIMER_1);
            num_samples_played = 0;
            playing = 0;
            LED_Off(LED5);
         }
      }
      else if (recording == 1)
      {
         if (num_samples_recorded < RECORD_ARRAY_SIZE)
         {
            // Wait for conversion to complete
            while(!SI32_SARADC_A_is_single_conversion_complete_interrupt_pending(SI32_SARADC_0));

            // Clear the conversion complete interrupt flag
            SI32_SARADC_A_clear_single_conversion_complete_interrupt(SI32_SARADC_0);

            // Fetch the ADC0 results
            adc0_data = SI32_SARADC_A_read_data(SI32_SARADC_0);

            // Turn on the red LED (DS8) if an ADC saturation occurs
            if ((adc0_data >= 0x0FFA)
                || (adc0_data <= 0x0002))
            {
               turn_on_red_led = 1;
            }

            // Convert the value to a signed, 14-bit number
            tmp_int16 = (int16_t) ((adc0_data - ACZERO_12B)) << 2;

            // muLaw compresses a signed 14-bit value to 8 bits
            // However, the range of the muLaw algorithm is actually -8159 to 8158
            // due to an increase in magnitude by 33, which is part of the algorithm.
            if (tmp_int16 > 8158)
            {
               tmp_int16 = 8158;
            }
            else if (tmp_int16 < -8159)
            {
               tmp_int16 = -8159;
            }

            compressed_data = muLaw_Encode(tmp_int16);

            // Write data to Flash
            // Data is in 8-bit values and Flash writes in 16-bit values, so write
            // two bytes at once. This means that if recording stops on an odd address
            // the last sample won't be written to Flash. This should never be
            // something noticeable to the user, however.
            if (flash_toggle == 0)
            {
               flash_data = compressed_data & 0x00FF;

               flash_toggle = 1;
            }
            else
            {
               flash_data |= (compressed_data << 8) & 0xFF00;

               SI32_FLASHCTRL_A_write_flash_key(SI32_FLASHCTRL_0, 0xA5);
               SI32_FLASHCTRL_A_write_flash_key(SI32_FLASHCTRL_0, 0xF1);

               SI32_FLASHCTRL_A_write_wrdata(SI32_FLASHCTRL_0, flash_data);

               num_samples_recorded += 2;
               flash_toggle = 0;
            }
         }
         else
         {
            // When Flash space runs out, end the recording session
            SI32_TIMER_A_stop_low_timer(SI32_TIMER_1);
            LED_Off(LED4);
            recording = 0;
         }
      }
   }

   #if ((P32_DEBUG == 1) || (UVISION_DEBUG == 1))

   SI32_PBSTD_A_write_pins_low(SI32_PBSTD_3, 0x0001);

   #endif
}

// This ISR handles the LED PWM updates for the six blue LEDs. The three actions
// available are BRIGHTEN, DARKEN, and DARKEN_THEN_BRIGHTEN_NEXT. This last action
// is used whenever switching between system modes.
void TIMER1H_high_overflow_handler(void)
{
   SI32_TIMER_A_clear_high_overflow_interrupt(SI32_TIMER_1);

   if (LED_direction == BRIGHTEN)
   {
     if (LED_index < NUM_LED_STEPS)
     {
        LED_PCA_Control(LED_to_use, percent_array[LED_index]);

        LED_index++;
     }
     else
     {
        SI32_TIMER_A_stop_high_timer(SI32_TIMER_1);
     }
   }
   // LED_direction must be DARKEN or DARKEN_THEN_BRIGHTEN_NEXT
   else
   {
      if (LED_index >= 0)
      {
         LED_PCA_Control(LED_to_use, percent_array[LED_index]);

         LED_index--;
      }
      else
      {
         if (LED_direction == DARKEN_THEN_BRIGHTEN_NEXT)
         {
            LED_index = 0;
            LED_direction = BRIGHTEN;
            switch (LED_to_use)
            {
               case LED0:
                  LED_to_use = LED1;
                  break;
               case LED1:
                  LED_to_use = LED2;
                  break;
               case LED2:
                  LED_to_use = LED3;
                  break;
               case LED3:
                  LED_to_use = LED0;
                  break;
               case LED4:
                  LED_to_use = LED0;
                  break;
               case LED5:
                  LED_to_use = LED0;
                  break;
            }
         }
         else
         {
            SI32_TIMER_A_stop_high_timer(SI32_TIMER_1);
         }
      }
   }
}
