// Copyright (c) 2012

// library
#include <si32UsbCoreComponent.h>
// hal
#include <si32_device.h>
#include <SI32_CLKCTRL_A_Type.h>
#include <SI32_USB_A_Type.h>
#include <SI32_TIMER_A_Type.h>
#include <SI32_USBEP_A_Type.h>
// application
#include "myApplication.h"
#include "myUsbAudioDevice.h"
#include "myUSB0.h"
#include "myDataPlane.h"
#include "class_d.h"

#define printf(args...)

// Most of the code in this file was taken verbatim from the demo_si32UsbAudio
// example

//------------------------------------------------------------------------------
void myUSB0_initialize_memory(void)
{
   extern void si32Usb_initialize_memory(void);


   // Initialize usb component memory (initializes library objects & buffers for
   // basic USB devices)
   si32Usb_initialize_memory();

   // (optionally) Initialize additional endpoints here, if used.
   // Note:  There must be 5 UsbEpIoObjects - even if only using the default
   _si32UsbIoObject_initialize_endpoint(_si32UsbIoObjects[1], 1, -1, -1);
   _si32UsbIoObject_initialize_endpoint(_si32UsbIoObjects[2], 2, -1, -1);
   _si32UsbIoObject_initialize_endpoint(_si32UsbIoObjects[3], 3, -1, -1);
   _si32UsbIoObject_initialize_endpoint(_si32UsbIoObjects[4], 4, -1, -1);
}

//------------------------------------------------------------------------------
si32UsbDeviceObject * myUsbDevice;

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

void USB_initialize ()
{
   printf("USB_initialize\n");

   // _SI32_USB_A_set_usb_clock_pll(SI32_USB_0);

   // Perform asynchronous reset of the USB module
   _SI32_USB_A_reset_module (SI32_USB_0);

   // Enable Endpoint 0 interrupts
   _SI32_USB_A_write_cmint (SI32_USB_0, 0x00000000);
   _SI32_USB_A_write_ioint (SI32_USB_0, 0x00000000);
   _SI32_USB_A_enable_ep0_interrupt (SI32_USB_0);

   // Enable Reset, Resume, Suspend interrupts
   _SI32_USB_A_enable_suspend_interrupt (SI32_USB_0);
   _SI32_USB_A_enable_suspend_detection (SI32_USB_0);
   _SI32_USB_A_enable_resume_interrupt (SI32_USB_0);
   _SI32_USB_A_enable_reset_interrupt (SI32_USB_0);
   _SI32_USB_A_enable_start_of_frame_interrupt (SI32_USB_0);

   // Enable Transceiver, fullspeed
   _SI32_USB_A_write_tcontrol (SI32_USB_0, 0x00);
   _SI32_USB_A_select_transceiver_full_speed (SI32_USB_0);
   _SI32_USB_A_enable_transceiver (SI32_USB_0);
   // _SI32_USB_A_enable_internal_pull_up (SI32_USB_0);

   // Enable clock recovery, single-step mode disabled
   _SI32_USB_A_enable_clock_recovery (SI32_USB_0);
   _SI32_USB_A_select_clock_recovery_mode_full_speed (SI32_USB_0);
   _SI32_USB_A_select_clock_recovery_normal_cal  (SI32_USB_0);
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
void USB_connect()
{
   _SI32_USB_A_enable_internal_pull_up (SI32_USB_0);
}
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
void USB_disconnect()
{
   _SI32_USB_A_disable_internal_pull_up (SI32_USB_0);
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
void USB_start()
{
   // Try to enable EP0 interrupt, where it may have been ommitted accidentally...
   SI32_USB_0->CMINTEPE.U32 |= (5<<16) ;

   // Enable USB interrupts
   NVIC_EnableIRQ (USB0_IRQn);

   // Uninhibit the module once all initialization is complete
   _SI32_USB_A_enable_module (SI32_USB_0);
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
void USB_stop()
{
   // Enable USB interrupts
   NVIC_DisableIRQ(USB0_IRQn);

   // Inhibit USB
   _SI32_USB_A_disable_module(SI32_USB_0);
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
void USB_shutdown()
{
   // Reset the block
   _SI32_USB_A_reset_module (SI32_USB_0);
}


#if 1
//------------------------------------------------------------------------------
// Store middleware usb device object
void USB0_IRQHandler(void)
{
   // Route the IRQ to app-specific usb device object
   si32UsbDeviceObject_IRQHandler(myUsbDevice);
   return;
}
#else
//------------------------------------------------------------------------------
void usb_dsr(obj* obj, void* context)
{
   // si32UsbDeviceObject * device =
   si32UsbDeviceObject_IRQHandler((si32UsbDeviceObject *) obj);
   NVIC_EnableIRQ (USB0_IRQn);
}

//------------------------------------------------------------------------------
void USB0_IRQHandler(void)
{
   // Route the IRQ to app-specific usb device object
   // si32UsbDeviceObject_IRQHandler(myUsbDevice);
   NVIC_DisableIRQ (USB0_IRQn);
   si32RunLoopObject_post_work_request(si32GetCurrentRunLoop(),
                                      SI32_RUN_LOOP_PRIORITY_HIGH,
                                      usb_dsr,
                                      myUsbDevice,
                                      NULL);
   return;
}
#endif

//------------------------------------------------------------------------------
void usb_device_state_change(uint32_t from, uint32_t to)
{
   printf("USB DEVICE STATE:  %d->%d\n", from, to);
}

//------------------------------------------------------------------------------
//
extern void usb_device_state_change(uint32_t old, uint32_t new);
void myUsbDevice_poller(obj* obj, void* context)
{
   si32UsbDeviceObject * pDevice = (si32UsbDeviceObject *) obj;
   static uint8_t last = 0xff;
   if ( last ^ pDevice->usbDeviceObject.State )
   {
      uint32_t save_last = last;
      last=pDevice->usbDeviceObject.State;
      usb_device_state_change(save_last, last);
   }
}

//------------------------------------------------------------------------------

// Reset the endpoints and re-enable Suspend detection every USB reset
//
// This allows the system to recover after a surprise removal during USB
// recording
void si32UsbDeviceObject_OnUsbReset(si32UsbDeviceObject * self)
{
   si32UsbDefaultEndpointObject_reset(self->usbDeviceObject.defaultEndpoint);
   si32UsbDeviceObject_SetState(self, USB_DEVICE_STATE_DEFAULT);
   _SI32_USB_A_enable_suspend_detection (SI32_USB_0);
}

// On USB Suspend events, clear the flags that prevent the system from
// changing mode while recording.
//
// This allows the system to recover after a surprise removal during USB
// recording
void si32UsbDeviceObject_OnUsbSuspend(si32UsbDeviceObject * theDevice)
{
   usb_recording = 0;
   usb_transmitting = 0;
}

// The Start-of-Frame call-back from si32Library handles most of the USB
// code for this application.  The pollers check to make sure the audio interfaces
// are still valid, and the Timer algorithm adjusts the Timer overflow rate
// relative to the Start-of-Frame to keep the Timer overflow in time with the host.
// This prevents any synchronization issues when receiving or sending data from
// or to the host.
void si32UsbDeviceObject_OnUsbStartOfFrame(si32UsbDeviceObject * theDevice, uint16_t frameNumber)
{
   int32_t diff;
   static int32_t diffLast;
   int32_t diffDelta;
   uint16_t timer_reload_value;
   static int8_t timer_adjust = 0;

   // SOF Handler
   // - Check all audio interfaces (out and in) for status updates
   // - Check USB transceiver for status updates
   // - Perform PCA adjustment calculations synchronized to buffer level
   //   as sampled at each SOF.

   demo_audio_out_poller(&myUsbAudioStreamingInterfaceOutObject_1, NULL);
   demo_audio_in_poller(&myUsbAudioStreamingInterfaceInObject_1, NULL);
   myUsbDevice_poller(myUsbDevice, NULL);

   // This routine adjusts the TIMER1L level such the output buffer remains exactly
   // 2 ms (192 samples, or 2 IN isochronous buffer) ahead of the USB.
   //
   // When out-of synch by 1 byte, this code will increment or decrement an
   // adjustment value that will be applied to the TIMER1L reload value.  In order to minimize
   // the frequency shift, the adjustment will only change by +/- 1 count (~21 ns) each
   // millisecond until the differences between the input and output buffer offsets begins
   // to lessen, at which point, the code will stop making adjustments.
   //
   // This routine depends on 2 calculations:
   //  - the number of buffered audio data samples
   //  - the rate of change in buffer level (buffer is draining, or filling too fast)
   //

   if (usb_recording == 1)
   {
      diff = usb_in_data_counter - sent_in_data_counter;
      diffDelta = diff - diffLast;
      diffLast = diff;

      // Maintain a buffer level of exactly 2 ms of samples (192 bytes)

      // If more than 2 ms of data has been sourced, and source is still filling
      // faster than sink is draining, decrease the TIMER1L overflow period by
      // lowering the overflow rate
      if ((diff > USB_IN_BUFFER_SIZE*2)
          && (diffDelta >= 0))
      {
         // Decrease the TIMER1L overflow rate and speed up the relative USB
         // transmission of data
         if (timer_adjust > -TIMER1L_LIMIT)
         {
            timer_adjust--;

            timer_reload_value = SI32_TIMER_A_get_low_capture(SI32_TIMER_1);
            timer_reload_value--;
            SI32_TIMER_A_set_low_reload(SI32_TIMER_1, timer_reload_value);
         }
      }
      // If less than 2 ms of data has been sourced, and sink is draining faster than
      // source is filling, increase the TIMER1L overflow period by increasing the
      // overflow rate
      if ((diff < USB_IN_BUFFER_SIZE*2)
          && (diffDelta <= 0))
      {
         // Increase the TIMER1L overflow rate and slow up the relative USB
         // transmission of data
         if (timer_adjust < TIMER1L_LIMIT)
         {
            timer_adjust++;

            timer_reload_value = SI32_TIMER_A_get_low_capture(SI32_TIMER_1);
            timer_reload_value++;
            SI32_TIMER_A_set_low_reload(SI32_TIMER_1, timer_reload_value);
         }
      }
      // If neither of the above conditions are true, then leave the TIMER1L overflow
      // rate where it is. We're either locked at 192 bytes or are catching up to it.

   }
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
void myUsb0_main(si32UsbDeviceObject * pDevice)
{
   /* Specify the device object instance */
   myUsbDevice = pDevice;

   /* --------------------- */
   /* Initialize USB Module */
   /* --------------------- */
   USB_initialize ();

   /* -------------------- */
   /* Set custom parameters */
   /* -------------------- */
   /* test setting ahb source to internal usb oscillator */
   // SI32_SIM3U1XX_CLKCTRL_A_set_ahb_source_usb0_oscillator(SI32_CLKCTRL_0);
   // SystemCoreClock = 48000000;
   // SysTick_Config(SystemCoreClock/1000);

   /* -------------------- */
   /* Start the USB Module */
   /* -------------------- */
   USB_start();

   /* ------------------ */
   /* Connect to the USB */
   /* ------------------ */
   USB_connect();
}

//------------------------------------------------------------------------------
void myUSB0_enter_active_mode()
{
  myUsbAudioDevice_main();
}

//-eof--------------------------------------------------------------------------
