// Copyright (c) 2012

//------------------------------------------------------------------------------
// Include Files
//------------------------------------------------------------------------------

#include <si32_device.h>
#include <SI32_PBHD_A_Type.h>
#include <SI32_PBSTD_A_Type.h>

#include "LED_control.h"
#include "myCAPSENSE0.h"
#include "class_d.h"

//------------------------------------------------------------------------------
// Definitions
//------------------------------------------------------------------------------

// volume levels
#define MAX_VOLUME_LEVEL      15    // 15 = full scale
#define VOLUME_CHANGE          1    // 1 = 1/16th full-scale
#define MIN_VOLUME_LEVEL       0

//------------------------------------------------------------------------------
// Global Variables
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// 0 = left output, 1 = right output, 2 = input (mic)
// Note: The Class-D board only implements volume control for both channels
// at the same time.  The volume control for the R speaker is used for both R
// and L.  The microphone doesn't have volume control on the board.
int16_t USBVolumeControl[3] = { -30, -30, 0 };
int16_t VolumeControl[3] = { 7, 7, 0 };

// 0 = not muted, 1 = muted
bool MuteControl[3] = { 0, 0, 0 };

// This is a semaphore to prevent the USB and the CAPSENSE from updating the
// volume at the same time.
bool volume_updating = 0;
bool volume_update = 0;

//------------------------------------------------------------------------------
// Global Non-Extern Variables or Defines
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Support Functions
//------------------------------------------------------------------------------

void convert_USB_to_non_USB(void)
{
   // USBVolumeControl ranges from -127 to 0
   // VolumeControl ranges from 0 to 15
   //
   // The maximum output of this function is 16 when USBVolumeControl is 0, but
   // the volume control saturates any input value above 15 as a value of 15.
   VolumeControl[1] = (USBVolumeControl[1] + 128) / 8;

   // Copy the R volume for both R and L
   VolumeControl[0] = VolumeControl[1];
}

void volume_control(void)
{
   uint32_t volume_level = VolumeControl[1];
   uint32_t remainder;
   uint32_t result;
   Class_D_LED_Enum_Type LED;

    if (volume_level > MAX_VOLUME_LEVEL)
    {
       volume_level = MAX_VOLUME_LEVEL;
    }

    if ((volume_level > 1) && (MuteControl[1] == 0))
    {
       SI32_PBHD_A_enable_drivers(SI32_PBHD_4);  // Enable HD pads

       // Adjust attenuation
       SI32_PBHD_A_select_nchannel_current_limit(SI32_PBHD_4, volume_level);

       if (volume_level > 7)
       {
          // P drive @ max
          SI32_PBHD_A_select_pchannel_current_limit(SI32_PBHD_4, MAX_VOLUME_LEVEL);
       }
       else
       {
          // P drive = 2x N level
          SI32_PBHD_A_select_pchannel_current_limit(SI32_PBHD_4, (volume_level << 1));
       }


       // LED Control

       // The volume level can range from 0-19, so a case statement for 6 is needed
       // even though there are only 6 LEDs (0-5).  Volume levels of 6 look identical
       // to volume levels of 5 on the LEDs.
       result = volume_level / 3;
       remainder = volume_level % 3;

       switch (result)
       {
          case 0:
             LED = LED0;
             LED_Manual_Control(LED1, OFF);
             LED_Manual_Control(LED2, OFF);
             LED_Manual_Control(LED3, OFF);
             LED_Manual_Control(LED4, OFF);
             LED_Manual_Control(LED5, OFF);
             break;
          case 1:
             LED_Manual_Control(LED0, ON);
             LED = LED1;
             LED_Manual_Control(LED2, OFF);
             LED_Manual_Control(LED3, OFF);
             LED_Manual_Control(LED4, OFF);
             LED_Manual_Control(LED5, OFF);
             break;
          case 2:
             LED_Manual_Control(LED0, ON);
             LED_Manual_Control(LED1, ON);
             LED = LED2;
             LED_Manual_Control(LED3, OFF);
             LED_Manual_Control(LED4, OFF);
             LED_Manual_Control(LED5, OFF);
             break;
          case 3:
             LED_Manual_Control(LED0, ON);
             LED_Manual_Control(LED1, ON);
             LED_Manual_Control(LED2, ON);
             LED = LED3;
             LED_Manual_Control(LED4, OFF);
             LED_Manual_Control(LED5, OFF);
             break;
          case 4:
             LED_Manual_Control(LED0, ON);
             LED_Manual_Control(LED1, ON);
             LED_Manual_Control(LED2, ON);
             LED_Manual_Control(LED3, ON);
             LED = LED4;
             LED_Manual_Control(LED5, OFF);
             break;
          case 5:
             LED_Manual_Control(LED0, ON);
             LED_Manual_Control(LED1, ON);
             LED_Manual_Control(LED2, ON);
             LED_Manual_Control(LED3, ON);
             LED_Manual_Control(LED4, ON);
             LED = LED5;
          case 6:
             LED_Manual_Control(LED0, ON);
             LED_Manual_Control(LED1, ON);
             LED_Manual_Control(LED2, ON);
             LED_Manual_Control(LED3, ON);
             LED_Manual_Control(LED4, ON);
             LED = LED5;
             remainder = 2;
             break;
          default:
             break;
       }

       LED_PCA_Control(LED, percent_array[remainder*5]);
   }
   else
   {
      // When volume is 0, turn off all of the LEDs and disable the HD drivers.
      LED_Manual_Control(LED0, OFF);
      LED_Manual_Control(LED1, OFF);
      LED_Manual_Control(LED2, OFF);
      LED_Manual_Control(LED3, OFF);
      LED_Manual_Control(LED4, OFF);
      LED_Manual_Control(LED5, OFF);
      SI32_PBHD_A_disable_drivers(SI32_PBHD_4);  // Disable HD pads
   }

   // These flags are used in the CAPSENSE interrupt handler to handle the LEDs
   // after a volume update from either the CAPSENSE slider or USB.
   volume_update = 1;
   touch_lost_hysteresis = 0;
}

//-eof-------------------------------------------------------------------------
