//------------------------------------------------------------------------------
// Copyright (c) 2012 by Silicon Laboratories.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Silicon Laboratories End User
// License Agreement which accompanies this distribution, and is available at
// http://developer.silabs.com/legal/version/v10/License_Agreement_v10.htm
// Original content and implementation provided by Silicon Laboratories.
//------------------------------------------------------------------------------
// library
#include <stdio.h>
// hal
#include <si32_device.h>
#include <SI32_RSTSRC_A_Type.h>
#include <SI32_ACCTR_A_Type.h>
#include <SI32_PBSTD_A_Type.h>
// application
#include "myCpu.h"
#include "myModes.h"
#include "myLCD0.h"
#include "myPMU.h"
#include "myACCTR0.h"

//------------------------------------------------------------------------------
// Global Variables
//------------------------------------------------------------------------------

static volatile uint32_t msTicks = 0;

//------------------------------------------------------------------------------
// Global Defines
//------------------------------------------------------------------------------

#define ACTIVE_MODE  0  // Active mode immediat update
#define MONITOR_MODE 1  // Sleep w/ rtc & lcd update every alarm_count
#define SLEEP_MODE   2  // Sleep lfo. update when wake button pressed

//------------------------------------------------------------------------------
// Local Function Prototypes
//------------------------------------------------------------------------------

uint32_t get_msTicks(void);

//------------------------------------------------------------------------------
// myApplication.
//------------------------------------------------------------------------------
int main()
{
  uint32_t mode = ACTIVE_MODE;
  uint32_t active = 1;
  char modes[3] = {'A', 'M', 'S'};

  // Initialize CMSIS global to reset oscillator value
  SystemCoreClock = 20000000;

  // Set SysTick to 1 msec
  SysTick_Config(SystemCoreClock / 1000);

  // Device configuraton
  myModes_enter_default_mode();

  // Echo configurable settings
  SI32_ACCTR_A_write_comp0(SI32_ACCTR_0, ACCTR_ALARM_COUNT);
  while(SI32_ACCTR_A_is_write_in_progress(SI32_ACCTR_0));
  myLCD0_printf("\n\rALRM=%d", ACCTR_ALARM_COUNT);
  wait_ms(1000);

  // Setup Interrupts
  SI32_ACCTR_A_clear_digital_comparator_0_interrupt_flag(SI32_ACCTR_0);
  SI32_ACCTR_A_enable_digital_comparator_0_interrupt(SI32_ACCTR_0);
  while(SI32_ACCTR_A_is_write_in_progress(SI32_ACCTR_0));


  // Start ACCTR module
  myACCTR0_start_lc_sequence();

  // CALIBRATE MODE

  // Clear MIN/MAX, then block until the user signals that meter has spun
  SI32_ACCTR_A_read_lc_limits(SI32_ACCTR_0);
  myLCD0_printf("\n\r--CAL--");
  while(SI32_PBSTD_A_read_pin(SI32_PBSTD_0, 1));

  // Start automatic tracking (min/max tracking) and auto discriminate
  SI32_ACCTR_A_enable_lc_automatic_tracking(SI32_ACCTR_0);
  SI32_ACCTR_A_enable_lc_automatic_center_discriminator(SI32_ACCTR_0);

  // Poll on the write busy flag
  while(SI32_ACCTR_A_is_write_in_progress(SI32_ACCTR_0));

  myLCD0_printf("\n\rPC=0x%02X", SI32_ACCTR_A_get_lc_count_0_discriminator(SI32_ACCTR_0));
  wait_ms(1000);

  myLCD0_printf("\n\rCOUNTING");
  wait_ms(1000);

  while(1)
  {
     // First update the LCD Display with the counted pulses since reset
     myLCD0_printf("\n\r%c 0X%04x",
                   modes[mode],
                   SI32_ACCTR_A_read_count0(SI32_ACCTR_0));

     if(mode == MONITOR_MODE)
     {
        // MONITOR MODE

        // If we are still in ACTIVE mode, move to MONITOR mode
        if(active)
        {
           myModes_enter_monitor_mode();
           active = 0;
        }
        else if(!SI32_PBSTD_A_read_pin(SI32_PBSTD_0, 1))
        {
           // If a pin wake occurred, move from MONITOR to ACTIVE
           myModes_enter_active_from_monitor();
           mode = ACTIVE_MODE;
        }
        else
        {
           // If the PMU woke from a source other than pin wake, do nothing
           myPMU_sleep();
        }
     }
     else if(mode == SLEEP_MODE)
     {
        // SLEEP MODE

        // If we are still in ACTIVE mode, move to SLEEP mode
        if(active)
        {
           // Active->Sleep
           myModes_enter_sleep_mode();
           active = 0;
        }
        else
        {
           myModes_enter_active_from_sleep();
           mode = ACTIVE_MODE;
        }
     }
     else if(mode == ACTIVE_MODE)
     {
        // ACTIVE MODE

        active = 1;

        // Pressing PB1.4 moves to MONITOR mode
        if(!SI32_PBSTD_A_read_pin(SI32_PBSTD_1, 6))
        {
           mode = MONITOR_MODE;
        }

        // Pressing PB1.5 moves to SLEEP mode
        if(!SI32_PBSTD_A_read_pin(SI32_PBSTD_1, 7))
        {
           mode = SLEEP_MODE;
        }

        wait_ms(100);
     }
  }
}

//------------------------------------------------------------------------------
// get_msTicks()
//------------------------------------------------------------------------------
uint32_t get_msTicks(void)
{
  return msTicks;
}

//------------------------------------------------------------------------------
// SysTick_Handler()
//------------------------------------------------------------------------------
void SysTick_Handler(void)
{
  msTicks++;
}

//-eof--------------------------------------------------------------------------
