/////////////////////////////////////////////////////////////////////////////
// SLAB_DFU.h
// For SLAB_DFU.dll version 1.1
/////////////////////////////////////////////////////////////////////////////

/// @file SLAB_DFU.h
/// The main header file for SLAB_DFU.dll version 1.1

#ifndef SLAB_DFU_H
#define SLAB_DFU_H

/////////////////////////////////////////////////////////////////////////////
// DLL Preprocessor Definitions
/////////////////////////////////////////////////////////////////////////////

#ifdef SLAB_DFU_EXPORTS
#define SLAB_DFU_API __declspec(dllexport)
#else
#define SLAB_DFU_API __declspec(dllimport)
#endif

/////////////////////////////////////////////////////////////////////////////
// Type Definitions - Windows Types
/////////////////////////////////////////////////////////////////////////////
/// @name Windows Type Definitions
/// @{

typedef int                 BOOL;
typedef unsigned char       BYTE;
typedef unsigned short      WORD;
typedef unsigned long       DWORD;
typedef char*               LPSTR;
typedef const char*         LPCSTR;
/// @}

/////////////////////////////////////////////////////////////////////////////
// Definitions - Windows Definitions
/////////////////////////////////////////////////////////////////////////////
/// @name Windows Definitions
/// @{

#define MAX_PATH            260
#define CALLBACK            __stdcall
#define WINAPI              __stdcall
/// @}

/////////////////////////////////////////////////////////////////////////////
// Type Definitions - DFU_CODE
/////////////////////////////////////////////////////////////////////////////

/// @details A DFU return code
///
/// @sa ::DFU_CODE_SUCCESS
/// @sa ::DFU_CODE_API_INVALID_PARAMETER
/// @sa ::DFU_CODE_API_INVALID_BUFFER_SIZE
/// @sa ::DFU_CODE_API_INVALID_HEX_RECORD
/// @sa ::DFU_CODE_API_NOT_IN_DFU_IDLE_STATE
/// @sa ::DFU_CODE_API_DNLOAD_PHASE_FAILED
/// @sa ::DFU_CODE_API_MANIFEST_PHASE_FAILED
/// @sa ::DFU_CODE_API_INVALID_IMAGE_FILE
/// @sa ::DFU_CODE_API_IMAGE_GENERATION_FAILED
/// @sa ::DFU_CODE_API_INVALID_DEVICE_OBJECT
/// @sa ::DFU_CODE_API_FATAL_ERROR
/// @sa ::DFU_CODE_API_ABORTED
/// @sa ::DFU_CODE_API_NOT_SUPPORTED
/// @sa ::DFU_CODE_HWIF_DEVICE_NOT_FOUND
/// @sa ::DFU_CODE_HWIF_DEVICE_NOT_OPENED
/// @sa ::DFU_CODE_HWIF_DEVICE_ERROR
/// @sa ::DFU_CODE_HWIF_TRANSFER_ERROR
/// @sa ::DFU_CODE_HWIF_TRANSFER_TIMEOUT
/// @sa ::DFU_CODE_HWIF_TRANSFER_INCOMPLETE
typedef BYTE DFU_CODE;

/////////////////////////////////////////////////////////////////////////////
// Definitions - DFU_CODE : Success Return Codes
/////////////////////////////////////////////////////////////////////////////
/// @name DFU_CODE : Success Return Codes
/// @{

/// The function returned
/// successfully.
#define DFU_CODE_SUCCESS                        0x00
/// @}

/////////////////////////////////////////////////////////////////////////////
// Definitions - DFU_CODE : API Errors
/////////////////////////////////////////////////////////////////////////////
/// @name DFU_CODE : API Errors
/// @{

/// The function parameter is invalid.
/// Make sure that no pointer arguments are @c NULL,
/// and verify that any specified indexes are within
/// valid ranges.
#define DFU_CODE_API_INVALID_PARAMETER          0x40

/// An array parameter is not large enough to return
/// the requested data.
#define DFU_CODE_API_INVALID_BUFFER_SIZE        0x41

/// The hex file contains an invalid
/// record.
#define DFU_CODE_API_INVALID_HEX_RECORD         0x42

/// The device could not be placed in the DFU_IDLE
/// state.
#define DFU_CODE_API_NOT_IN_DFU_IDLE_STATE      0x43

/// An error occurred in the DFU download
/// state. Check the ::DFU_GET_STATUS structure
/// for more information.
#define DFU_CODE_API_DNLOAD_PHASE_FAILED        0x44

/// An error occurred in the DFU manifestation
/// state. Check the ::DFU_GET_STATUS structure
/// for more information.
#define DFU_CODE_API_MANIFEST_PHASE_FAILED      0x45

/// Failed to open the specified image file or the
/// image file is invalid. Check that the file
/// exists and is readable prior to continuing.
#define DFU_CODE_API_INVALID_IMAGE_FILE         0x46

/// Failed to create the specified image file.
/// Check that folder is writeable and that the
/// file does not already exist.
#define DFU_CODE_API_IMAGE_GENERATION_FAILED    0x47

/// The specified device object pointer is invalid.
/// Make sure that the device hasn't already been
/// closed.
#define DFU_CODE_API_INVALID_DEVICE_OBJECT      0x48

/// An unknown error or system error
/// occurred.
#define DFU_CODE_API_FATAL_ERROR                0x49

/// The user aborted a time-lengthy operation by
/// returning abort (@c FALSE) from a
/// ::DFU_PROGRESS_CALLBACK.
#define DFU_CODE_API_ABORTED                    0x4a

/// The function is not supported by the device
/// especially if the device interface type is
/// UART.
#define DFU_CODE_API_NOT_SUPPORTED              0x4b

/// @}

/////////////////////////////////////////////////////////////////////////////
// Definitions - DFU_CODE : Hardware Interface Errors
/////////////////////////////////////////////////////////////////////////////
/// @name DFU_CODE : Hardware Interface Errors
/// @{

/// The specified USB device could not be
/// found. Ensure that the specified device is
/// plugged in and not already in use. Call
/// ::DFU_GetNumDevices() to update the library device
/// list.
#define DFU_CODE_HWIF_DEVICE_NOT_FOUND          0x80

/// The current USB device is not opened.
/// Close and reopen the device before continuing.
#define DFU_CODE_HWIF_DEVICE_NOT_OPENED         0x81

/// An error occurred while communicating with the
/// device or while retrieving device information.
/// The device may be in an invalid state.
#define DFU_CODE_HWIF_DEVICE_ERROR              0x82

/// A USB control transfer request failed.
/// Verify that the device is in the appropriate
/// state before sending the request.
#define DFU_CODE_HWIF_TRANSFER_ERROR            0x83

/// A control transfer operation timed
/// out.
#define DFU_CODE_HWIF_TRANSFER_TIMEOUT          0x84

/// A control transfer operation returned with
/// less bytes transferred than requested
#define DFU_CODE_HWIF_TRANSFER_INCOMPLETE       0x85

/// @}

/////////////////////////////////////////////////////////////////////////////
// Type Definitions - DFU_STATUS
/////////////////////////////////////////////////////////////////////////////

/// @details A DFU_GETSTATUS status
///
/// @sa ::DFU_STATUS_OK
/// @sa ::DFU_STATUS_ERR_TARGET
/// @sa ::DFU_STATUS_ERR_FILE
/// @sa ::DFU_STATUS_ERR_WRITE
/// @sa ::DFU_STATUS_ERR_ERASE
/// @sa ::DFU_STATUS_ERR_CHECK_ERASED
/// @sa ::DFU_STATUS_ERR_PROG
/// @sa ::DFU_STATUS_ERR_VERIFY
/// @sa ::DFU_STATUS_ERR_ADDRESS
/// @sa ::DFU_STATUS_ERR_NOT_DONE
/// @sa ::DFU_STATUS_ERR_FIRMWARE
/// @sa ::DFU_STATUS_ERR_VENDOR
/// @sa ::DFU_STATUS_ERR_USBR
/// @sa ::DFU_STATUS_ERR_POR
/// @sa ::DFU_STATUS_ERR_UNKNOWN
/// @sa ::DFU_STATUS_ERR_STALLED_PKT
/// @sa ::DFU_STATUS_ERR_COMM
typedef BYTE DFU_STATUS;

/////////////////////////////////////////////////////////////////////////////
// Definitions - DFU_STATUS : Status Values
/////////////////////////////////////////////////////////////////////////////
/// @name DFU_STATUS : Status Values
/// @{

/// No error condition is
/// present.
#define DFU_STATUS_OK                           0x00

/// File is not targeted for use
/// by this device.
#define DFU_STATUS_ERR_TARGET                   0x01

/// File is for this device but fails some
/// vendor-specific verification test.
#define DFU_STATUS_ERR_FILE                     0x02

/// Device is unable to write
/// memory.
#define DFU_STATUS_ERR_WRITE                    0x03

/// Memory erase function
/// failed.
#define DFU_STATUS_ERR_ERASE                    0x04

/// Memory erase check
/// failed.
#define DFU_STATUS_ERR_CHECK_ERASED             0x05

/// Program memory function
/// failed.
#define DFU_STATUS_ERR_PROG                     0x06

/// Programmed memory failed
/// verification.
#define DFU_STATUS_ERR_VERIFY                   0x07

/// Cannot progrm memory due to received
/// address that is out of range.
#define DFU_STATUS_ERR_ADDRESS                  0x08

/// Received DFU_DNLOAD with wLength = 0, but
/// device does not think it has all of the data
/// yet.
#define DFU_STATUS_ERR_NOT_DONE                 0x09

/// Device's firmware is corrupt. It cannot return
/// to run-time (non-DFU) operations.
#define DFU_STATUS_ERR_FIRMWARE                 0x0A

/// iString indicates a vendor-specific
/// error.
#define DFU_STATUS_ERR_VENDOR                   0x0B

/// Device detected unexpected USB reset
/// signaling.
#define DFU_STATUS_ERR_USBR                     0x0C

/// Device detected unexpected power on
/// reset.
#define DFU_STATUS_ERR_POR                      0x0D

/// Something went wrong, but the device
/// does not know what it was.
#define DFU_STATUS_ERR_UNKNOWN                  0x0E

/// Device stalled an unexpected
/// request.
#define DFU_STATUS_ERR_STALLED_PKT              0x0F

/// Device encountered a communications
/// error
#define DFU_STATUS_ERR_COMM                     0x10

/// @}

/////////////////////////////////////////////////////////////////////////////
// Type Definitions - DFU_INTERFACE
/////////////////////////////////////////////////////////////////////////////

/// @details A DFU_INTERFACE type
///
/// @sa ::DFU_INTERFACE_UART
/// @sa ::DFU_INTERFACE_USB
typedef BYTE DFU_INTERFACE;

/////////////////////////////////////////////////////////////////////////////
// Definitions - DFU_INTERFACE : Interface Types
/////////////////////////////////////////////////////////////////////////////
/// @name DFU_INTERFACE : Interface Types
/// @{

/// Communications interface is over
/// USB
#define DFU_INTERFACE_USB                       0

/// Communications interface is over
/// UART
#define DFU_INTERFACE_UART                      1

/// @}

/////////////////////////////////////////////////////////////////////////////
// Type Definitions - DFU_DATA_BITS
/////////////////////////////////////////////////////////////////////////////

/// @details A DFU_DATA_BITS type
///
/// @sa ::DFU_FIVE_DATA_BITS
/// @sa ::DFU_SIX_DATA_BITS
/// @sa ::DFU_SEVEN_DATA_BITS
/// @sa ::DFU_EIGHT_DATA_BITS
typedef BYTE DFU_DATA_BITS;

/////////////////////////////////////////////////////////////////////////////
// Definitions - DFU_DATA_BITS
/////////////////////////////////////////////////////////////////////////////
/// @name DFU_DATA_BITS
/// @{

#define DFU_FIVE_DATA_BITS                      5
#define DFU_SIX_DATA_BITS                       6
#define DFU_SEVEN_DATA_BITS                     7
#define DFU_EIGHT_DATA_BITS                     8

/// @}

/////////////////////////////////////////////////////////////////////////////
// Type Definitions - DFU_PARITY
/////////////////////////////////////////////////////////////////////////////

/// @details A DFU_PARITY type
///
/// @sa ::DFU_NO_PARITY
/// @sa ::DFU_ODD_PARITY
/// @sa ::DFU_EVEN_PARITY
/// @sa ::DFU_MARK_PARITY
/// @sa ::DFU_SPACE_PARITY
typedef BYTE DFU_PARITY;

/////////////////////////////////////////////////////////////////////////////
// Definitions - DFU_PARITY
/////////////////////////////////////////////////////////////////////////////
/// @name DFU_PARITY
/// @{

#define DFU_NO_PARITY                           0
#define DFU_ODD_PARITY                          1
#define DFU_EVEN_PARITY                         2
#define DFU_MARK_PARITY                         3
#define DFU_SPACE_PARITY                        4

/// @}

/////////////////////////////////////////////////////////////////////////////
// Type Definitions - DFU_STOP_BITS
/////////////////////////////////////////////////////////////////////////////

/// @details A DFU_STOP_BITS type
///
/// @sa ::DFU_ONE_STOP_BIT
/// @sa ::DFU_ONE5_STOP_BITS
/// @sa ::DFU_TWO_STOP_BITS
typedef BYTE DFU_STOP_BITS;

/////////////////////////////////////////////////////////////////////////////
// Definitions - DFU_STOP_BITS
/////////////////////////////////////////////////////////////////////////////
/// @name DFU_STOP_BITS
/// @{

#define DFU_ONE_STOP_BIT                        0
#define DFU_ONE5_STOP_BITS                      1
#define DFU_TWO_STOP_BITS                       2

/// @}

/////////////////////////////////////////////////////////////////////////////
// Type Definitions - DFU_FLOW_CONTROL
/////////////////////////////////////////////////////////////////////////////

/// @details A DFU_FLOW_CONTROL type
///
/// @sa ::DFU_NO_FLOW_CONTROL
/// @sa ::DFU_RTS_CTS_FLOW_CONTROL
typedef BYTE DFU_FLOW_CONTROL;

/////////////////////////////////////////////////////////////////////////////
// Definitions - DFU_FLOW_CONTROL
/////////////////////////////////////////////////////////////////////////////
/// @name DFU_FLOW_CONTROL
/// @{

#define DFU_NO_FLOW_CONTROL                     0
#define DFU_RTS_CTS_FLOW_CONTROL                1

/// @}

/////////////////////////////////////////////////////////////////////////////
// Type Definitions - DFU_STATE
/////////////////////////////////////////////////////////////////////////////

/// @details A DFU_GETSTATUS state
///
/// @sa ::DFU_STATE_APP_IDLE
/// @sa ::DFU_STATE_APP_DETACH
/// @sa ::DFU_STATE_DFU_IDLE
/// @sa ::DFU_STATE_DFU_DNLOAD_SYNC
/// @sa ::DFU_STATE_DFU_DNBUSY
/// @sa ::DFU_STATE_DFU_DNLOAD_IDLE
/// @sa ::DFU_STATE_DFU_MANIFEST_SYNC
/// @sa ::DFU_STATE_DFU_MANIFEST
/// @sa ::DFU_STATE_DFU_MANIFEST_WAIT_RESET
/// @sa ::DFU_STATE_DFU_UPLOAD_IDLE
/// @sa ::DFU_STATE_DFU_ERROR
typedef BYTE DFU_STATE;

/////////////////////////////////////////////////////////////////////////////
// Definitions - DFU_STATE : State Values
/////////////////////////////////////////////////////////////////////////////
/// @name DFU_STATE : State Values
/// @{

/// Device is running its normal
/// application.
#define DFU_STATE_APP_IDLE                      0

/// Device is running its normal application, has
/// received the DFU_DETACH request, and is
/// waiting for a USB reset.
#define DFU_STATE_APP_DETACH                    1

/// Device is operating in the DFU mode and is
/// waiting for requests.
#define DFU_STATE_DFU_IDLE                      2

/// Device has received a block and is waiting
/// for the host to solicit the status via
/// DFU_GETSTATUS.
#define DFU_STATE_DFU_DNLOAD_SYNC               3

/// Device is programming a control-write block
/// into its nonvolatile memories.
#define DFU_STATE_DFU_DNBUSY                    4

/// Device is processing a download operation.
/// Expecting DFU_DNLOAD requests.
#define DFU_STATE_DFU_DNLOAD_IDLE               5

/// Device has received the final block of firmware
/// from the host and is waiting for receipt of
/// DFU_GETSTATUS to begin the Manifestation phase;
/// or device has completed the Manifestation phase
/// and is waiting for receipt of DFU_GETSTATUS.
/// (Devices that can enter this state after the
/// Manifestation phase set bmAttributes bit
/// bitManifestationTolerant to 1.)
#define DFU_STATE_DFU_MANIFEST_SYNC             6

/// Device is in the Manifestation phase.  (Not
/// all devices will be able to respond to
/// DFU_GETSTATUS when in this state.)
#define DFU_STATE_DFU_MANIFEST                  7

/// Device has programmed its memories and is
/// waiting for a USB reset or a power on reset.
/// (Devices that must enter this state clear
/// bitManifestationTolerant to 0.)
#define DFU_STATE_DFU_MANIFEST_WAIT_RESET       8

/// The device is processing an upload operation.
/// Expecting DFU_UPLOAD requests.
#define DFU_STATE_DFU_UPLOAD_IDLE               9

/// An error has occurred.  Awaiting the
/// DFU_CLRSTATUS request.
#define DFU_STATE_DFU_ERROR                     10

/// @}

/////////////////////////////////////////////////////////////////////////////
// String Definitions
/////////////////////////////////////////////////////////////////////////////

/// An ASCII, null-terminated device string used when retrieving a
/// device path string
typedef char DFU_DEVICE_STR[MAX_PATH];

/////////////////////////////////////////////////////////////////////////////
// Type Definitions
/////////////////////////////////////////////////////////////////////////////

/// A DFU device object pointer used to access a DFU class
// USB device
typedef void* DFU_DEVICE;

/// @details A progress callback used to indicate progress of time-lengthy
/// operations.
///
/// @param percent Indicates the progress in percent between 0 and 100.
/// @param workInProgress If @c TRUE, then this parameter indicates that the
/// operation is still in progress.
/// @param blockID Indicates the block ID of the last block transferred. A
/// value of -1 indicates that the operation has just started or has just
/// completed (successfully or in error).
/// @param userData The user-parameter passed to the callback function.
/// @return Return @c FALSE to abort the operation. Otherwise, return @c TRUE
/// to continue the operation.
///
/// @remark If the user-callback returns FALSE, then the time-lengthy function
/// returns ::DFU_CODE_API_ABORTED.
///
/// @sa ::DFU_DownloadImage()
/// @sa ::DFU_UploadImage()
typedef BOOL (CALLBACK *DFU_PROGRESS_CALLBACK)(int percent, BOOL workInProgress, int blockID, void* userData);

/////////////////////////////////////////////////////////////////////////////
// Structures
/////////////////////////////////////////////////////////////////////////////

/// The setup packet structure for a USB
/// control request
typedef struct DFU_CONTROL_SETUP_PACKET
{
    BYTE RequestType;
    BYTE Request;
    WORD Value;
    WORD Index;
    WORD Length;

} DFU_CONTROL_SETUP_PACKET, *PDFU_CONTROL_SETUP_PACKET;

/// The data structure returned from the
/// DFU_GETSTATUS request
typedef struct DFU_GET_STATUS
{
    /// An indication of the status resulting from the
    /// execution of the most recent request
    DFU_STATUS bStatus;

    /// Minimum time, in milliseconds, that the host
    /// should wait before sending a subsequent
    /// DFU_GETSTATUS request
    DWORD bwPollTimeout;

    /// An indication of the state that the device is
    /// going to enter immediately following transmission
    /// of this response. (By the time the host receives
    /// this information, this is the current state of the
    /// device.)
    DFU_STATE bState;

    /// Index of status description in string
    /// table
    BYTE iString;

} DFU_GET_STATUS, *PDFU_GET_STATUS;

/// The data structure returned from the
/// SLAB_GETINFO request
typedef struct DFU_GET_INFO
{
    /// The total length of the SLAB_GETINFO structure
    /// including the length field.
    WORD length;

    /// The bootloader version in binary coded decimal
    /// (example: 0x01, 0x00 = 1.00)
    BYTE bBootloaderRevision_Major;
    BYTE bBootloaderRevision_Minor;

    /// See DFU functional descriptor definition
    /// in the DFU 1.1 Specification
    DWORD bmAttributes;

    /// The number of bytes that can be transferred
    /// using DFU_DNLOAD and DFU_UPLOAD, or the
    /// block size
    DWORD wBlockSize;

    /// The maximum amount of code space reserved for the
    /// application in bytes. This is the maximum DFU
    /// image size.
    DWORD maxAppSize;

    /// Device identification register (DEVICEID0)
    /// value from the device.
    BYTE bDeviceID[16];
    
    /// Universally Unique Identifier (UUID)
    /// value from the device.
    BYTE bUUID[16];

} DFU_GET_INFO, *PDFU_GET_INFO;

/// The USB device descriptor
typedef struct DFU_DEVICE_DESCRIPTOR
{
    BYTE bLength;
    BYTE bDescriptorType;
    WORD bcdUSB;
    BYTE bDeviceClass;
    BYTE bDeviceSubClass;
    BYTE bDeviceProtocol;
    BYTE bMaxPacketSize0;
    WORD idVendor;
    WORD idProduct;
    WORD bcdDevice;
    BYTE iManufacturer;
    BYTE iProduct;
    BYTE iSerialNumber;
    BYTE bNumConfigurations;

} DFU_DEVICE_DESCRIPTOR, *PDFU_DEVICE_DESCRIPTOR;

/// The USB interface descriptor
typedef struct DFU_INTERFACE_DESCRIPTOR
{
    BYTE bLength;
    BYTE bDescriptorType;
    BYTE bInterfaceNumber;
    BYTE bAlternateSetting;
    BYTE bNumEndpoints;
    BYTE bInterfaceClass;
    BYTE bInterfaceSubClass;
    BYTE bInterfaceProtocol;
    BYTE iInterface;

} DFU_INTERFACE_DESCRIPTOR, *PDFU_INTERFACE_DESCRIPTOR;

/// The DFU class functional descriptor
typedef struct DFU_FUNCTIONAL_DESCRIPTOR
{
    BYTE bLength;
    BYTE bDescriptorType;
    BYTE bmAttributes;
    WORD wDetachTimeOut;
    WORD wTransferSize;
    WORD bcdDFUVersion;

} DFU_FUNCTIONAL_DESCRIPTOR, *PDFU_FUNCTIONAL_DESCRIPTOR;

/////////////////////////////////////////////////////////////////////////////
// Exported Library Functions
/////////////////////////////////////////////////////////////////////////////

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

/////////////////////////////////////////////////////////////////////////////
// Device Management
/////////////////////////////////////////////////////////////////////////////
/// @name Device Management Functions
/// @{

SLAB_DFU_API DFU_CODE WINAPI DFU_GetGuid(LPGUID guid);
SLAB_DFU_API DFU_CODE WINAPI DFU_SetGuid(LPGUID guid);
SLAB_DFU_API DFU_CODE WINAPI DFU_GetNumDevices(DWORD* numDevices);
SLAB_DFU_API DFU_CODE WINAPI DFU_GetDevicePath(DWORD deviceIndex, LPSTR path);
SLAB_DFU_API DFU_CODE WINAPI DFU_GetVidPid(DWORD deviceIndex, WORD* vid, WORD* pid);
SLAB_DFU_API DFU_CODE WINAPI DFU_GetInterfaceByIndex(DWORD deviceIndex, DFU_INTERFACE* type);
SLAB_DFU_API DFU_CODE WINAPI DFU_GetInterfaceByDevicePath(LPCSTR path, DFU_INTERFACE* type);
SLAB_DFU_API DFU_CODE WINAPI DFU_GetOpenedDevicePath(DFU_DEVICE device, LPSTR path);
SLAB_DFU_API DFU_CODE WINAPI DFU_GetOpenedVidPid(DFU_DEVICE device, WORD* vid, WORD* pid);
SLAB_DFU_API DFU_CODE WINAPI DFU_GetOpenedInterface(DFU_DEVICE device, DFU_INTERFACE* type);

SLAB_DFU_API DFU_CODE WINAPI DFU_OpenByIndex(DFU_DEVICE* device, DWORD deviceIndex);
SLAB_DFU_API DFU_CODE WINAPI DFU_OpenByDevicePath(DFU_DEVICE* device, LPCSTR path);
SLAB_DFU_API DFU_CODE WINAPI DFU_Close(DFU_DEVICE device);
SLAB_DFU_API BOOL WINAPI DFU_IsOpened(DFU_DEVICE device);
/// @}

/////////////////////////////////////////////////////////////////////////////
// Information
/////////////////////////////////////////////////////////////////////////////
/// @name Information Functions
/// @{

SLAB_DFU_API DFU_CODE WINAPI DFU_GetLibraryVersion(BYTE* major, BYTE* minor, BOOL* release);

SLAB_DFU_API DFU_CODE WINAPI DFU_GetDeviceDescriptor(DFU_DEVICE device, PDFU_DEVICE_DESCRIPTOR descriptor);
SLAB_DFU_API DFU_CODE WINAPI DFU_GetInterfaceDescriptor(DFU_DEVICE device, PDFU_INTERFACE_DESCRIPTOR descriptor);
/// @}

/////////////////////////////////////////////////////////////////////////////
// Control Transfers
/////////////////////////////////////////////////////////////////////////////
/// @name Control Transfer Functions
/// @{

SLAB_DFU_API DFU_CODE WINAPI DFU_ControlTransfer(DFU_DEVICE device, PDFU_CONTROL_SETUP_PACKET setup, BYTE buffer[], DWORD size, DWORD* bytesTransferred);
SLAB_DFU_API DFU_CODE WINAPI DFU_SetTransferTimeout(DFU_DEVICE device, DWORD timeout);
SLAB_DFU_API DFU_CODE WINAPI DFU_GetTransferTimeout(DFU_DEVICE device, DWORD* timeout);

SLAB_DFU_API DFU_CODE WINAPI DFU_SetUartConfig(DFU_DEVICE device, DWORD baud, DFU_DATA_BITS dataBits, DFU_PARITY parity, DFU_STOP_BITS stopBits, DFU_FLOW_CONTROL flowControl);
SLAB_DFU_API DFU_CODE WINAPI DFU_GetUartConfig(DFU_DEVICE device, DWORD* baud, DFU_DATA_BITS* dataBits, DFU_PARITY* parity, DFU_STOP_BITS* stopBits, DFU_FLOW_CONTROL* flowControl);

/// @}

/////////////////////////////////////////////////////////////////////////////
// DFU Mode
/////////////////////////////////////////////////////////////////////////////
/// @name DFU Mode Functions
/// @{

SLAB_DFU_API DFU_CODE WINAPI DFU_DfuDnload(DFU_DEVICE device, BYTE buffer[], WORD size, WORD block, WORD* transferred);
SLAB_DFU_API DFU_CODE WINAPI DFU_DfuUpload(DFU_DEVICE device, BYTE buffer[], WORD size, WORD block, WORD* transferred);
SLAB_DFU_API DFU_CODE WINAPI DFU_DfuGetStatus(DFU_DEVICE device, PDFU_GET_STATUS status);
SLAB_DFU_API DFU_CODE WINAPI DFU_DfuClearStatus(DFU_DEVICE device);
SLAB_DFU_API DFU_CODE WINAPI DFU_DfuGetState(DFU_DEVICE device, DFU_STATE* state);    
SLAB_DFU_API DFU_CODE WINAPI DFU_DfuAbort(DFU_DEVICE device);
SLAB_DFU_API DFU_CODE WINAPI DFU_SlabGetInfo(DFU_DEVICE device, PDFU_GET_INFO info);
SLAB_DFU_API DFU_CODE WINAPI DFU_SlabReset(DFU_DEVICE device);

SLAB_DFU_API DFU_CODE WINAPI DFU_DownloadImage(DFU_DEVICE device, WORD blockSize, LPCSTR fileName, DFU_PROGRESS_CALLBACK callback, void* userData, PDFU_GET_STATUS status);
SLAB_DFU_API DFU_CODE WINAPI DFU_UploadImage(DFU_DEVICE device, WORD blockSize, LPCSTR fileName, DFU_PROGRESS_CALLBACK callback, void* userData);

/// @}

#ifdef __cplusplus
}
#endif // __cplusplus

#endif // SLAB_DFU_H
