/*****************************************************************************
 * @file cmu_common.c
 * @brief Common functions for CMU application note
 * @version  1.12

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "main_cmu_example.h"

/* External variables */
extern volatile bool runKey;
extern volatile bool menuKey;
extern volatile bool progRun;
extern volatile uint8_t menuLevel;

/**********************************************************
 * @brief Interrupt handler for right side push button. 
 **********************************************************/
#if defined(_EFM32_TINY_FAMILY) || defined(_EFM32_HAPPY_FAMILY)
void GPIO_EVEN_IRQHandler(void)
#else
void GPIO_ODD_IRQHandler(void)
#endif
{
  GPIO->IFC = GPIO->IF;

  /* Next menu item if no example is running */
  if (!progRun && !menuKey)
  {
    menuKey = true;
    if (++menuLevel == MENU_MAX)
    {
      menuLevel = LFRCO_CAL;
    }
  }
  else
  {
    progRun = false;
  }    
}

/**********************************************************
 * @brief Interrupt handler for left side push button. 
 **********************************************************/
#if defined(_EFM32_TINY_FAMILY) || defined(_EFM32_HAPPY_FAMILY)
void GPIO_ODD_IRQHandler(void)
#else
void GPIO_EVEN_IRQHandler(void)
#endif
{
  GPIO->IFC = GPIO->IF;

  /* No action if example is running */
  if (!progRun)
  {
    runKey = true;
  }
}

/**************************************************************************//**
 * @brief Setup GPIO interrupt for push buttons.
 *****************************************************************************/
void gpioSetup(void)
{
  /* Enable GPIO clock */
  CMU_ClockEnable(cmuClock_GPIO, true);

  /* Configure push button 0 as input and enable interrupt  */
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, BSP_GPIO_PB0_PIN, false, true, true);

  /* Configure push button 1 as input and enable interrupt */
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, BSP_GPIO_PB1_PIN, false, true, true);
  
  /* Enable interrupt in core for even and odd GPIO interrupts */
  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);

  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
  
  /* Configure CMU output if enable */
#if CMU_OUT_EN == 1
#if defined(_EFM32_GECKO_FAMILY) || (_EFM32_TINY_FAMILY)
  GPIO_PinModeSet(CMU_OUT0_PORT, CMU_OUT0_PIN, gpioModePushPull, 0);
#endif
#if !defined(_EFM32_TINY_FAMILY)
  GPIO_PinModeSet(CMU_OUT1_PORT, CMU_OUT1_PIN, gpioModePushPull, 0);
#endif
#if defined(_CMU_ROUTEPEN_MASK)
  CMU->ROUTEPEN = CMU_OUT_PEN;
  CMU->ROUTELOC0 = CMU_OUT_LOC;
#else
  CMU->ROUTE = CMU_OUT_PEN_LOC;
#endif
#endif  
}

#if defined(_PRS_CH_CTRL_SOURCESEL_CMU)
/**************************************************************************//**
 * @brief Toggle Memory LCD external COM inversion signal pin automatically.
 *        Pin toggle is achieved by output ULFRCO to one of the PRS channel
 *        outputs.
 *
 * @param[in] gpioPort  Do not care.
 * @param[in] gpioPin   Do not care.
 * @param[in] frequency Do not care.
 *
 * @return  EMSTATUS code of the operation.
 *****************************************************************************/
EMSTATUS PAL_GpioPinAutoToggle (unsigned int gpioPort,
                                unsigned int gpioPin,
                                unsigned int frequency)
{
  CMU->CTRL |= CMU_CTRL_CLKOUTSEL0_ULFRCOQ;											
  CMU_ClockEnable(cmuClock_PRS, true);  /* Can work on EM2/3 with Async PRS */											
  PRS_SourceAsyncSignalSet(LCD_AUTO_TOGGLE_PRS_CH, PRS_CH_CTRL_SOURCESEL_CMU, PRS_CH_CTRL_SIGSEL_CMUCLKOUT0);											
  LCD_AUTO_TOGGLE_PRS_ROUTELOC();											
  PRS->ROUTEPEN |= LCD_AUTO_TOGGLE_PRS_ROUTEPEN;											
  CMU_ClockEnable(cmuClock_PRS, false);											

  EMSTATUS status = EMSTATUS_OK;
  return status;
}
#endif

/***************************************************************************//**
* @brief Print the menu.
*******************************************************************************/
void printMenu(void)
{
#if defined(USE_SEGMENT_LCD)
  switch (menuLevel)
  {
  case LFRCO_CAL:
    SegmentLCD_Write("LFRCO");
    break;
    
  case HFRCO_CAL:
    SegmentLCD_Write("HFRCO");
    break;
    
  case AUXHFRCO_CAL:
    SegmentLCD_Write("AUXRCO");
    break;
    
  case USHFRCO_CAL:
    SegmentLCD_Write("USRCO");
    break;
    
  default:
    break;
  }
  LCD_NUMBER_OFF();
  menuKey = false;
#else
  printf("\f");
  printf("Example %d\n", menuLevel+1);
  
  switch (menuLevel)
  {
  case LFRCO_CAL:
    printf("HFXO to Calibrate\n");
    printf("LFRCO\n\n"); 
    break;
    
  case HFRCO_CAL:
    printf("LFXO to Calibrate\n");
    printf("HFRCO\n\n"); 
    break;
    
  case AUXHFRCO_CAL:
    printf("LFXO to Calibrate\n");
    printf("AUXHFRCO\n\n"); 
    break;
    
  case USHFRCO_CAL:
    printf("LFXO to Calibrate\n");
    printf("USHFRCO\n\n"); 
    break;

  default:
    break;
  }
  printf(NEXT_BUTTON); 
  printf("menu");
  printf(RUN_BUTTON); 
  menuKey = false;
#endif
}
