/**************************************************************************//**
 * @file main_cmu_example.c
 * @brief CMU examples for EFM32 Gecko Series 0 and 1
 * @version  1.12

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "main_cmu_example.h"

/* Global variables */
volatile bool runKey;
volatile bool menuKey;
volatile bool progRun;
volatile uint8_t menuLevel;

/**************************************************************************//**
 * @brief Main function
 *****************************************************************************/
int main(void)
{
#if !defined(USE_SEGMENT_LCD)
  /* Tune counter to display on Memory LCD, not used in segment LCD */
  uint8_t tuneCount = 0;
#endif
  
  /* Chip errata */
  CHIP_Init();

#if defined( _EMU_DCDCCTRL_MASK )  
  /* Initialize DCDC regulator if available */
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif

  /* Initialize HFXO with specific parameters */
  CMU_HFXOInit_TypeDef hfxoInit = CMU_HFXOINIT_DEFAULT;
  CMU_HFXOInit(&hfxoInit);

  /* Initialize LFXO with specific parameters */
  CMU_LFXOInit_TypeDef lfxoInit = CMU_LFXOINIT_DEFAULT;
  CMU_LFXOInit(&lfxoInit);
  
  /* Optimize flash wait state with default HFRCO as HFCLK */
#if defined( _CMU_HFRCOCTRL_BAND_MASK )
  CMU_HFRCOBandSet(cmuHFRCOBand_14MHz);
#else
  CMU_HFRCOBandSet(cmuHFRCOFreq_19M0Hz);
#endif
  
#if defined (_MSC_CTRL_IFCREADCLEAR_MASK)  
  /* Enable atomic read-clear operation on reading IFC register */
  MSC->CTRL |= MSC_CTRL_IFCREADCLEAR;
#endif
  
  /* Setup GPIO with interrupts to serve the pushbuttons */
  gpioSetup();

#if defined(USE_SEGMENT_LCD)
  /* Initialize LCD, use LFRCO for LFA */
  SegmentLCD_Init(false);
#else
  /* Initialize the display module. */
  DISPLAY_Init();

  /* Retarget stdio to a text display. */
  if (RETARGET_TextDisplayInit() != TEXTDISPLAY_EMSTATUS_OK)
  {
    /* Text display initialization failed. */
    while (1) ;
  }
#endif
  
  /* Display menu */
  printMenu();
  
  /* Enter EM1 or EM2 to wait key press, EM1 is for RC clock output */
  while (1)
  {
#if CMU_OUT_EN == 1
    EMU_EnterEM1();
#else
    EMU_EnterEM2(false);
#endif
    
    if (menuKey)         /* Left side push button to browse menu */
    {
      printMenu();
#if !defined(USE_SEGMENT_LCD)
      tuneCount = 0;
#endif
    }

    if (runKey)         /* Right side push button to run selected menu */
    {
      runKey = false;
      progRun = true;
#if !defined(USE_SEGMENT_LCD)
      printf(TEXTDISPLAY_ESC_SEQ_CURSOR_HOME_VT100);
      printf("\n\n\n\n");
#endif
      
      switch (menuLevel)
      {
      case LFRCO_CAL:
        /* Display the existing tuning value, then start calibration */
#if defined(USE_SEGMENT_LCD)
        SegmentLCD_Write("LF TUNE");
        SegmentLCD_Number(CMU_OscillatorTuningGet(cmuOsc_LFRCO));
#else
        printf("Old tune value: %3lu\n", CMU_OscillatorTuningGet(cmuOsc_LFRCO));
        printf("                     \n");
        printf("                     \n");
        printf(TEXTDISPLAY_ESC_SEQ_CURSOR_UP_ONE_LINE TEXTDISPLAY_ESC_SEQ_CURSOR_UP_ONE_LINE);
#endif
        calibrateRcOsc(cmuOsc_LFRCO, LFRCO_TUNE_FREQ, false, false);
#if !defined(USE_SEGMENT_LCD)
        printf("UpCnt Target: %lu\n\n", getUpcount(true));
#endif

        while(!endOfTune())     /* Display the tuning result if finish */
          ;
#if defined(USE_SEGMENT_LCD)
        SegmentLCD_Write("LF DONE");
        SegmentLCD_Number(CMU_OscillatorTuningGet(cmuOsc_LFRCO));
#else
        printf("Tune count: %3d    \n", ++tuneCount);
        printf("UpCnt Actual: %lu\n", getUpcount(false));
        printf("New tune value: %3lu\n", CMU_OscillatorTuningGet(cmuOsc_LFRCO));
        printf(RUN_BUTTON); 
        printf(EXIT_BUTTON); 
#endif
        progRun = false;
        break;
        
      case HFRCO_CAL:
        /* Get tuning value from DI page if existing band != tuning band */
        if (CMU_HFRCOBandGet() != HFRCO_TUNE_BAND)
        {
          CMU_HFRCOBandSet(HFRCO_TUNE_BAND);
        }
        /* Display the existing tuning value, then start calibration */
#if defined(USE_SEGMENT_LCD)
        SegmentLCD_Write("HF TUNE");
        SegmentLCD_Number(CMU_OscillatorTuningGet(cmuOsc_HFRCO));
#else
        printf("Old tune value: %3lu\n", CMU_OscillatorTuningGet(cmuOsc_HFRCO));
        printf("                     \n");
        printf("                     \n");
        printf(TEXTDISPLAY_ESC_SEQ_CURSOR_UP_ONE_LINE TEXTDISPLAY_ESC_SEQ_CURSOR_UP_ONE_LINE);
#endif
        calibrateRcOsc(cmuOsc_HFRCO, HFRCO_TUNE_FREQ, false, true);
#if !defined(USE_SEGMENT_LCD)
#if defined(_CMU_HFRCOCTRL_FINETUNINGEN_MASK)
        if ((CMU->HFRCOCTRL & CMU_HFRCOCTRL_FINETUNINGEN) != 0)
        {
          printf("Old finetune val: %3lu\n", CMU_OscillatorFineTuningGet(cmuOsc_HFRCO));
        }
#endif
        printf("UpCnt Target: %lu\n\n", getUpcount(true));
#endif        

        while(!endOfTune())     /* Display the tuning result if finish */
          ;
#if defined(USE_SEGMENT_LCD)
        SegmentLCD_Write("HF DONE");
        SegmentLCD_Number(CMU_OscillatorTuningGet(cmuOsc_HFRCO));
#else
        printf("Tune count: %3d\n", ++tuneCount);
        printf("UpCnt Actual: %lu\n", getUpcount(false));
        printf("New tune value: %3lu\n", CMU_OscillatorTuningGet(cmuOsc_HFRCO));
#if defined(_CMU_HFRCOCTRL_FINETUNINGEN_MASK)
        if ((CMU->HFRCOCTRL & CMU_HFRCOCTRL_FINETUNINGEN) != 0)
        {
          printf("New finetune val: %3lu\n", CMU_OscillatorFineTuningGet(cmuOsc_HFRCO));
        }
#endif
        printf(RUN_BUTTON); 
        printf(EXIT_BUTTON); 
#endif
        progRun = false;
        break;
        
      case AUXHFRCO_CAL:
#if defined( _CMU_AUXHFRCOCTRL_BAND_MASK ) || defined( _CMU_AUXHFRCOCTRL_FREQRANGE_MASK )
        /* Get tuning value from DI page if existing band != tuning band */
        if (CMU_AUXHFRCOBandGet() != AUXHFRCO_TUNE_BAND)
        {
          CMU_AUXHFRCOBandSet(AUXHFRCO_TUNE_BAND);
        }
#endif
        /* Display the existing tuning value, then start calibration */
#if defined(USE_SEGMENT_LCD)
        SegmentLCD_Write("AU TUNE");
        SegmentLCD_Number(CMU_OscillatorTuningGet(cmuOsc_AUXHFRCO));
#else
        printf("Old tune value: %3lu\n", CMU_OscillatorTuningGet(cmuOsc_AUXHFRCO));
        printf("                     \n");
        printf("                     \n");
        printf(TEXTDISPLAY_ESC_SEQ_CURSOR_UP_ONE_LINE TEXTDISPLAY_ESC_SEQ_CURSOR_UP_ONE_LINE);
#endif
        calibrateRcOsc(cmuOsc_AUXHFRCO, AUXHFRCO_TUNE_FREQ, false, true);
#if !defined(USE_SEGMENT_LCD)
#if defined(_CMU_AUXHFRCOCTRL_FINETUNINGEN_MASK)
        if ((CMU->AUXHFRCOCTRL & CMU_AUXHFRCOCTRL_FINETUNINGEN) != 0)
        {
          printf("Old finetune val: %3lu\n", CMU_OscillatorFineTuningGet(cmuOsc_AUXHFRCO));
        }
#endif
        printf("UpCnt Target: %lu\n\n", getUpcount(true));
#endif        

        while(!endOfTune())     /* Display the tuning result if finish */
          ;
#if defined(USE_SEGMENT_LCD)
        SegmentLCD_Write("AU DONE");
        SegmentLCD_Number(CMU_OscillatorTuningGet(cmuOsc_AUXHFRCO));
#else
        printf("Tune count: %3d\n", ++tuneCount);
        printf("UpCnt Actual: %lu\n", getUpcount(false));
        printf("New tune value: %3lu\n", CMU_OscillatorTuningGet(cmuOsc_AUXHFRCO));
#if defined(_CMU_AUXHFRCOCTRL_FINETUNINGEN_MASK)
        if ((CMU->AUXHFRCOCTRL & CMU_AUXHFRCOCTRL_FINETUNINGEN) != 0)
        {
          printf("New finetune val: %3lu\n", CMU_OscillatorFineTuningGet(cmuOsc_AUXHFRCO));
        }
#endif
        printf(RUN_BUTTON); 
        printf(EXIT_BUTTON); 
#endif
        progRun = false;
        break;
        
#if defined( _CMU_STATUS_USHFRCOENS_MASK )
      case USHFRCO_CAL:
        /* Get tuning value from DI page if existing band != tuning band */
        if (CMU_USHFRCOBandGet() != USHFRCO_TUNE_BAND)
        {
          CMU_USHFRCOBandSet(USHFRCO_TUNE_BAND);  
        }
        /* Display the existing tuning value, then start calibration */
        printf("Old tune value: %3lu\n", USHFRCO_TUNE_GET);
        printf("Old finetune val: %3lu\n", CMU_OscillatorFineTuningGet(cmuOsc_USHFRCO));
        printf("                     \n");                     
        printf(TEXTDISPLAY_ESC_SEQ_CURSOR_UP_ONE_LINE);
        calibrateRcOsc(cmuOsc_USHFRCO, USHFRCO_TUNE_FREQ, false, true);
        printf("UpCnt Target: %lu\n\n", getUpcount(true));

        while(!endOfTune())     /* Display the tuning result if finish */
          ;
        printf("Tune count: %3d    \n", ++tuneCount);
        printf("UpCnt Actual: %lu\n", getUpcount(false));
        printf("New tune value: %3lu\n", USHFRCO_TUNE_GET);
        printf("New finetune val: %3lu\n", CMU_OscillatorFineTuningGet(cmuOsc_USHFRCO));
        printf(RUN_BUTTON); 
        printf(EXIT_BUTTON); 
        progRun = false;
        break;
#endif
        
      default:
        break;
      }
    }
  }
}
