/*****************************************************************************
 * @file cmu_common.c
 * @brief Common functions for CMU application note
 * @version  1.13

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "main_cmu_example_s1.h"

// External variables
extern volatile bool runKey;
extern volatile bool menuKey;
extern volatile bool progRun;
extern volatile uint8_t menuLevel;

/**********************************************************
 * @brief Interrupt handler for BTN0
 **********************************************************/
void GPIO_ODD_IRQHandler(void)
{
  GPIO->IFC = GPIO->IF;

  // Next menu item if no example is running
  if (!progRun && !menuKey)
  {
    menuKey = true;
    if (++menuLevel == MENU_MAX)
    {
      menuLevel = LFRCO_CAL;
    }
  }
  else
  {
    progRun = false;
  }    
}

/**********************************************************
 * @brief Interrupt handler for BTN1
 **********************************************************/
void GPIO_EVEN_IRQHandler(void)
{
  GPIO->IFC = GPIO->IF;

  // No action if example is running
  if (!progRun)
  {
    runKey = true;
  }
}

/**************************************************************************//**
 * @brief GPIO pin and interrupt configuration for push buttons
 *****************************************************************************/
void gpioSetup(void)
{
  // Enable GPIO clock
  CMU_ClockEnable(cmuClock_GPIO, true);

  // Configure button 0 pin as an input and enable interrupt
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, BSP_GPIO_PB0_PIN, false, true, true);

  // Configure button 1 pin as an input and enable interrupt
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, BSP_GPIO_PB1_PIN, false, true, true);
  
  // Clear pending NVIC GPIO interrupts and enable
  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);

  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
  
  // Configure CMU output, if enabled
#ifdef CMU_OUT_EN
  GPIO_PinModeSet(CMU_OUT_PORT, CMU_OUT_PIN, gpioModePushPull, 0);
  CMU->ROUTEPEN = CMU_OUT_PEN;
  CMU->ROUTELOC0 = CMU_OUT_LOC;
#endif  
}

/***************************************************************************//**
* @brief Print the menu.
*******************************************************************************/
void printMenu(void)
{
  printf("\f");
  printf("Calibration Example %d\n\n", menuLevel+1);
  
  switch (menuLevel)
  {
  case LFRCO_CAL:
    printf("LFRCO against HFXO\n\n");
    break;
    
  case HFRCO_CAL:
    printf("HFRCO against LFXO\n\n");
    break;
    
  case AUXHFRCO_CAL:
    printf("AUXHFRCO against LFXO\n\n");
    break;
    
  case USHFRCO_CAL:
    printf("USHFRCO against LFXO\n\n");
    break;

  default:
    break;
  }

  printf(NEXT_BUTTON); 
  printf(START_BUTTON);
  menuKey = false;
}
