/**************************************************************************//**
 * @file main_cmu_example_s1.c
 * @brief CMU Examples for EFR32/EFM32 Series 1
 * @version  1.13

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "main_cmu_example_s1.h"

// Global variables
volatile bool runKey;
volatile bool menuKey;
volatile bool progRun;
volatile uint8_t menuLevel;

/**************************************************************************//**
 * @brief Main function
 *****************************************************************************/
int main(void)

  {
  // Tuning count
  uint8_t tuneCount = 0;

  // Chip errata
  CHIP_Init();

#if defined( _EMU_DCDCCTRL_MASK )
  /*
   * Initialize DC-DC converter, if present.  Applies to Pearl, Blue,
   * Flex, and Mighty Gecko devices.
   */
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif

  // Initialize HFXO with safe default parameters
  CMU_HFXOInit_TypeDef hfxoInit = CMU_HFXOINIT_DEFAULT;
  CMU_HFXOInit(&hfxoInit);

  // Initialize LFXO with safe default parameters
  CMU_LFXOInit_TypeDef lfxoInit = CMU_LFXOINIT_DEFAULT;
  CMU_LFXOInit(&lfxoInit);

  /*
   * Set HFRCO to default band.  This has the side effect of setting the
   * number of flash wait states to the optimal value vs. the default out
   * of power-on reset, which is 1 wait state.
   */
  CMU_HFRCOBandSet(cmuHFRCOFreq_19M0Hz);

  // Enable atomic read-clear operation for reads of IFC registers
  MSC->CTRL |= MSC_CTRL_IFCREADCLEAR;

  // Setup GPIO with interrupts to enable push buttons
  gpioSetup();

  // Initialize display
  DISPLAY_Init();

  // Retarget STDIO to display
  if (RETARGET_TextDisplayInit() != TEXTDISPLAY_EMSTATUS_OK)
  {
    // Text display initialization failed
    __BKPT(0);
  }
  
  // Display demo menu
  printMenu();

  //Wait for button press in EM2, or EM1 is clock output is enabled.
  while (1)
  {
#ifdef CMU_OUT_EN
    EMU_EnterEM1();
#else
    EMU_EnterEM2(false);
#endif

    // BTN1 rotates through menu items
    if (menuKey)
    {
      printMenu();
      tuneCount = 0;
    }

    // BTN0 runs calibration for the selected oscillator
    if (runKey)
    {
      runKey = false;
      progRun = true;

      switch (menuLevel)
      {
      case LFRCO_CAL:
        // Display current tuning value, then run calibration
        printf("\nOld tuning value: %lu\n", CMU_OscillatorTuningGet(cmuOsc_LFRCO));
        calibrateRcOsc(cmuOsc_LFRCO, LFRCO_TUNE_FREQ, false, false);
        printf("Up count target: %lu\n\n", getUpcount(true));

        // Display result when finished
        while(!endOfTune());

        printf("Tuning count: %u\n", ++tuneCount);
        printf("Up count value: %lu\n", getUpcount(false));
        printf("New tuning value: %lu\n", CMU_OscillatorTuningGet(cmuOsc_LFRCO));

        printf(AGAIN_BUTTON);
        printf(EXIT_BUTTON);

        progRun = false;
        break;

      case HFRCO_CAL:
        // Get tuning value from DI page if existing band != tuning band
        if (CMU_HFRCOBandGet() != HFRCO_TUNE_BAND)
          CMU_HFRCOBandSet(HFRCO_TUNE_BAND);

        // Display current tuning value, then run calibration
        printf("\nOld tuning value: %lu\n", CMU_OscillatorTuningGet(cmuOsc_HFRCO));
        printf("Old fine tuning\nvalue: %lu\n", CMU_OscillatorFineTuningGet(cmuOsc_AUXHFRCO));

        calibrateRcOsc(cmuOsc_HFRCO, HFRCO_TUNE_FREQ, false, true);
        printf("Up count target: %lu\n\n", getUpcount(true));

        // Display result when finished
        while(!endOfTune());

        printf("Tuning count: %u\n", ++tuneCount);
        printf("Up count value: %lu\n", getUpcount(false));
        printf("New tuning value: %lu\n", CMU_OscillatorTuningGet(cmuOsc_HFRCO));
        if ((CMU->HFRCOCTRL & CMU_HFRCOCTRL_FINETUNINGEN) != 0)
          printf("New fine tuning\nvalue: %lu\n", CMU_OscillatorFineTuningGet(cmuOsc_HFRCO));

        printf(AGAIN_BUTTON);
        printf(EXIT_BUTTON);

        progRun = false;
        break;

      case AUXHFRCO_CAL:

        // Get tuning value from DI page if existing band != tuning band
        if (CMU_AUXHFRCOBandGet() != AUXHFRCO_TUNE_BAND)
          CMU_AUXHFRCOBandSet(AUXHFRCO_TUNE_BAND);

        // Display current tuning value, then run calibration
        printf("\nOld tuning value: %lu\n", CMU_OscillatorTuningGet(cmuOsc_AUXHFRCO));
        printf("Old fine tuning\nvalue: %lu\n", CMU_OscillatorFineTuningGet(cmuOsc_AUXHFRCO));

        calibrateRcOsc(cmuOsc_AUXHFRCO, AUXHFRCO_TUNE_FREQ, false, true);
        printf("Up count target: %lu\n\n", getUpcount(true));

        // Display result when finished
        while(!endOfTune());

        printf("Tuning count: %u\n", ++tuneCount);
        printf("Up count value: %lu\n", getUpcount(false));
        printf("New tuning value: %lu\n", CMU_OscillatorTuningGet(cmuOsc_AUXHFRCO));
        if ((CMU->AUXHFRCOCTRL & CMU_AUXHFRCOCTRL_FINETUNINGEN) != 0)
          printf("New fine tuning\nvalue: %lu\n", CMU_OscillatorFineTuningGet(cmuOsc_AUXHFRCO));

        printf(AGAIN_BUTTON);
        printf(EXIT_BUTTON);

        progRun = false;
        break;

#if defined( _CMU_STATUS_USHFRCOENS_MASK )
      case USHFRCO_CAL:
        // Get tuning value from DI page if existing band != tuning band
        if (CMU_USHFRCOBandGet() != USHFRCO_TUNE_BAND)
          CMU_USHFRCOBandSet(USHFRCO_TUNE_BAND);

        // Display current tuning value, then run calibration
        printf("\nOld tuning value: %lu\n", USHFRCO_TUNE_GET);
        printf("Old fine tuning value: %lu\n", CMU_OscillatorFineTuningGet(cmuOsc_USHFRCO));

        calibrateRcOsc(cmuOsc_USHFRCO, USHFRCO_TUNE_FREQ, false, true);
        printf("Up count target: %lu\n\n", getUpcount(true));

        // Display result when finished
        while(!endOfTune());

        printf("Tuning count: %u\n", ++tuneCount);
        printf("Up count value: %lu\n", getUpcount(false));
        printf("New tuning value: %lu\n", USHFRCO_TUNE_GET);
        printf("New fine tuning\nvalue: %lu\n", CMU_OscillatorFineTuningGet(cmuOsc_USHFRCO));

        printf(AGAIN_BUTTON);
        printf(EXIT_BUTTON);

        progRun = false;
        break;
#endif

      default:
        break;
      }
    }
  }
}
