/*****************************************************************************
 * @file main_cmu_hfxo_autostart.c
 * @brief CMU automatic HFXO start example for  EFR32/EFM32 Series 1
 * @version  1.13

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "main_cmu_hfxo_autostart.h"

// Defines
#define USE_LFTIMEOUT           1
#define USE_AUTO_HFXO           1

#define WAKEUP_INTERVAL_MS      1000
#define COUNT_BETWEEN_WAKEUP    (((32768 * WAKEUP_INTERVAL_MS) / 1000) - 1)

// Global variables
volatile uint8_t wakeUpCnt;     // EM2 wake-up count

/**********************************************************
 * @brief Interrupt handler for odd GPIO. 
 **********************************************************/
void GPIO_ODD_IRQHandler(void)
{
  // Toggle pin to denote GPIO wake-up IRQ handler entry
  GPIO_PinOutToggle(GPIO_INT_TOGGLE_PORT, GPIO_INT_TOGGLE_PIN);
  GPIO_IntClear(_GPIO_IFC_MASK);
}

/**************************************************************************//**
 * @brief CMU IRQ handler.
 *****************************************************************************/
void CMU_IRQHandler(void)
{
  // Toggle pin to denote CMU IRQ handler entry
  GPIO_PinOutToggle(CMU_INT_TOGGLE_PORT, CMU_INT_TOGGLE_PIN);

  /*
   * If the CMU low-frequency time out error interrupt is enabled
   * and were actually to occur (e.g. the crystal has failed), code
   * in this IRQ handler should detect this and react accordingly
   * as opposed to simply clearing all CMU interrupts and exiting as
   * is down below.
   */
  CMU_IntClear(_CMU_IFC_MASK);
}

/**************************************************************************//**
 * @brief GPIO initialization
 *****************************************************************************/
void initGPIO(void)
{
  CMU_ClockEnable(cmuClock_GPIO, true);

  // LETIMER pulse output pin
  GPIO_PinModeSet(LETIMER_OUTPUT_PORT, LETIMER_OUTPUT_PIN, gpioModePushPull, 0);

  // GPIO IRQ wake from EM2 input pin
  GPIO_PinModeSet(GPIO_INPUT_PORT, GPIO_INPUT_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(GPIO_INPUT_PORT, GPIO_INPUT_PIN, GPIO_INPUT_PIN, false, true, true);

  // GPIO IRQ handler entry output pin
  GPIO_PinModeSet(GPIO_INT_TOGGLE_PORT, GPIO_INT_TOGGLE_PIN, gpioModePushPull, 0);

  // CMU IRQ handler entry output pin
  GPIO_PinModeSet(CMU_INT_TOGGLE_PORT, CMU_INT_TOGGLE_PIN, gpioModePushPull, 0);

  // Enable GPIO ODD IRQ NVIC source
  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
  
  // CMU clock output pin
  GPIO_PinModeSet(CMU_CLK_PORT, CMU_CLK_PIN, gpioModePushPull, 0);

  // Output HFSRCCLK on CMU_CLKOUT0
  CMU->CTRL |= CMU_CTRL_CLKOUTSEL0_HFSRCCLK;
  CMU->ROUTELOC0 &= ~_CMU_ROUTELOC0_CLKOUT0LOC_MASK;
  CMU->ROUTELOC0 |= CMU_ROUTE_LOCN;
  CMU->ROUTEPEN = CMU_ROUTEPEN_CLKOUT0PEN;
}

/**************************************************************************//**
 * @brief  LETIMER initialization
 *****************************************************************************/
void initLETIMER(void)
{
  // Enable clock LE interface
  CMU_ClockEnable(cmuClock_HFLE, true);

  // Select LFXO as LFACLK source
  CMU_ClockSelectSet(cmuClock_LFA, cmuSelect_LFXO);
  CMU_ClockEnable(cmuClock_LETIMER0, true);

  // Set location of output and enable
  LETIMER0->ROUTELOC0 &= ~LETIMER_ROUTE_MASK;
  LETIMER0->ROUTELOC0 |= LETIMER_ROUTE_LOCN;
  LETIMER0->ROUTEPEN |= LETIMER_ROUTEPEN_OUT0PEN;

  // Comparator 0 match at wake-up count
  LETIMER_CompareSet(LETIMER0, 0, COUNT_BETWEEN_WAKEUP);

  // Repeat must be non-zero value for comparator reload
  LETIMER_RepeatSet(LETIMER0, 0, 1);

  // LETIMER configuration
  LETIMER_Init_TypeDef letimerInit = LETIMER_INIT_DEFAULT;

  letimerInit.comp0Top = true;              // Load counter on underflow
  letimerInit.ufoa0    = letimerUFOAPulse;  // Pulse output on underflow
  letimerInit.debugRun = true;
  
  // Initialize and start LETIMER
  LETIMER_Init(LETIMER0, &letimerInit);
}

/**************************************************************************//**
 * @brief  Main function
 *****************************************************************************/
int main(void)
{
  // Initialize chip
  CHIP_Init();

#if defined( _EMU_DCDCCTRL_MASK )
  // Initialize DC-DC converter
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif

  // Initialize HFXO with default parameters
  CMU_HFXOInit_TypeDef hfxoInit = CMU_HFXOINIT_DEFAULT;
  CMU_HFXOInit(&hfxoInit);

#if USE_LFTIMEOUT == 1
  /*
   * The HFXO start-up delay is timed in LFECLK cycles and must only
   * be changed when both the HFXO and the LFECLK are disabled.
   */
  CMU->HFXOCTRL |= CMU_HFXOCTRL_LFTIMEOUT_16CYCLES;
#endif

  // Setup GPIO
  initGPIO();

  /* Setup RTCC, LFECLK is now enabled for LFTIMEOUT */
  initLETIMER();

  /*
   * Display initialization is performed here for two reasons:
   *
   * 1) As noted above, for proper functionality, the HFXOCTRL
   * register's LFTIMEOUT can only be written when both the LFECLK
   * and the HFXO are disabled.  This matters because...
   *
   * 2) The LFECLK must, obviously, be enabled for HFXO autostart
   * to function, something the DISPLAY driver happens to do.
   *
   * When running on the Starter Kits for all Series 1 devices or
   * on most EFR32 radio boards, the DISPLAY driver uses the RTCC
   * to generate a clock that is routed via the PRS to a pin used
   * to auto-toggle the display's EXTCOMIN signal.  The RTCC runs
   * off the LFECLK.
   *
   * If the DISPLAY driver is not initialized here, the subsequent
   * call to CMU_HFXOAutoStartEnable() will fail because the LFECLK
   * is not running.
   */
  DISPLAY_Init();

  // Retarget STDIO to a text display
  if (RETARGET_TextDisplayInit() != TEXTDISPLAY_EMSTATUS_OK)
  {
    // Text display initialization failed
    while (1);
  }

  #if USE_AUTO_HFXO == 1
  /*
   * Enable automatic start-up of the HFXO, its selection as the
   * HFSRCLK, and HFRCO shutdown upon exit from EM2/EM3 into EM0/EM1.
   *
   * Note that this has immediate effect in EM0 simply by nature of
   * running the code that sets the relevant CMU_HFXOCTRL bits.
   */
  CMU_HFXOAutostartEnable(0, false, true);

  // Wait for the HFRCO to be disabled before clearing interrupts
  while ((CMU_IntGet() & CMU_IF_HFRCODIS) == 0);

  // Clear CMU interrupts
  CMU_IntClear(_CMU_IFC_MASK);

#if USE_LFTIMEOUT == 0
  // Enable the CMU HFXO automatic switch interrupt
  CMU_IntEnable(CMU_IEN_HFXOAUTOSW);
#else

  /*
   * Enable the CMU HFXO automatic switch and low-frequency timeout
   * error interrupts.
   */
  CMU_IntEnable(CMU_IEN_HFXOAUTOSW | CMU_IEN_LFTIMEOUTERR);
#endif  
  NVIC_ClearPendingIRQ(CMU_IRQn);
  NVIC_EnableIRQ(CMU_IRQn);
#endif

  // Clear the screen
  printf("\f");

#if USE_AUTO_HFXO == 0
  printf("Normal HFRCO Start\n");
  printf("Example\n\n"); 
#else
#if USE_LFTIMEOUT == 0
  printf("Normal HFXO\n");
  printf("Auto Start Example\n\n");
#else
  printf("Deterministic HFXO\n");
  printf("Auto Start Example\n\n");
#endif
#endif

  // Print output message
  printf("EXP Outputs:\n");
  printf("LETIMER (P%c%d) on %d\n", (char)('A' + (char)(LETIMER_OUTPUT_PORT)), LETIMER_OUTPUT_PIN,  EXP_LETIMER_OUT);
  printf("GPIO IRQ (P%c%d) on %d\n", (char)('A' + (char)(GPIO_INT_TOGGLE_PORT)), GPIO_INT_TOGGLE_PIN,  EXP_GPIO_OUT);
  printf("CMU IRQ (P%c%d) on %d\n", (char)('A' + (char)(CMU_INT_TOGGLE_PORT)), CMU_INT_TOGGLE_PIN,  EXP_CMU_INT);
  printf("HFSRCLK (P%c%d) on %d\n", (char)('A' + (char)(CMU_CLK_PORT)), CMU_CLK_PIN,  EXP_CMU_CLK);
  printf("\n");

  // Print connection message
  printf("Connect expansion\n");
  printf("header pins %d and %d\n", EXP_LETIMER_OUT, EXP_GPIO_IN);
  printf("to wake the device\n");
  printf("from EM2 every second\n\n");
  
  while (1)
  {
    // Enter EM2 (remember; do this with the debugger disconnected)
    EMU_EnterEM2(false);

    // Update wake-up count
    printf("EM2 wake count: %3d\r", ++wakeUpCnt);
//    printf(TEXTDISPLAY_ESC_SEQ_CURSOR_UP_ONE_LINE);
  }
}
