/*****************************************************************************
 * @file main_cmu_lfxo_dpll.c
 * @brief CMU LFXO DPLL example for EFM32 Gecko Series 1 (EFM32xG12/13)
 * @version  1.13

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include <stdio.h>
#include "em_chip.h"
#include "em_cmu.h"
#include "em_device.h"
#include "em_emu.h"
#include "em_gpio.h"
#include "em_prs.h"
#include "em_usart.h"
#include "display.h"
#include "textdisplay.h"
#include "retargettextdisplay.h"
#include "bsp.h"

/* HFRCO TARGET_FREQ = FREF * (DPLL_FACTOR_N + 1)/(DPLL_FACTOR_M + 1) */
#define FREF                    32768           /* Use LFXO as reference */
#define TARGET_FREQ             24576000        /* Target is 24.576 MHz */
#define DPLL_FACTOR_N           1499            /* Recommend to be >300 */
#define DPLL_FACTOR_M           (FREF * (DPLL_FACTOR_N + 1))/TARGET_FREQ - 1

/* Defines for CMU clock output */
#define CMU_OUT_EN            0
#define CMU_OUT1_PORT           gpioPortD       /* Pin 12 of EXP header */
#define CMU_OUT1_PIN            10
#define CMU_OUT1_LOC            CMU_ROUTELOC0_CLKOUT1LOC_LOC4

/* Defines for I2S */
#define I2S_USART               USART3          /* I2S in USART1 or USART3 */
#define I2S_USART_CLK           cmuClock_USART3
#define I2S_CLKDIV_VALUE        0x0300          /* Divider for I2S clock */

#if defined(_EFM32_GIANT_FAMILY)
#define I2S_TX_PORT     gpioPortI
#define I2S_TX_PIN      12
#define I2S_CLK_PORT    gpioPortI
#define I2S_CLK_PIN     14
#define I2S_WS_PORT     gpioPortI
#define I2S_WS_PIN      15
#define I2S_TX_LOC      USART_ROUTELOC0_TXLOC_LOC5
#define I2S_WS_LOC      USART_ROUTELOC0_CSLOC_LOC5
#define I2S_CLK_LOC     USART_ROUTELOC0_CLKLOC_LOC5
#elif defined (_EFM32_PEARL_FAMILY)
#define I2S_TX_PORT     gpioPortB       /* Pin 7 of EXP header */
#define I2S_TX_PIN      6
#define I2S_CLK_PORT    gpioPortB       /* Pin 11 of EXP header */
#define I2S_CLK_PIN     8
#define I2S_WS_PORT     gpioPortD       /* Pin 13 of EXP header */
#define I2S_WS_PIN      8
#define I2S_TX_LOC      USART_ROUTELOC0_TXLOC_LOC10
#define I2S_WS_LOC      USART_ROUTELOC0_CSLOC_LOC29
#define I2S_CLK_LOC     USART_ROUTELOC0_CLKLOC_LOC10
#else
#error "Device not supported by this code example."
#endif

#define I2S_DUMMY_DATA          0x55AA

/* Global variable */
volatile uint32_t cmuIntFlag;           /* Buffer for CMU interrupt flag */

/**************************************************************************//**
 * @brief
 *   Setup USART in I2S master mode.
 *****************************************************************************/
static void setupI2s(void)
{
  USART_InitI2s_TypeDef init = USART_INITI2S_DEFAULT;

  /* Enable USART clock */
  CMU_ClockEnable(I2S_USART_CLK, true);

  /* Configure USART to transmit stereo 48 KHz 32 bit data */
  init.sync.baudrate = 2 * 48000 * 32;          /* Bit clock = 3.072 MHz */
  init.format = usartI2sFormatW32D32;
  USART_InitI2s(I2S_USART, &init);

  /* The CMU_ClockFreqGet() does not reflect the DPLL effect */
  /* Need manual setup on CLKDIV for 3.072 MHz bit clock (= 24.576 MHz/8) */
  I2S_USART->CLKDIV = I2S_CLKDIV_VALUE;
  
  /* Enable GPIO clock and I2S pins */
  CMU_ClockEnable(cmuClock_GPIO, true);
  GPIO_PinModeSet(I2S_TX_PORT, I2S_TX_PIN, gpioModePushPull, 0);
  GPIO_PinModeSet(I2S_CLK_PORT, I2S_CLK_PIN, gpioModePushPull, 0);
  GPIO_PinModeSet(I2S_WS_PORT, I2S_WS_PIN, gpioModePushPull, 0);
  
  /* Enable pins at location */
  I2S_USART->ROUTEPEN = USART_ROUTEPEN_TXPEN |
                        USART_ROUTEPEN_CSPEN |
                        USART_ROUTEPEN_CLKPEN;
  I2S_USART->ROUTELOC0 = I2S_TX_LOC | I2S_WS_LOC | I2S_CLK_LOC; 
}

/**************************************************************************//**
 * @brief CMU IRQ handler.
 *****************************************************************************/
void CMU_IRQHandler(void)
{
  /* Get and clear interrupt flag */
  cmuIntFlag = CMU->IF;
  CMU->IFC = cmuIntFlag;
}

/***************************************************************************//**
 * @brief
 *   Setup DPLL for target HFRCO frequency
 *
 * @details
 *   Use frequency-lock mode and automatic recovery is enabled.
 *
 * @note
 *   The reference clock (LFXO or HFXO) must be enabled and stable before
 *   calling this function.
 *
 * @param[in] xtalOsc
 *   The crystal oscillator for reference clock, one of:
 *   @li #cmuOsc_LFXO
 *   @li #cmuOsc_HFXO
 *
 * @param[in] factorN
 *   It is required to be larger than 32.
 *
 * @param[in] factorM
 *   It can be any 12-bit value. 
 *
 * @return
 *   True if DPLL is locked.
 ******************************************************************************/
bool setupDpll(CMU_Osc_TypeDef xtalOsc, uint16_t factorN, uint16_t factorM)
{
  /* Make sure factor N must > 32 */
  if(factorN < 32)
  {
	  printf("factor N must be > 32");
	  while(1) {;}
  }
  
  /* Set HFCLK to HFRCODIV2 to avoid overclocking */
  CMU->HFCLKSEL = CMU_HFCLKSEL_HF_HFRCODIV2;

  /* FINETUNINGEN bit must be set to make DPLL work */
  CMU->HFRCOCTRL |= CMU_HFRCOCTRL_FINETUNINGEN;

  /* Enable DPLL related interrupt */
  CMU_IntClear(_CMU_IFC_MASK);
  CMU_IntEnable(CMU_IEN_DPLLRDY + CMU_IEN_DPLLLOCKFAILLOW + CMU_IEN_DPLLLOCKFAILHIGH);        
  NVIC_ClearPendingIRQ(CMU_IRQn);
  NVIC_EnableIRQ(CMU_IRQn);

  /* Use FREQLL, set FREF and enable AUTORECOVER */
  CMU->DPLLCTRL = _CMU_DPLLCTRL_RESETVALUE;
  if (xtalOsc == cmuOsc_LFXO)
  {
    CMU->DPLLCTRL = (CMU_DPLLCTRL_REFSEL_LFXO + CMU_DPLLCTRL_AUTORECOVER); 
  }
  else
  {
    CMU->DPLLCTRL = (CMU_DPLLCTRL_REFSEL_HFXO + CMU_DPLLCTRL_AUTORECOVER); 
  }
  
  /* Set output target frequency and enable DPLL*/
  CMU->DPLLCTRL1 = (factorN << _CMU_DPLLCTRL1_N_SHIFT) + (factorM << _CMU_DPLLCTRL1_M_SHIFT);
  CMU->OSCENCMD = CMU_OSCENCMD_DPLLEN;
  
  /* Wait DPLL interrupt on EM1, check any errors occur */
  EMU_EnterEM1();
  if (cmuIntFlag & (CMU_IF_DPLLLOCKFAILLOW + CMU_IF_DPLLLOCKFAILHIGH))
  {
    return false;
  }
  else
  {
    /* Set HFCLK to HFRCO after DPLL is locked */
    CMU->HFCLKSEL = CMU_HFCLKSEL_HF_HFRCO;
    return true;
  }
}

/**************************************************************************//**
 * @brief  Main function
 *****************************************************************************/
int main(void)
{
  /* Initialize chip */
  CHIP_Init();

#if defined( _EMU_DCDCCTRL_MASK )
  /* Initialize DCDC regulator */
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif

  /* Set HFRCO band for DCO, close to target frequency */
  CMU_HFRCOBandSet(cmuHFRCOFreq_26M0Hz);

#if defined (MSC_CTRL_IFCREADCLEAR)  
  /* Enable atomic read-clear operation on reading IFC register */
  MSC->CTRL |= MSC_CTRL_IFCREADCLEAR;
#endif

  /* Enable LFXO and wait it ready, use as DPLL reference clock */
  CMU_OscillatorEnable(cmuOsc_LFXO, true, true);

  /* Initialize the display module. */
  DISPLAY_Init();

  /* Retarget stdio to a text display. */
  if (RETARGET_TextDisplayInit() != TEXTDISPLAY_EMSTATUS_OK)
  {
    /* Text display initialization failed. */
    while (1) ;
  }

  /* Print messages */
  printf("\f");
  printf("DPLL Example\n");
  printf("HFRCO is tuned to\n");
  printf("24.576 MHz with LFXO\n\n");
  
#if CMU_OUT_EN == 1
  /* Configure HFSRCCLK to CLKOUT1  */
  CMU_ClockEnable(cmuClock_GPIO, true);
  GPIO_PinModeSet(CMU_OUT1_PORT, CMU_OUT1_PIN, gpioModePushPull, 0);

  CMU->ROUTEPEN = CMU_ROUTEPEN_CLKOUT1PEN;
  CMU->ROUTELOC0 = CMU_OUT1_LOC;
  CMU->CTRL |= CMU_CTRL_CLKOUTSEL1_HFSRCCLK;
#endif
  
  /* Setup and start DPLL */
  if (setupDpll(cmuOsc_LFXO, DPLL_FACTOR_N, DPLL_FACTOR_M))
  {
    printf("DPLL is locked\n\n");
    printf("Transmit 48kHz 32-bit\n");
    printf("I2S data on USART3\n\n");

#if defined(BSP_STK_BRD2204A)
    printf("I2S_WS on PI15\n");
    printf("I2S_CLK on PI14\n");
    printf("I2S_TX on PI12\n");
#else
    printf("I2S_WS on PD8\n");
    printf("I2S_CLK on PB8\n");
    printf("I2S_TX on PB6\n");
#endif

  }
  else
  {
    printf("DPLL fails to lock!\n");
    printf("Select another\n");
    printf("HFRCO frequency band\n");
    while (1)
      ;
  }
  
  /* Use DPLL HFRCO as I2S clock */
  setupI2s();

  /* Infinite loop */
  while (1)
  {
    /* Transmit I2S data continuously */
    USART_TxDouble(I2S_USART, I2S_DUMMY_DATA);
  }
}
