/**************************************************************************//**
 * @file cmu_calibrate.c
 * @brief CMU RC calibration functions for EFR32 Gecko Series 2
 * @version  1.11

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "main_cmu_example_s2.h"

/* Global variables */
CMU_Osc_TypeDef oscCal;                 /* RC oscillator under calibration */
volatile bool downRef;                  /* True if down clock is reference */
volatile bool tuneEnd;                  /* True if tuning is finished */
volatile bool fineTune;                 /* True if fine tuning is enabled */
volatile bool upCountLess;              /* upCounter < upCounterTuned before */
volatile bool upCountGreat;             /* upCounter > upCounterTuned before */
uint16_t tuningVal;                     /* Runtime tunning value */
uint8_t fineTuningVal;                  /* Runtime fine tunning value */
uint32_t upCounter;                     /* Runtime up counter value */
uint32_t upCounterTuned;                /* Target up counter value */
volatile uint32_t upCounterPrevious;    /* History up counter value */

/***************************************************************************//**
 * @brief
 *   Get RC oscillator frequency fine tuning value.
 * @param[in] osc
 *   RC oscillator to get fine tuning value for, one of:
 *   @li #cmuOsc_HFRCO
 *   @li #cmuOsc_AUXHFRCO
 *   @li #cmuOsc_USHFRCO
 * @return
 *   The RC oscillator frequency fine tuning value in use.
 ******************************************************************************/
uint32_t CMU_OscillatorFineTuningGet(CMU_Osc_TypeDef osc)
{
  uint32_t ret;
  
  switch (osc)
  {
#if defined( _HFRCO_CAL_FINETUNING_MASK )      
    /* how do we differentiate between the two HFCROs?*/
  case cmuOsc_HFRCODPLL:
    ret = (HFRCO0->CAL & _HFRCO_CAL_FINETUNING_MASK)
      >> _HFRCO_CAL_FINETUNING_SHIFT;
    break;
#endif
    
#if defined( _HFRCO_CAL_FINETUNING_MASK ) && defined(_SILICON_LABS_32B_SERIES_2_CONFIG_1)
  case cmuOsc_HFRCOEM23:
    ret = (HFRCOEM23->CAL & _HFRCO_CAL_FINETUNING_MASK)
      >> _HFRCO_CAL_FINETUNING_SHIFT;
    break;
#endif
  default:
    EFM_ASSERT(0);
    ret = 0;
    break;
  }
  
  return ret;
}

/***************************************************************************//**
 * @brief
 *   Set RC oscillator frequency fine tuning value.
 * @param[in] osc
 *   RC oscillator to set fine tuning value for, one of:
 *   @li #cmuOsc_HFRCODPLL
 *   @li #cmuOsc_HFRCOEM23
 *   @param[in] val
 *   The RC oscillator frequency fine tuning value to use.
 ******************************************************************************/
void CMU_OscillatorFineTuningSet(CMU_Osc_TypeDef osc, uint32_t val)
{
  switch (osc)
  {
#if defined( _HFRCO_CAL_FINETUNING_MASK )      
  case cmuOsc_HFRCODPLL:
    EFM_ASSERT(val <= (_HFRCO_CAL_FINETUNING_MASK
                       >> _HFRCO_CAL_FINETUNING_SHIFT));
    while(BUS_RegBitRead(&HFRCO0->STATUS, _HFRCO_STATUS_SYNCBUSY_SHIFT))
    {
    }
    HFRCO0->CAL = (HFRCO0->CAL & ~(_HFRCO_CAL_FINETUNING_MASK))
      | (val << _HFRCO_CAL_FINETUNING_SHIFT);
    break;
#endif
    
#if defined( _HFRCO_CAL_FINETUNING_MASK ) && defined(_SILICON_LABS_32B_SERIES_2_CONFIG_1)
  case cmuOsc_HFRCOEM23:
    EFM_ASSERT(val <= (_HFRCO_CAL_FINETUNING_MASK
                       >> _HFRCO_CAL_FINETUNING_SHIFT));
    while(BUS_RegBitRead(&HFRCOEM23->STATUS, _HFRCO_STATUS_SYNCBUSY_SHIFT))
    {
    }
    HFRCOEM23->CAL = (HFRCOEM23->CAL & ~(_HFRCO_CAL_FINETUNING_MASK))
      | (val << _HFRCO_CAL_FINETUNING_SHIFT);
    break;
#endif
    
  default:
    EFM_ASSERT(0);
    break;
  }
}

/**************************************************************************//**
 * @brief End of tuning check.
 * @return
 *   True if tuning is finished.
 *****************************************************************************/
bool endOfTune(void)
{
  return(tuneEnd);
}

/**************************************************************************//**
 * @brief 
 *   Get target and actual up counter value.
 * @param[in] targetCnt
 *   Select up counter value to return
 * @return
 *   Target up counter value if targetCnt is true.
 *   Actual up counter value if targetCnt is false.
 *****************************************************************************/
uint32_t getUpcount(bool targetCnt)
{
  if (targetCnt)
  {
    return upCounterTuned;
  }
  else
  {
    return(upCounter);
  }
}

/**************************************************************************//**
 * @brief CMU IRQ handler.
 *****************************************************************************/
void CMU_IRQHandler(void)
{
  /* Clear interrupt flag, get the up counter value */
  CMU_IntClear(CMU_IF_CALRDY_DEF);
  upCounter = CMU_CalibrateCountGet();

  /* Up counter result is smaller than the desired value */
  if (upCounter < upCounterTuned)
  {
    if (upCountGreat)                   
    {
      if (!fineTune)                    /* End of calibration if no fine tuning */
      {                                 /* and upCounter > upCounterTuned before */
        CMU_CalibrateStop();
        if ((upCounterTuned - upCounter) > (upCounterPrevious - upCounterTuned))
        {
          tuningVal++;
        }
        tuneEnd = true;
        return;
      }
      else
      {                                  
        if ((upCounterTuned - upCounter) == 1)
        {
          CMU_CalibrateStop();
          tuneEnd = true;
          return;
        }
        /* Next fine tuning if count difference not <=1 */ 
        /* Increase tuning value if down counter is reference; othewise decrease */
        if (downRef)
        {
          fineTuningVal++;
        }
        else
        {
          fineTuningVal--;
        }
        CMU_OscillatorFineTuningSet(oscCal,fineTuningVal);
        upCountLess = true;             /* Set upCounter < upCounterTuned before */
#if defined( CMU_CALCTRL_CONT )         
        if ((CMU->CALCTRL & CMU_CALCTRL_CONT) == 0)
        {
          /* CALCNT was RW access, but in Panther it is R-only*/
         //CMU->CALCNT = DOWN_TOP_VALUE; /* Reload TOP value if not cont mode */
          CMU->CALCTRL = (CMU->CALCTRL & ~(_CMU_CALCTRL_CALTOP_MASK)) | DOWN_TOP_VALUE;
          
          CMU->CALCMD = CMU_CALCMD_CALSTART;
        }
#else
        CMU->CALCNT = DOWN_TOP_VALUE;
        CMU->CMD = CMU_CMD_CALSTART;
#endif
          return;
      }
    }
    else
    {
      /* Store the previous value for comparison */
      upCountLess = true;
      upCounterPrevious = upCounter;
      
      /* Increase tuning value if down counter is reference; othewise decrease */
      if (downRef)
      {
        tuningVal++;
      }
      else
      {
        tuningVal--;
      }
      /* Write the new tuning value to RC oscillator */

      CMU_OscillatorTuningSet(oscCal, tuningVal);
#if defined( CMU_CALCTRL_CONT )
      if ((CMU->CALCTRL & CMU_CALCTRL_CONT) == 0)
      {
        //CMU->CALCNT = DOWN_TOP_VALUE;   /* Reload TOP value if not cont mode */
        CMU->CALCTRL = (CMU->CALCTRL & ~(_CMU_CALCTRL_CALTOP_MASK)) | DOWN_TOP_VALUE;
        CMU->CALCMD = CMU_CALCMD_CALSTART;
      }
#else
      CMU->CALCNT = DOWN_TOP_VALUE;
      CMU->CMD = CMU_CMD_CALSTART;
#endif
      return;
    }
  }

  /* If the up counter result is higher than the desired value */
  if (upCounter > upCounterTuned)
  {
    if (upCountLess)
    {
      if (!fineTune)                    /* End of calibration if no fine tuning */
      {                                 /* and upCounter < upCounterTuned before */
        CMU_CalibrateStop();
        if ((upCounter - upCounterTuned) > (upCounterTuned - upCounterPrevious))
        {
          tuningVal--;
        }
        tuneEnd = true;
        return;
      }
      else
      {
        if ((upCounter - upCounterTuned) == 1)
        {
          CMU_CalibrateStop();
          tuneEnd = true;
          return;
        }          
        /* Next fine tuning if count difference not <=1 */ 
        /* Decrease tuning value if down counter is reference; othewise increase */
        if (downRef)
        {
          fineTuningVal--;
        }
        else
        {
          fineTuningVal++;
        }
        CMU_OscillatorFineTuningSet(oscCal,fineTuningVal);
        upCountGreat = true;            /* Set upCounter > upCounterTuned before */
#if defined( CMU_CALCTRL_CONT )
        if ((CMU->CALCTRL & CMU_CALCTRL_CONT) == 0)
        {
          //CMU->CALCNT = DOWN_TOP_VALUE; /* Reload TOP value if not cont mode */
          CMU->CALCTRL = (CMU->CALCTRL & ~(_CMU_CALCTRL_CALTOP_MASK)) | DOWN_TOP_VALUE;
          CMU->CALCMD = CMU_CALCMD_CALSTART;
        }
#else
        CMU->CALCNT = DOWN_TOP_VALUE;
        CMU->CMD = CMU_CMD_CALSTART;
#endif
        return;
      }
    }      
    else
    {
      /* Store the previous value for comparison */
      upCountGreat = true;  
      upCounterPrevious = upCounter;

      /* Decrease tuning value if down counter is reference; othewise increase */
      if (downRef)
      {
        tuningVal--;
      }
      else
      {
        tuningVal++;
      }
      /* Write the new tuning value to RC oscillator */

      CMU_OscillatorTuningSet(oscCal, tuningVal);
#if defined( CMU_CALCTRL_CONT )
      if ((CMU->CALCTRL & CMU_CALCTRL_CONT) == 0)
      {
        //CMU->CALCNT = DOWN_TOP_VALUE;   /* Reload TOP value if not cont mode */
        CMU->CALCTRL = (CMU->CALCTRL & ~(_CMU_CALCTRL_CALTOP_MASK)) | DOWN_TOP_VALUE;
        CMU->CALCMD = CMU_CALCMD_CALSTART;
      }
#else
      CMU->CALCNT = DOWN_TOP_VALUE;
      CMU->CMD = CMU_CMD_CALSTART;
#endif
      return;
    }
  }

  /* Up counter result is equal to the desired value, end of calibration */
  if (upCounter == upCounterTuned)
  {
    CMU_CalibrateStop();
    tuneEnd = true;
  }
}

/***************************************************************************//**
 * @brief
 *   Configure RC oscillator calibration
 *
 * @details
 *   Configure a calibration for a selectable clock source against another
 *   selectable reference clock.
 *   Refer to the reference manual, CMU chapter, for further details.
 *
 * @note
 *   After configuration, a call to CMU_CalibrateStart() is required, and
 *   the resulting calibration value can be read out with the
 *   CMU_CalibrateCountGet() function call in CMU ISR.
 *
 * @param[in] rcOsc
 *   The RC Oscillator for calibration, one of:
 *   @li #cmuOsc_LFRCO
 *   @li #cmuOsc_HFRCO
 *   @li #cmuOsc_AUXHFRCO
 *   @li #cmuOsc_USHFRCO
 *
 * @param[in] rcOscFreq
 *   The RC oscillator frequency in Hz to be tuned for. 
 *
 * @param[in] calCont
 *   True to enable continuous mode if this feature is available. 
 *
 * @param[in] fineEnable
 *   True to enable fine tuning if this feature is available. 
******************************************************************************/

/*TODO probably need to change 4th arg to uint8_t since finetuning is no longer enableable*/
void calibrateRcOsc(CMU_Osc_TypeDef rcOsc, uint32_t rcOscFreq, bool calCont, bool fineEnable)
{
  /* Setup related variables */
  oscCal = rcOsc;
  tuneEnd = false;
  fineTune = false;
  upCountLess = false;
  upCountGreat = false;

  CMU_OUT_CLR;                                  /* Reset CMU output */
  tuningVal = CMU_OscillatorTuningGet(oscCal);  
  
  switch (oscCal)
  {
  case cmuOsc_LFRCO:
    /* series 2 */
   // CMU_ClockSelectSet(cmuClock_HCLK, cmuSelect_HFXO);
    upCounterTuned = (uint32_t)(((float)rcOscFreq/(float)CMU_ClockFreqGet(cmuClock_SYSCLK))*(float)(DOWN_TOP_VALUE+1));
    /* USE API*/
     CMU_CalibrateConfig(DOWN_TOP_VALUE,cmuSelect_HFXO,cmuSelect_LFRCO);
    /* end series 2 */
    CMU_OUT_LFRCO;                      /* Enable CMU output for LFRCO */
    downRef = true;                     /* Higher tune value means higher frequency */
    break;
    
  case cmuOsc_HFRCODPLL:
    /* Enable LFXO to tune HFRCO */  
    /* Force enable this oscillator*/
    HFRCO0->CTRL |= HFRCO_CTRL_FORCEEN;
    //CMU_ClockSelectSet(cmuClock_SYSCLK, cmuSelect_HFRCODPLL);
    upCounterTuned = (uint32_t)(((float)SystemLFXOClockGet()/(float)rcOscFreq)*(float)(DOWN_TOP_VALUE+1));
    CMU_CalibrateConfig(DOWN_TOP_VALUE, cmuSelect_HFRCODPLL,cmuSelect_LFXO);
    GPIO_SlewrateSet(CMU_OUT1_PORT, 4,7);
    CMU_OUT_HFRCODPLL;                      /* Enable CMU output for LFRCO */
    if (fineEnable)                     /* Check use finetune or not? */
    {
      fineTuningVal = CMU_OscillatorFineTuningGet(oscCal);
      fineTune = true;
    }
    downRef = true;                     /* Higher tune value means lower frequency */
    break;
    
#if defined(_SILICON_LABS_32B_SERIES_2_CONFIG_1)  /* HFRCOEM23 only on EFR32xG21*/
  case cmuOsc_HFRCOEM23:
    CMU_OUT_HFRCOEM23;                   /* Enable CMU output for AUXHFRCO */
    //force enable this oscillator
    HFRCOEM23->CTRL |= HFRCO_CTRL_FORCEEN;
    upCounterTuned = (uint32_t)(((float)SystemLFXOClockGet()/(float)rcOscFreq)*(float)(DOWN_TOP_VALUE+1));
    CMU_CalibrateConfig(DOWN_TOP_VALUE, cmuSelect_HFRCOEM23, cmuSelect_LFXO);
    if (fineEnable)                     /* Check use finetune or not? */
    {
      fineTuningVal = CMU_OscillatorFineTuningGet(oscCal);
      fineTune = true;
    }
    downRef = true;                     /* Higher tune value means lower frequency */
    break;
#endif
    
  default:
    EFM_ASSERT(0);
    break;
  }

  CMU_IntEnable(CMU_IF_CALRDY_DEF);        /* Enable calibration ready interrupt */
  NVIC_ClearPendingIRQ(CMU_IRQn);
  NVIC_EnableIRQ(CMU_IRQn);
  
#if defined( CMU_CALCTRL_CONT )
  if (calCont)
  {
    CMU_CalibrateCont(true);            /* Enable continuous calibration */
  }
#endif
  CMU_CalibrateStart();                 /* Start calibration */
}
