/*****************************************************************************
 * @file cmu_common.c
 * @brief Common functions for CMU application note
 * @version  1.11

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "main_cmu_example_s2.h"

/* External variables */
extern volatile bool runKey;
extern volatile bool menuKey;
extern volatile bool progRun;
extern volatile uint8_t menuLevel;

/**********************************************************
 * @brief Interrupt handler for right side push button. 
 **********************************************************/

void GPIO_ODD_IRQHandler(void)
{
  /* Get and clear all pending GPIO interrupts */
  uint32_t interruptMask = GPIO_IntGet();
  GPIO_IntClear(interruptMask);

  /* Next menu item if no example is running */
  if (!progRun && !menuKey)
  {
    menuKey = true;
    if (++menuLevel == MENU_MAX)
    {
      menuLevel = MENU_START;
    }
  }
  else
  {
    progRun = false;
  }    
}

/**********************************************************
 * @brief Interrupt handler for left side push button. 
 **********************************************************/
void GPIO_EVEN_IRQHandler(void)
{
  /* Get and clear all pending GPIO interrupts */
  uint32_t interruptMask = GPIO_IntGet();
  GPIO_IntClear(interruptMask);

  /* No action if example is running */
  if (!progRun)
  {
    runKey = true;
  }
}

/**************************************************************************//**
 * @brief Setup GPIO interrupt for push buttons.
 *****************************************************************************/
void gpioSetup(void)
{
  /* Enable GPIO clock */
  CMU_ClockEnable(cmuClock_GPIO, true);

#if CMU_OUT_EN == 1 || defined(_SILICON_LABS_32B_SERIES_2_CONFIG_2)
  /* Configure push button 0 as input and enable interrupt  */
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, BSP_GPIO_PB0_PIN, false, true, true);

  /* Configure push button 1 as input and enable interrupt */
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, BSP_GPIO_PB1_PIN, false, true, true);
#else
  /* To wake from EM2, push buttons must be mapped to ports A or B
   * User must jumper P4 and P6 to P14 and P20, respectively, on WSTK Breakout Connector J101 */
  /* Configure push button 0 as input and enable interrupt  */
  GPIO_PinModeSet(PB0_REMAP_PORT, PB0_REMAP_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(PB0_REMAP_PORT, PB0_REMAP_PIN, PB0_REMAP_PIN, false, true, true);

  /* Configure push button 1 as input and enable interrupt */
  GPIO_PinModeSet(PB1_REMAP_PORT, PB1_REMAP_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(PB1_REMAP_PORT, PB1_REMAP_PIN, PB1_REMAP_PIN, false, true, true);
#endif

  /* Enable interrupt in core for even and odd GPIO interrupts */
  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);

  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
  
  /* Configure CMU output if enable */
#if CMU_OUT_EN == 1
  GPIO_PinModeSet(CMU_OUT1_PORT, CMU_OUT1_PIN, gpioModePushPull, 0);
  GPIO_SlewrateSet(CMU_OUT1_PORT, 4,7);
#endif

#if defined(_GPIO_CMU_ROUTEEN_MASK)
  GPIO->CMUROUTE.ROUTEEN = CMU_OUT_PEN;
  GPIO->CMUROUTE.CLKOUT1ROUTE = CMU_OUT1_PORT + CMU_OUT1_PIN;
#endif  
}

/***************************************************************************//**
* @brief Print the menu.
*******************************************************************************/
void printMenu(void)
{
  printf("\fExample %d\r\n", menuLevel+1);
  
  switch (menuLevel)
  {
#if defined(_SILICON_LABS_32B_SERIES_2_CONFIG_1)  /* LFRCO0 cal. available only on EFR32xG21*/
  case LFRCO_CAL:
    printf("HFXO to Calibrate LFRCO\r\n\r\n");
    break;
#endif
    
  case HFRCODPLL_CAL:
    printf("LFXO to Calibrate HFRCODPLL\r\n\r\n");
    break;
    
#if defined(_SILICON_LABS_32B_SERIES_2_CONFIG_1)  /* HFRCOEM23 only on EFR32xG21*/
  case HFRCOEM23_CAL:
    printf("LFXO to Calibrate HFRCOEM23\r\n\r\n");
    break;
#endif
    
  default:
    break;
  }
  printf(RUN_BUTTON);
  printf(NEXT_BUTTON); 
  printf("menu\r\n\r\n");
  menuKey = false;
  RETARGET_SerialFlush();
}
