/*****************************************************************************
 * @file main_cmu_lfxo_dpll.c
 * @brief CMU LFXO DPLL example for EFR32 Gecko Series 2 (EFR32xG21)
 * @version  1.11

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

 /*
This project demonstrates using the LFXO as the DPLL reference clock to generate
a locked DCO frequency of 24.576 MHz (32768 Hz * 750). The SYSCLK is then switched 
to the HFRCODLL output, and the UARTn peripheral is configured to generate a 3.072 MHz
bit clock (24.576 MHz / 8) for a stereo 48kHz 32-bit I2S data stream. The UARTn then
transmits dummy data (0x55AA55AA) continuously.

How To Test:
1.  Update the kit's firmware from the Simplicity Launcher (if necessary)
2.  Build the project and download to the Starter Kit
3.  Run the example
4.  Observe I2S word clock, I2S bit clock, and I2S TX with an oscilloscope on EXP10, EXP9 and 
    EXP8 of the WSTK, respectively. Observe 3.072 bit clock frequency.
	
Peripherals Used:
DPLL         - Configured to multiple LFXO by 750 to generate 24.576 MHz clock
LFXO		 - Source oscillator for DPLL
USARTn       - 3.072 bit clock (3072000 baud)
             - 32-bit word, 32-bit data format
USARTm		 - VCOM

***
               
Board:  Silicon Labs EFR32xG21 Radio Board (BRD4181A) + 
        Wireless Starter Kit Mainboard
Device: EFR32MG21A010F1024IM32
USART2 configured for I2S
USART0 configured for VCOM
PC02 (EXP 8)- 	GPIO Push/Pull output, USART2 I2S TX, Expansion Header Pin 8
PD03 (EXP 9)- 	GPIO Push/Pull output, USART2 I2S CLK, Expansion Header Pin 9
PC03 (EXP 10)- 	GPIO Push/Pull output, USART2 I2S WS, Expansion Header Pin 10

***

Board:  Silicon Labs EFR32xG22 Radio Board (BRD4182A) +
        Wireless Starter Kit Mainboard
Device: EFR32MG22C224F512IM40
USART0 configured for I2S
USART1 configured for VCOM
PC02 (EXP 8)- 	GPIO Push/Pull output, USART0 I2S TX, Expansion Header Pin 8
PB01 (EXP 9)- 	GPIO Push/Pull output, USART0 I2S CLK, Expansion Header Pin 9
PC03 (EXP 10)- 	GPIO Push/Pull output, USART0 I2S WS, Expansion Header Pin 10

*/

#include <stdio.h>
#include "em_chip.h"
#include "em_cmu.h"
#include "em_device.h"
#include "em_emu.h"
#include "em_gpio.h"
#include "em_prs.h"
#include "em_usart.h"
#include "bsp.h"
#include "expconfig.h"

#include "retargetserial.h"

/* Defines for DPLL */

#define FREF                    32768           /* Use LFXO as reference */
#define TARGET_FREQ             24576000        /* Target is 24.576 MHz */
#define DPLL_FACTOR_N           1499            /* Recommend to be >300 */
#define DPLL_FACTOR_M           (FREF * (DPLL_FACTOR_N+1) )/TARGET_FREQ - 1

/* Defines for CMU clock output */

#define CMU_OUT_EN              0
#define CMU_OUT1_PORT           BSP_EXP_HEADER4_PORT       /* Pin 4 of EXP header */
#define CMU_OUT1_PIN            BSP_EXP_HEADER4_PIN

/* Defines for I2S */
#if defined(_SILICON_LABS_32B_SERIES_2_CONFIG_2)  /* Defines for EFR32xG22/BRD4182A */
#define I2S_USART               USART0
#define I2S_USART_CLOCK			cmuClock_USART0
#define USART_REG_INDEX				0
#else  /* Defines for EFR32xG21/BRD4181A */
#define I2S_USART               USART2
#define I2S_USART_CLOCK			cmuClock_USART2
#define USART_REG_INDEX				2
#endif

#define I2S_CLKDIV_VALUE        0x0300          /* Divider for I2S clock */
#define I2S_TX_PORT             BSP_EXP_HEADER8_PORT       /* Pin 8 of EXP header */
#define I2S_TX_PIN              BSP_EXP_HEADER8_PIN
#define I2S_CLK_PORT            BSP_EXP_HEADER9_PORT       /* Pin 9 of EXP header */
#define I2S_CLK_PIN             BSP_EXP_HEADER9_PIN
#define I2S_WS_PORT             BSP_EXP_HEADER10_PORT       /* Pin 10 of EXP header */
#define I2S_WS_PIN              BSP_EXP_HEADER10_PIN
#define I2S_DUMMY_DATA          0x55AA

/* Global variables */

volatile uint32_t dpllIntFlag;           /* Buffer for DPLL interrupt flag */

/**************************************************************************//**
 * @brief
 *   Setup USART in I2S master mode.
 *****************************************************************************/
static void initI2S(void)
{
  USART_InitI2s_TypeDef init = USART_INITI2S_DEFAULT;

#if defined(_SILICON_LABS_32B_SERIES_2_CONFIG_2)
  /* Enable USART and GPIO peripheral clock branches (necessary for EFR32xG22) */
  CMU_ClockEnable(I2S_USART_CLOCK, true);
  CMU_ClockEnable(cmuClock_GPIO, true);
#endif

  /* Configure USART to transmit stereo 48 KHz 32 bit data */
  init.sync.baudrate = 2 * 48000 * 32;          /* Bit clock = 3.072 MHz */
  init.format = usartI2sFormatW32D32;
  USART_InitI2s(I2S_USART, &init);

  /* The CMU_ClockFreqGet() does not reflect the DPLL effect */
  /* Need manual setup on CLKDIV for 3.072 MHz bit clock (= 24.576 MHz/8) */
  I2S_USART->CLKDIV = I2S_CLKDIV_VALUE;
  
  /* Enable GPIO I2S pins */
  GPIO_PinModeSet(I2S_TX_PORT, I2S_TX_PIN, gpioModePushPull, 0);
  GPIO_PinModeSet(I2S_CLK_PORT, I2S_CLK_PIN, gpioModePushPull, 0);
  GPIO_PinModeSet(I2S_WS_PORT, I2S_WS_PIN, gpioModePushPull, 0);
  
  /* Enable pins at location */
  GPIO->USARTROUTE[USART_REG_INDEX].ROUTEEN = GPIO_USART_ROUTEEN_TXPEN |
                                GPIO_USART_ROUTEEN_CLKPEN |
                                GPIO_USART_ROUTEEN_CSPEN;
  GPIO->USARTROUTE[USART_REG_INDEX].TXROUTE = I2S_TX_PORT|I2S_TX_PIN <<_GPIO_USART_TXROUTE_PIN_SHIFT;
  GPIO->USARTROUTE[USART_REG_INDEX].CLKROUTE = I2S_CLK_PORT|I2S_CLK_PIN <<_GPIO_USART_CLKROUTE_PIN_SHIFT;
  GPIO->USARTROUTE[USART_REG_INDEX].CSROUTE = I2S_WS_PORT|I2S_WS_PIN <<_GPIO_USART_CSROUTE_PIN_SHIFT;
}

/***************************************************************************//**
 * @brief
 *   Setup DPLL for target HFRCO frequency
 *
 * @details
 *   Use frequency-lock mode and automatic recovery is enabled.
 *
 * @note
 *   The reference clock (LFXO or HFXO) must be enabled and stable before
 *   calling this function.
 *
 * @param[in] xtalOsc
 *   The crystal oscillator for reference clock, one of:
 *   @li #cmuOsc_LFXO
 *   @li #cmuOsc_HFXO
 *
 * @param[in] factorN
 *   It is required to be larger than 300.
 *
 * @param[in] factorM
 *   It can be any 12-bit value. 
 *
 * @return
 *   True if DPLL is locked.
 ******************************************************************************/
bool initDPLL(CMU_Osc_TypeDef xtalOsc, uint16_t factorN, uint16_t factorM)
{
  bool dpllInitStatus;
  CMU_DPLLInit_TypeDef dpllInit = {  .autoRecover = true,
                                     .m = factorM,
                                     .n = factorN,
                                     .frequency = TARGET_FREQ};
  
  if(xtalOsc == cmuOsc_LFXO){
     dpllInit.refClk = cmuSelect_LFXO;

#if defined(_SILICON_LABS_32B_SERIES_2_CONFIG_2)
     /* Enable LFXO and DPLL0 APB clock branches (necessary for EFR32xG22) */
     CMU_ClockEnable(cmuClock_LFXO, true);
     CMU_ClockEnable(cmuClock_DPLL0, true);
#endif

     /* Enable LFXO and wait until it's ready; use as DPLL reference clock */
     CMU_LFXOInit_TypeDef lfxoInit = CMU_LFXOINIT_DEFAULT;
     CMU_LFXOInit(&lfxoInit);
  } 
  else {
    dpllInit.refClk = cmuSelect_HFXO;
  }

  /* Set output target frequency and enable DPLL*/
  dpllInitStatus = CMU_DPLLLock(&dpllInit);
  
  if(dpllInitStatus == true)
  {
    /* Set SYSCLK to HFRCODPLL after DPLL is locked */
    CMU_ClockSelectSet(cmuClock_SYSCLK, cmuSelect_HFRCODPLL);

    /*re-init USART to make sure baud rate stays the same*/
    RETARGET_SerialInit();
  }
  return dpllInitStatus;
}

/**************************************************************************//**
 * @brief  Main function
 *****************************************************************************/
int main(void)
{
  /* Initialize chip */
  CHIP_Init();
  
  /* Initialize the serial console */
  RETARGET_SerialInit();

  /* Print messages */
  printf("\f");
  printf("DPLL Example\r\n");
  printf("HFRCO is tuned to\r\n");
  printf("24.576 MHz with LFXO\r\n\n");
  
#if CMU_OUT_EN == 1
  /* Configure HFSRCCLK to CLKOUT1  */
  GPIO_PinModeSet(CMU_OUT1_PORT, CMU_OUT1_PIN, gpioModePushPull, 0);
  GPIO->CMUROUTE.ROUTEEN = GPIO_CMU_ROUTEEN_CLKOUT1PEN; 
  GPIO->CMUROUTE.CLKOUT1ROUTE = CMU_OUT1_PORT|CMU_OUT1_PIN;
  CMU->EXPORTCLKCTRL = CMU_EXPORTCLKCTRL_CLKOUTSEL1_HCLK;
#endif
  
  /* Setup and start DPLL */
  if (initDPLL(cmuOsc_LFXO, DPLL_FACTOR_N, DPLL_FACTOR_M))
  {
    printf("DPLL is locked\r\n\n");
    printf("Transmit 48kHz 32-bit\r\n");
    printf("I2S data on USART3:\r\n\n");
    printf(" I2S_WS on PC3 (EXP10)\r\n");
    printf(" I2S_CLK on PD3(EXP9)\r\n");
    printf(" I2S_TX on PC2(EXP8)\r\n");
  }
  else
  {
    printf("DPLL fails to lock!\r\n");
    printf("Select another\r\n");
    printf("HFRCO frequency band\r\n");
    while (1);
  }
  
  /* Use HFRCODPLL as I2S clock */
  initI2S();

  /* Infinite loop */
  while (1)
  {
    /* Transmit I2S data continuously */
    USART_TxDouble(I2S_USART, I2S_DUMMY_DATA);
  }
}
