/**************************************************************************//**
 * @file main_cmu_example.c
 * @brief CMU examples for EFR32 Gecko Series 2
 * @version  1.11

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

 /*
 This project demonstrates the routines for calibrating the LFRCO, HFRCODPLL, and HFRCOEM23
clocks from a HFXO and LFXO external oscillator. The example displays the initial and 
resulting tuning values. Multiple calibrations can be made with each clock to establish a
baseline average. The RC oscillator output is driven to an expansion header pin (EXP4) so
that the calibrated clock can be observed with an oscilloscope.

How To Test:
1.  Update the kit's firmware from the Simplicity Launcher (if necessary)
2.  Build the project and download to the Starter Kit
3.  Run the example
4.  Press PB1 to cycle through the three different RC oscillators available for calibrations
5.  Press PB0 to start a calibration routine.
6.  Observe the old and new tuning values. Attach an oscilloscope probe to PC00 (EXP4) to
    observe the RC oscillator clock output.
7.  If disabling the clock output (#define CMU_OUT_EN 0 on line 64 of main_cmu_example_s2.h),
    main while loop will enter EM2 instead of EM1. When this is the case, PB0 and PB1 of the
    WSTK can no longer wake the device, as only GPIO on ports A or B can be used as external
    wake pins. See required jumpers specific to device below.

Peripherals Used:
HFXO        - 38.4 MHz for Series 2
LFXO        - 32768 Hz for Series 2
LFRCO       - 32768 Hz for Series 2
HFRCODPLL   - 19 MHz
HFRCOEM23   - 19 MHz (EFR32xG21 only)

***
               
Board:  Silicon Labs EFR32xG21 Radio Board (BRD4181A) + 
        Wireless Starter Kit Mainboard
Device: EFR32MG21A010F1024IM32
PC00 - GPIO Push/Pull output, Expansion Header Pin 4, WSTK Pin 1
PD02 - GPIO pull filtered input, Push Button 0 (PB0), WSTK Pin 4/EXP Pin 7
PD03 - GPIO pull filtered input, Push Button 1 (PB1), WSTK Pin 6/EXP Pin 9
for #define CMU_OUT_EN 0:
Jumpers must be placed from P4 and P6 to P14 and P20, respectively, on WSTK Breakout Connector J101.
PA00 - GPIO pull filtered input, Re-mapped Push Button 0 (PB0), WSTK Breakout Pin 14
PA03 - GPIO pull filtered input, Re-mapped Push Button 1 (PB1), WSTK Breakout Pin 20

***

Board:  Silicon Labs EFR32xG21 Radio Board (BRD4182A) +
        Wireless Starter Kit Mainboard
Device: EFR32MG22C224F512IM40
PC00 - GPIO Push/Pull output, Expansion Header Pin 4, WSTK Pin 1
PD02 - GPIO pull filtered input, Push Button 0 (PB0), WSTK Pin 4/EXP Pin 7
PD03 - GPIO pull filtered input, Push Button 1 (PB1), WSTK Pin 6/EXP Pin 9
for #define CMU_OUT_EN 0:
Jumpers must be placed from P4 and P6 to P14 and P20, respectively, on WSTK Breakout Connector J101.
PA00 - GPIO pull filtered input, Re-mapped Push Button 0 (PB0), WSTK Breakout Pin 14
PA03 - GPIO pull filtered input, Re-mapped Push Button 1 (PB1), WSTK Breakout Pin 20

*/

#include "main_cmu_example_s2.h"

/* Global variables */
volatile bool runKey;
volatile bool menuKey;
volatile bool progRun;
volatile uint8_t menuLevel;

/**************************************************************************//**
 * @brief Main function
 *****************************************************************************/
int main(void)
{
  /* Tune counter */
  uint8_t tuneCount = 0;
  
  /* Chip errata */
  CHIP_Init();

#if defined(_SILICON_LABS_32B_SERIES_2_CONFIG_2)
  /* Enable Oscillator APB clocks for EFR32xG22/BRD4182A */
  CMU_ClockEnable(cmuClock_HFXO, true);
  CMU_ClockEnable(cmuClock_LFXO, true);
  CMU_ClockEnable(cmuClock_LFRCO, true);
  CMU_ClockEnable(cmuClock_HFRCO0, true);
  CMU_ClockEnable(cmuClock_DPLL0, true);
#endif

  /* Initialize HFXO with specific parameters */
  CMU_HFXOInit_TypeDef hfxoInit = CMU_HFXOINIT_DEFAULT;
  CMU_HFXOInit(&hfxoInit);

  /* Initialize LFXO with specific parameters */
  CMU_LFXOInit_TypeDef lfxoInit = CMU_LFXOINIT_DEFAULT;
  CMU_LFXOInit(&lfxoInit);
  
  /* Optimize flash wait state with default HFRCO as HCLK   */
#if defined(_HFRCO_CAL_MASK)
  /* Are these the right values to use? */
  CMU_HFRCODPLLBandSet(cmuHFRCODPLLFreq_13M0Hz);
#else
  CMU_HFRCODPLLBandSet(cmuHFRCODPLLFreq_19M0Hz);
#endif
  
#if defined (_MSC_CTRL_IFCREADCLEAR_MASK)  
  /* Enable atomic read-clear operation on reading IFC register */
  MSC->CTRL |= MSC_CTRL_IFCREADCLEAR;
#endif
  
  /* Setup GPIO with interrupts to serve the pushbuttons */
  gpioSetup();
  
  /*try sending out HCLK just for fun*/
  CMU->EXPORTCLKCTRL = CMU_EXPORTCLKCTRL_CLKOUTSEL1_HCLK;

  /* try using HFXO as timebase for USART to try to get more stable output*/
  CMU_ClockSelectSet(cmuClock_SYSCLK,cmuSelect_HFXO);
  RETARGET_SerialInit();
  printf("RC calibration example start\r\n\r\n");

  /* Display menu */
  printMenu();
  
  /* Enter EM1 or EM2 to wait for key press, EM1 is for RC clock output */
  while (1)
  {
    RETARGET_SerialFlush();
#if CMU_OUT_EN == 1
    EMU_EnterEM1();
#else
    EMU_EnterEM2(false);
#endif
    
    if (menuKey)         /* Left side push button to browse menu */
    {
      printMenu();
      tuneCount = 0;
    }

    if (runKey)         /* Right side push button to run selected menu */
    {
      runKey = false;
      progRun = true;

      printf("Starting calibration...\r\n\r\n");
      switch (menuLevel)
      {
#if defined(_SILICON_LABS_32B_SERIES_2_CONFIG_1)  /* LFRCO0 user calibration available on EFR32xG21 only */
      case LFRCO_CAL:
        /* Display the existing tuning value, then start calibration */
        printf("Old tune value: %3lu\r\n\r\n", CMU_OscillatorTuningGet(cmuOsc_LFRCO));
        calibrateRcOsc(cmuOsc_LFRCO, LFRCO_TUNE_FREQ, false, false);
        while(!endOfTune());

        /* Display the tuning result if finish */
        printf("Tune count: %3d\r\n", ++tuneCount);
        printf("UpCnt Actual: %lu\r\n", getUpcount(false));
        printf("New tune value: %3lu\r\n\r\n", CMU_OscillatorTuningGet(cmuOsc_LFRCO));
        break;
#endif
        
      case HFRCODPLL_CAL:
        /* Get tuning value from DI page if existing band != tuning band */
        if (CMU_HFRCODPLLBandGet() != HFRCODPLL_TUNE_BAND)
        {
          CMU_HFRCODPLLBandSet(HFRCODPLL_TUNE_BAND);
        }
        /* Display the existing tuning value, then start calibration */
        printf("Old tune value: %3lu\r\n", CMU_OscillatorTuningGet(cmuOsc_HFRCODPLL));
        calibrateRcOsc(cmuOsc_HFRCODPLL, HFRCODPLL_TUNE_FREQ, false, true);

#if defined(_HFRCO_CAL_FINETUNING_MASK)
        if ((HFRCO0->CAL & _HFRCO_CAL_FINETUNING_MASK) != 0)
        {
          printf("Old finetune val: %3lu\r\n", CMU_OscillatorFineTuningGet(cmuOsc_HFRCODPLL));
        }
#endif
        printf("UpCnt Target: %lu\r\n\r\n", getUpcount(true));
        while(!endOfTune());

        /* Display the tuning result if finish */
        printf("Tune count: %3d\r\n", ++tuneCount);
        printf("UpCnt Actual: %lu\r\n", getUpcount(false));
        printf("New tune value: %3lu\r\n", CMU_OscillatorTuningGet(cmuOsc_HFRCODPLL));
#if defined(_CMU_HFRCOCTRL_FINETUNINGEN_MASK)
        if ((HFRCODPLL->CAL & CMU_HFRCOCAL_FINETUNING) != 0)
        {
          printf("New finetune val: %3lu\r\n", CMU_OscillatorFineTuningGet(cmuOsc_HFRCO));
        }
#endif
        printf("\r\n");
        break;

#if defined(_SILICON_LABS_32B_SERIES_2_CONFIG_1)  /* HFRCOEM23 available on EFR32xG21 only */
      case HFRCOEM23_CAL:
        /* Get tuning value from DI page if existing band != tuning band */
        if (CMU_HFRCOEM23BandGet() != HFRCOEM23_TUNE_BAND)
        {
          CMU_HFRCOEM23BandSet(HFRCOEM23_TUNE_BAND);
        }

        /* Display the existing tuning value, then start calibration */
        printf("Old tune value: %3lu\r\n", CMU_OscillatorTuningGet(cmuOsc_HFRCOEM23));
        calibrateRcOsc(cmuOsc_HFRCOEM23, HFRCOEM23_TUNE_FREQ, false, true);
#if defined(_HFRCO_CAL_FINETUNING_MASK)
        if ((HFRCOEM23->CAL & _HFRCO_CAL_FINETUNING_MASK) != 0)
        {
          printf("Old finetune val: %3lu\r\n", CMU_OscillatorFineTuningGet(cmuOsc_HFRCOEM23));
        }
#endif
        printf("UpCnt Target: %lu\r\n\r\n", getUpcount(true));
        while(!endOfTune());

        /* Display the tuning result if finish */
        printf("Tune count: %3d\r\n", ++tuneCount);
        printf("UpCnt Actual: %lu\r\n", getUpcount(false));
        printf("New tune value: %3lu\r\n", CMU_OscillatorTuningGet(cmuOsc_HFRCOEM23));
#if defined(_HFRCO_CAL_FINETUNING_MASK)
        if ((HFRCOEM23->CAL & _HFRCO_CAL_FINETUNING_MASK) != 0)
        {
          printf("New finetune val: %3lu\r\n", CMU_OscillatorFineTuningGet(cmuOsc_HFRCOEM23));
        }
#endif
        printf("\r\n");
        break;
#endif

      default:
        break;
      }

      printf(RUN_BUTTON);
      printf(EXIT_BUTTON);
      progRun = false;
    }
  }
}
