/*****************************************************************************
 * @file main_cmu_hfxo_autostart.c
 * @brief CMU automatic HFXO start example for  EFR32/EFM32 Series 2
 * @version  1.11

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

 /*
 This project demonstrates the time it takes, when using HFXO as the system clock, to
wake from EM2 and begin code execution. The code example default configuration relies
on the HFXO auto-start feature. The project switches the SYSCLK source to the HFXO
and then goes into EM2. Pressing Push Button 0 on the WSTK wakes the program from EM2 and immediately
toggles PC02 (EXP8). When the system first wakes, FSRCO is the clock source for SYSCLK.
The HFXO status register is polled as it warms back up to be used as the SYSCLK. When 
the HFXO is ready, PC03 (EXP10) is toggled. The time span required for HFXO to settle
can measured using an oscilloscope or logic analyzer and measuring the time between toggle
events on the two pins.

How To Test:
1.  Update the kit's firmware from the Simplicity Launcher (if necessary)
2.  [BRD4181A/xG21 only] Apply jumper from Push Button 0 (PD02/WSTKBO pin P4) to pin PA00 (WSTKBO pin P14)
	to enable wake-up from EM2 (see device settings below)
3.  Build the project and download to the Starter Kit
4.  Run the example
5.  Observe EXP8 and EXP10 with an oscilloscope. Measure time span between the toggle on
    EXP8 and the toggle on EXP10 to determine warm-up time for HFXO.
	
Peripherals Used:
HFXO    - Configured as SYSCLK; auto-start enabled
USARTn  - VCOM
CMU/LFXO/HFXO

***
               
Board:  Silicon Labs EFR32xG21 Radio Board (BRD4181A) + 
        Wireless Starter Kit Mainboard
Device: EFR32MG21A010F1024IM32

Input:
Note: Jumper must be placed from P4 to P14 on WSTK Breakout Connector (WSTKBO) J101.
PA00(WSTKBO pin P14) - 	GPIO pull filtered input, Re-mapped via jumper
						from Push Button 0 (PD02/WSTKBO pin P4)
Push Button 1 - 		Debug trap pin.  Press button 1 and reset device to allow debugger to connect.

Output:
PC02 (EXP 8) - 	GPIO Push/Pull output, Expansion Header Pin 8
PC03 (EXP 10)- 	GPIO Push/Pull output, Expansion Header Pin 10
VCOM - 			UI messages via retargeted serial to VCOM

***

Board:  Silicon Labs EFR32xG22 Radio Board (BRD4182A) +
        Wireless Starter Kit Mainboard
Device: EFR32MG22C224F512IM40

Input:
Push Button 0 (PB00) - 	GPIO pull filtered input
Push Button 1 - 		Debug trap pin.  Press button 1 and reset device to allow debugger to connect.

Output:
PC02 (EXP 8) - 	GPIO Push/Pull output, Expansion Header Pin 8
PC03 (EXP 10)- 	GPIO Push/Pull output, Expansion Header Pin 10
VCOM - 			UI messages via retargeted serial to VCOM

*/

#include "main_cmu_hfxo_on_demand_s2.h"

#define HXFO_ON_DEMAND          1
#define USE_HFXO_AS_SYSCLK      1

/*******************************************************************************
 ***************************   GLOBAL VARIABLES   *******************************
 ******************************************************************************/

volatile uint8_t wakeUpCnt;     /* EM2 wake-up count */

/**********************************************************
 * @brief Interrupt handler for even GPIO. 
 **********************************************************/
void GPIO_EVEN_IRQHandler(void)
{
  /* Toggle pin to denote GPIO wake-up IRQ handler entry */
  GPIO_PinOutToggle(GPIO_INT_TOGGLE_PORT, GPIO_INT_TOGGLE_PIN);
  GPIO_IntClear(_GPIO_IF_MASK);
}

/**************************************************************************//**
 * @brief GPIO initialization
 *****************************************************************************/
void initGPIO(void)
{
  /* GPIO Clock enable for EFR32xG22 */
  CMU_ClockEnable(cmuClock_GPIO, true);

  // GPIO IRQ wake from EM2 input pin
  GPIO_PinModeSet(GPIO_INPUT_PORT, GPIO_INPUT_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(GPIO_INPUT_PORT, GPIO_INPUT_PIN, GPIO_INPUT_PIN, false, true, true);

  // GPIO IRQ handler entry output pin
  GPIO_PinModeSet(GPIO_INT_TOGGLE_PORT, GPIO_INT_TOGGLE_PIN, gpioModePushPull, 0);
  //HFXO RDY output pin
  GPIO_PinModeSet(HFXO_RDY_PORT, HFXO_RDY_PIN, gpioModePushPull, 0);
  
  // Enable GPIO EVEN IRQ NVIC source
  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);
  
  GPIO_PinModeSet(RECOVERY_PORT, RECOVERY_PIN, gpioModeInput, 1);
}

/**************************************************************************//**
 * @brief  Main function
 *****************************************************************************/
int main(void)
{
  /* Initialize chip */
  CHIP_Init();

#if defined(_SILICON_LABS_32B_SERIES_2_CONFIG_2)
  /* Enable Oscillator APB clocks for EFR32xG22/BRD4182A */
  CMU_ClockEnable(cmuClock_HFXO, true);
  CMU_ClockEnable(cmuClock_LFXO, true);
#endif

  /* Initialize HFXO with default parameters */
  CMU_HFXOInit_TypeDef hfxoInit = CMU_HFXOINIT_WSTK_DEFAULT;
  CMU_HFXOInit(&hfxoInit);

  /* Initialize LFXO with default parameters */
  CMU_LFXOInit_TypeDef lfxoInit = BSP_LFXOINIT_DEFAULT;
  CMU_LFXOInit(&lfxoInit);
  
  /* Setup GPIO */
  initGPIO();
  
  /* Trap to allow debugger to connect */
  if (GPIO_PinInGet(RECOVERY_PORT, RECOVERY_PIN) == 0){
     while(1);
  }

  /******************************************************************
   * Enable automatic start-up of the HFXO, its selection as the
   * HFSRCLK, and HFRCO shutdown upon exit from EM2/EM3 into EM0/EM1.
   *
   * Note that this has immediate effect in EM0 simply by nature of
   * running the code that sets the relevant CMU_HFXOCTRL bits.
   ******************************************************************/
#if USE_HFXO_AS_SYSCLK  
  CMU_ClockSelectSet(cmuClock_SYSCLK , cmuSelect_HFXO);
#else
  /* FSRCO is default SYSCLK */
#endif
  
  RETARGET_SerialInit();
  printf("\fHFXO autostart example\r\n\r\n");
   
#if USE_AUTO_HFXO == 0
  printf("Normal HFRCO Start\r\n");
  printf("Example\r\n\r\n");
#else
#if USE_LFTIMEOUT == 0
  printf("Normal HFXO\r\n");
  printf("Auto Start Example\r\n\r\n");
#else
  printf("Deterministic HFXO\r\n");
  printf("Auto Start Example\r\n\r\n");
#endif
#endif

  /* Print output message */
  printf("EXP Header Outputs:\r\n");
  printf("GPIO IRQ (P%c%d) on Pin %d\r\n", (char)('A' + (char)(GPIO_INT_TOGGLE_PORT)), GPIO_INT_TOGGLE_PIN,  EXP_GPIO_OUT);
  printf("HFXO RDY (P%c%d) on Pin %d\r\n", (char)('A' + (char)(HFXO_RDY_PORT)), HFXO_RDY_PIN,  EXP_HFXO_RDY);
  printf("\r\n");

  /* Print connection message */
  printf("Press PB0 to wake from EM2 \r\n");
  printf("Compare EXP header pins %d and %d to measure HFXO wake time\r\n\r\n", EXP_GPIO_OUT, EXP_HFXO_RDY);
  
  while (1)
  {
    /* Enter EM2 - do this with the debugger disconnected */
#if (HXFO_ON_DEMAND == 0)
   HFXO0->CTRL = HFXO_CTRL_DISONDEMAND;
#endif
	
    EMU_EnterEM2(false);

#if (HXFO_ON_DEMAND == 0)
	HFXO0->CTRL = HFXO_CTRL_FORCEEN;
#endif

	/* No interrupt for 'autostart' so just poll */
    while((HFXO0->STATUS & HFXO_STATUS_RDY) == 0);
    GPIO_PinOutToggle(HFXO_RDY_PORT, HFXO_RDY_PIN);
    
    /* Update wake-up count */
    printf("EM2 wake count: %3d\r\n", ++wakeUpCnt);
  }
}
