/*****************************************************************************
 * @file main_rtc_rtcc.c
 * @brief Real Time Counter Demo Application
 * @author Silicon Labs
 * @version 1.09
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
 
#include "em_device.h"
#include "em_chip.h"
#include "em_rtc.h"
#include "em_rtcc.h"
#include "em_cmu.h"
#include "em_emu.h"

/* Defines*/
#define LFRCO_FREQUENCY                 32768
#define WAKEUP_INTERVAL_MS              500
#define RTC_COUNT_BETWEEN_WAKEUP        (((LFRCO_FREQUENCY * WAKEUP_INTERVAL_MS) / 1000)-1)


#if defined(RTCC_PRESENT) && (RTCC_COUNT > 0)
/**************************************************************************//**
 * @brief RTCC Interrupt Handler, clears the flag.
 *****************************************************************************/
void RTCC_IRQHandler(void)
{
  /* Clear interrupt source */
  RTCC_IntClear(RTCC_IFC_CC1);
}
#else
/**************************************************************************//**
 * @brief RTC Interrupt Handler, clears the flag.
 *****************************************************************************/
void RTC_IRQHandler(void)
{
  /* Clear interrupt source */
  RTC_IntClear(RTC_IFC_COMP0);
}
#endif


/**************************************************************************//**
 * @brief  Start LFRCO for RTC/RTCC
 * Starts the low frequency RC oscillator (LFRCO) and routes it to the RTC/RTCC
 *****************************************************************************/
void startLfrcoForRtc(void)
{
  /* Starting LFRCO and waiting until it is stable */
  CMU_OscillatorEnable(cmuOsc_LFRCO, true, true);

  /* Enabling clock to the interface of the low energy modules */
  CMU_ClockEnable(cmuClock_CORELE, true);

#if defined(RTCC_PRESENT) && (RTCC_COUNT > 0)
  /* Routing the LFRCO clock to the RTCC */
  CMU_ClockSelectSet(cmuClock_LFE, cmuSelect_LFRCO);
  CMU_ClockEnable(cmuClock_RTCC, true);
#else
  /* Routing the LFRCO clock to the RTC */
  CMU_ClockSelectSet(cmuClock_LFA,cmuSelect_LFRCO);
  CMU_ClockEnable(cmuClock_RTC, true);
#endif
}


/**************************************************************************//**
 * @brief  Setup RTC/RTCC
 * On compare match with compare channel, clear counter and set interrupt
 *****************************************************************************/
void setupRtc(void)
{
  /* Configuring clocks in the Clock Management Unit (CMU) */
  startLfrcoForRtc();
  
#if defined(RTCC_PRESENT) && (RTCC_COUNT > 0)
  RTCC_Init_TypeDef rtccInit = RTCC_INIT_DEFAULT;
  RTCC_CCChConf_TypeDef rtccInitCompareChannel = RTCC_CH_INIT_COMPARE_DEFAULT;
    
  rtccInit.cntWrapOnCCV1 = true;        /* Clear counter on compare match */
  rtccInit.presc = rtccCntPresc_1;

  /* Setting the compare value of the RTCC */
  RTCC_ChannelInit(1, &rtccInitCompareChannel);
  RTCC_ChannelCCVSet(1, RTC_COUNT_BETWEEN_WAKEUP);
  
  /* Enabling Interrupt from RTCC */
  RTCC_IntEnable(RTCC_IEN_CC1);
  NVIC_ClearPendingIRQ(RTCC_IRQn);
  NVIC_EnableIRQ(RTCC_IRQn);

  /* Initialize the RTCC */
  RTCC_Init(&rtccInit);
#else
  RTC_Init_TypeDef rtcInit = RTC_INIT_DEFAULT;

  /* Setting the compare value of the RTC */
  RTC_CompareSet(0, RTC_COUNT_BETWEEN_WAKEUP);

  /* Enabling Interrupt from RTC */
  RTC_IntEnable(RTC_IEN_COMP0);
  NVIC_ClearPendingIRQ(RTC_IRQn);
  NVIC_EnableIRQ(RTC_IRQn);

  /* Initialize the RTC */
  RTC_Init(&rtcInit);
#endif
}


/**************************************************************************//**
 * @brief  Main function
 *****************************************************************************/
int main(void)
{
  /* Initialize chip */
  CHIP_Init();

#if defined( _EMU_DCDCCTRL_MASK )
  /* Init DCDC regulator */
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif
  
  /* Setting up rtc */
  setupRtc();

  while (1)
  {
    /* Forever enter EM2. */
    EMU_EnterEM2(false);
  }
}
