/*****************************************************************************
 * @file main_rtc_calendar.c
 * @brief Real Time Counter and Calendar Demo Application
 * @author Silicon Labs
 * @version 1.09
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include <stdio.h>
#include "em_device.h"
#include "em_chip.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "em_rtcc.h"

#include "display.h"
#include "textdisplay.h"
#include "retargettextdisplay.h"

#define START_TIME      0x00235945      /* 23:59:45 */
#define START_DATE      0x06160228      /* 2016 Feb 28 */

uint32_t rtccFlag;

/**************************************************************************//**
* @brief Dummy funtion for LCD initialization, use RTCC to toggle GPIO
*****************************************************************************/
EMSTATUS PAL_GpioPinAutoToggle (unsigned int gpioPort,
                                unsigned int gpioPin,
                                unsigned int frequency)
{
  EMSTATUS status = EMSTATUS_OK;
  return status;
}

/**************************************************************************//**
* @brief RTCC Interrupt handler
*****************************************************************************/
void RTCC_IRQHandler(void)
{
  /* Read and clear interrupt source */
  rtccFlag = RTCC->IFC;

  /* Toggle GPIO pin for LCD display. */
  GPIO_PinOutToggle((GPIO_Port_TypeDef)LCD_PORT_EXTCOMIN, LCD_PIN_EXTCOMIN );
}

/**************************************************************************//**
* @brief Setup RTCC for calendar.
*****************************************************************************/
void rtccSetup(void)
{
  RTCC_Init_TypeDef rtccInit = RTCC_INIT_DEFAULT;

  /* Enable LFE domain for RTCC */
  CMU_ClockEnable(cmuClock_CORELE, true);
  CMU_ClockSelectSet(cmuClock_LFE, cmuSelect_LFXO);

  /* Enable RTC clock */
  CMU_ClockEnable(cmuClock_RTCC, true);

  /* In calendar mode, the pre-counter should be configured to give ticks with a period of   */
  /* one second, i.e. RTCC_CTRL_CNTTICK should be set to PRESC, and the CNTPRESC bitfield    */
  /* of the RTCC_CTRL register should be set to DIV32768 if a 32768 Hz clock source is used. */
  rtccInit.enable   = false;
  rtccInit.presc = rtccCntPresc_32768;
  rtccInit.cntMode = rtccCntModeCalendar;
  RTCC_Init(&rtccInit);

  /* Initialize date and time */
  RTCC_DateSet(START_DATE);
  RTCC_TimeSet(START_TIME);
    
  /* Enable required interrupt */
  RTCC_IntEnable(RTCC_IEN_CNTTICK + RTCC_IEN_DAYTICK);

  /* Enable RTCC interrupt */
  NVIC_ClearPendingIRQ(RTCC_IRQn);
  NVIC_EnableIRQ(RTCC_IRQn);

  /* Start Counter */
  RTCC_Enable(true);
}

/**************************************************************************//**
* @brief Print RTCC date.
*****************************************************************************/
void rtccPrintDate(void)
{
  uint32_t rtccTemp;
  
  rtccTemp = RTCC_DateGet();
  printf ("Date:20%lx%lx/%lx%lx/%lx%lx\n\n", 
          (rtccTemp >> _RTCC_DATE_YEART_SHIFT) & 0x0f,
          (rtccTemp >> _RTCC_DATE_YEARU_SHIFT) & 0x0f,
          (rtccTemp >> _RTCC_DATE_MONTHT_SHIFT) & 0x0f,
          (rtccTemp >> _RTCC_DATE_MONTHU_SHIFT) & 0x0f,
          (rtccTemp >> _RTCC_DATE_DAYOMT_SHIFT) & 0x0f,
          (rtccTemp & 0x0f));
}

/**************************************************************************//**
* @brief Print RTCC time.
*****************************************************************************/
void rtccPrintTime(void)
{
  uint32_t rtccTemp;
  
  rtccTemp = RTCC_TimeGet();
  printf ("Time:%lx%lx:%lx%lx:%lx%lx\n", 
          (rtccTemp >> _RTCC_TIME_HOURT_SHIFT),
          (rtccTemp >> _RTCC_TIME_HOURU_SHIFT) & 0x0f,
          (rtccTemp >> _RTCC_TIME_MINT_SHIFT) & 0x0f,
          (rtccTemp >> _RTCC_TIME_MINU_SHIFT) & 0x0f,
          (rtccTemp >> _RTCC_TIME_SECT_SHIFT) & 0x0f,
          (rtccTemp & 0x0f));
}

/**************************************************************************//**
 * @brief  Main function
 *****************************************************************************/
int main(void)
{
  /* Chip errata */
  CHIP_Init();

  /* Init DCDC regulator */
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);

  /* Enable atomic read-clear operation on reading IFC register */
  MSC->CTRL |= MSC_CTRL_IFCREADCLEAR;
  
  /* Initialize the display module. */
  DISPLAY_Init();

  /* Initialize RTCC. */
  rtccSetup();

  /* Retarget stdio to a text display. */
  if (RETARGET_TextDisplayInit() != TEXTDISPLAY_EMSTATUS_OK)
  {
    while (1) ;
  }

  /* Output time and date on Memory LCD */
  printf("RTCC Calendar\nDemo\n");
  rtccPrintTime();
  rtccPrintDate();

  /* Update Memory LCD display every second */
  while (1)
  {
    EMU_EnterEM2(false);
    if (rtccFlag & RTCC_IEN_DAYTICK)
    {
      rtccFlag = 0;
      rtccPrintTime();
      rtccPrintDate();
    }
    else
    {
      rtccFlag = 0;
      rtccPrintTime();
    }
  }
}
