/******************************************************************************
 * @file energymodes.c
 * @brief Energy Modes Demo Application
 * @author Silicon Labs
 * @version  1.11

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#include <stdlib.h>
#include "em_device.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "rtcdriver.h"
#include "config.h"

// Defines
// When this define is present, the program will enter EM4 at the end.
// If it is not present, EM3 is entered.
#define ENTER_EM4

// Clock defines
#define RTC_TIMEOUT_MS          3000

// Timer used for bringing the system back to EM0.
RTCDRV_TimerID_t xTimerForWakeUp;


/******************************************************************************
 * @brief Enables all clocks and waits
 *****************************************************************************/
void enableAllClocks(void)
{
  // Turning on all oscillators
  // Then waiting for all oscillators to stabilize
  CMU_OscillatorEnable(cmuOsc_AUXHFRCO, true, true);
  CMU_OscillatorEnable(cmuOsc_HFXO, true, true);
  CMU_OscillatorEnable(cmuOsc_LFRCO, true, true);
  CMU_OscillatorEnable(cmuOsc_LFXO, true, true);

  // Switching the CPU clock source to HFXO
  // This will increase current consumption
  CMU_ClockSelectSet(cmuClock_HF, cmuSelect_HFXO);

  // Enabling clocks to all core and peripheral modules
#if defined( _CMU_HFCORECLKEN0_MASK )
  CMU->HFCORECLKEN0 = 0xFFFFFFFF;
#endif
#if defined( _CMU_HFBUSCLKEN0_MASK )
  CMU->HFBUSCLKEN0 = 0xFFFFFFFF;
#endif
  CMU->HFPERCLKEN0  = 0xFFFFFFFF;
#if defined(_CMU_HFPERCLKEN1_MASK)
  CMU->HFPERCLKEN1  = 0xFFFFFFFF;
#endif
  // Wait
  RTCDRV_Delay(RTC_TIMEOUT_MS);
}

/******************************************************************************
 * @brief Disables all clocks and waits
 *****************************************************************************/
void disableAllClocks(void)
{
  // Switching the CPU clock source to HFRCO
  CMU_ClockSelectSet(cmuClock_HF, cmuSelect_HFRCO);

  // Disabling every oscillator except hfrco and lfxo
  // NOTE: MAKE SURE NOT TO DISABLE THE CURRENT CPU CLOCK!!
  CMU_OscillatorEnable(cmuOsc_AUXHFRCO, false, true);
  CMU_OscillatorEnable(cmuOsc_HFXO, false, true);
  CMU_OscillatorEnable(cmuOsc_LFRCO, false, true);

  // Disabling all unused clocks. The LE clock must be on in order to use the RTC
#if defined( _CMU_HFCORECLKEN0_MASK )
  CMU->HFCORECLKEN0 = CMU_HFCORECLKEN0_LE;
#endif
#if defined( _CMU_HFBUSCLKEN0_MASK )
  CMU->HFBUSCLKEN0 = CMU_HFBUSCLKEN0_LE;
#endif
  CMU->HFPERCLKEN0  = 0;
#if defined(_CMU_HFPERCLKEN1_MASK)
  CMU->HFPERCLKEN1  = 0;
#endif

  // Wait
  RTCDRV_Delay(RTC_TIMEOUT_MS);
}

/******************************************************************************
 * @brief Changes band and prescales core clock
 *****************************************************************************/
void downScaleCoreClock(void)
{
  // Changing the band of the HFRCO
#if defined( _CMU_HFRCOCTRL_BAND_MASK )
  CMU_HFRCOBandSet(cmuHFRCOBand_7MHz);
#endif
#if defined( _CMU_HFRCOCTRL_FREQRANGE_MASK )
  CMU_HFRCOBandSet(cmuHFRCOFreq_7M0Hz);
#endif
  // Setting prescaling of the CPU clock
  CMU_ClockDivSet(cmuClock_CORE, cmuClkDiv_4);

  // Wait
  RTCDRV_Delay(RTC_TIMEOUT_MS);
}

/******************************************************************************
 * @brief Config EM4 Shut Off mode
 *****************************************************************************/
static void configEM4(void)
{
#if defined(CRYOTIMER_PRESENT)
  // Configure EM4 behavior
  EMU_EM4Init_TypeDef init = EMU_EM4INIT_DEFAULT;

  init.em4State = emuEM4Shutoff;  // emuEM4Hibernate also works
  init.retainUlfrco = true;
  init.pinRetentionMode = emuPinRetentionDisable;
  EMU_EM4Init(&init);
#endif
}
/******************************************************************************
 * @brief  EnergyModes function
 *****************************************************************************/
void energyModes(void)
{
  // Initialize RTC timer.
  RTCDRV_Init();
  RTCDRV_AllocateTimer(&xTimerForWakeUp);

  // Turn on all clocks and oscillators. Then wait in EM0
  enableAllClocks();

  // Turn off all clocks and wait in EM0
  // Current consumption drops drastically
  disableAllClocks();

  // Adjusting core clock down to further reduce current consumption
  downScaleCoreClock();

  // EM1 - 3 sec
  RTCDRV_StartTimer(xTimerForWakeUp, rtcdrvTimerTypeOneshot, RTC_TIMEOUT_MS, NULL, NULL);
  EMU_EnterEM1();

  // EM2 - 3 sec
  RTCDRV_StartTimer(xTimerForWakeUp, rtcdrvTimerTypeOneshot, RTC_TIMEOUT_MS, NULL, NULL);
  EMU_EnterEM2(false);

#ifdef ENTER_EM4
  // Enter EM4
  configEM4();
  EMU_EnterEM4();
#else
  // Enter EM3
  EMU_EnterEM3(false);
#endif
  return;
}


