/******************************************************************************
 * @file em4_wakeup.c
 * @brief EM4 Wake up example
 * @author Silicon Labs
 * @version  1.11

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#include <stdio.h>
#include "em_device.h"
#include "em_chip.h"
#include "em_rtc.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "em_gpio.h"
#include "em_burtc.h"
#include "em_rmu.h"
#include "em_cryotimer.h"
#include "config.h"

// System ms tick counter
static uint32_t volatile msTicks = 0;
#if defined(TFT_LCD)
DISPLAY_Device_t displayDevice;    // Display device handle.
#endif

/**************************************************************************//**
 * @brief Triggered every ms
 *****************************************************************************/
void SysTick_Handler(void)
{
  msTicks++;       // increment counter necessary in Delay()
}

/**************************************************************************//**
 * @brief Delays number of msTick Systicks (typically 1 ms)
 * @param dlyTicks Number of ticks to delay
 *****************************************************************************/
void delay(uint32_t dlyTicks)
{
  uint32_t curTicks;

  curTicks = msTicks;
  while ((msTicks - curTicks) < dlyTicks) ;
}


/**************************************************************************//**
 * @brief   Configure settings for EM4 before going to sleep
 *****************************************************************************/
static void configEM4(void)
{
#if defined(BURTC_PRESENT)
  EMU_EM4Init_TypeDef em4Init = EMU_EM4INIT_DEFAULT;
  em4Init.lockConfig    = true;             // Lock regulator, oscillator and BOD configuration.
										    // This needs to be set when using the
										    // voltage regulator in EM4
  em4Init.osc           = emuEM4Osc_ULFRCO; // Select ULFRCO
  em4Init.buRtcWakeup   = true;             // BURTC compare or overflow will generate reset
  em4Init.vreg          = true;             // Enable voltage regulator. Needed for BURTC
  EMU_EM4Init(&em4Init);
#endif
#if defined(CRYOTIMER_PRESENT)
  // Configure EM4 behavior
  EMU_EM4Init_TypeDef init = EMU_EM4INIT_DEFAULT;

  init.em4State = emuEM4Shutoff;  // Set emuEM4Hibernate to enter Hibernate mode
  init.retainUlfrco = true;
  init.pinRetentionMode = emuPinRetentionDisable;
  EMU_EM4Init(&init);
#endif
}

#if defined(BURTC_PRESENT)
// The retention registers used for counter values
#define BURTC_WAKEUP_COUNT BURTC->RET[0].REG
#define GPIO_WAKEUP_COUNT BURTC->RET[1].REG

/**************************************************************************//**
 * @brief Set up BURTC to count and trigger a wakeup in EM4
 *****************************************************************************/
static void configBURTC(void)
{
  BURTC_Init_TypeDef burtcInit = BURTC_INIT_DEFAULT;

  burtcInit.mode      = burtcModeEM4;           // BURTC is enabled in EM0-EM4
  burtcInit.clkSel    = burtcClkSelULFRCO;      // Select ULFRCO as clock source
  burtcInit.clkDiv    = burtcClkDiv_2;          // Choose 1kHz ULFRCO clock frequency

  BURTC_CompareSet(0, BURTC_COUNT_TO_WAKEUP);   // Set top value for comparator
  BURTC_IntEnable(BURTC_IF_COMP0);              // Enable compare interrupt flag
  BURTC_Init(&burtcInit);
}

#else
uint32_t BURTC_WAKEUP_COUNT;
uint32_t GPIO_WAKEUP_COUNT;
#endif


#if defined(CRYOTIMER_PRESENT)
/**************************************************************************//**
 * @brief CRYOTIMER interrupt service routine
 *****************************************************************************/
void CRYOTIMER_IRQHandler(void)
{
  // Clear the CRYOTIMER interrupt
  CRYOTIMER_IntClear(CRYOTIMER_IF_PERIOD);

  // Flush instructions to make sure the interrupt is not re-triggered.
  // This is generally required when the peripheral clock is slower than
  // the CPU core clock.
  __DSB();
}

/**************************************************************************//**
 * @brief Note: No need to enable the ULFRCO since it is always on and
 * cannot be shut off under software control.
 *****************************************************************************/
void enableCRYOTimerWakeup(void)
{
  CRYOTIMER_Init_TypeDef init = CRYOTIMER_INIT_DEFAULT;

  // Clear CRYOTIMER_IF PERIOD flag; it will be set upon EM4 wake
  CRYOTIMER_IntClear(CRYOTIMER_IF_PERIOD);

  //
  // Set CRYOTIMER parameters.  Note that disabling the CRYOTIMER is
  // necessary after EM4 wake in order to reset the counter, otherwise
  // the next delay before wake won't be the full 2K periods.
  //
  init.enable = false;
  init.em4Wakeup = true;
  init.osc = cryotimerOscULFRCO;
  init.presc = cryotimerPresc_1;
  init.period = cryotimerPeriod_2k; // nominally 2 seconds

  CRYOTIMER_Init(&init);

  // Interrupt setup
  CRYOTIMER_IntClear(CRYOTIMER_IF_PERIOD);
  CRYOTIMER_IntEnable(CRYOTIMER_IEN_PERIOD);
  NVIC_ClearPendingIRQ(CRYOTIMER_IRQn);
  NVIC_EnableIRQ(CRYOTIMER_IRQn);

  CRYOTIMER_Enable(true);
}
#endif

/**************************************************************************//**
 * @brief Note: Enables EM4 timers. BURTC or CRYO which is device independent
 *****************************************************************************/
void enableTimer()
{
#if defined(BURTC_PRESENT)
  // Enable access to BURTC registers
  RMU_ResetControl(rmuResetBU, rmuResetModeClear);
  BURTC_Enable(true);
#endif
#if defined(CRYOTIMER_PRESENT)
  // Enable CRYOTIMER clock
  CMU_ClockEnable(cmuClock_CRYOTIMER, true);
#endif
}

/**************************************************************************//**
 * @brief Enables wake up source
 *****************************************************************************/
void enableWakeupSource(void)
{
  configEM4();
#if defined(_GPIO_EM4WUEN_MASK)
  enableGPIOWakeup();
#endif
#if defined(BURTC_PRESENT)
  configBURTC();
#endif
#if defined(CRYOTIMER_PRESENT)
  enableCRYOTimerWakeup();
#endif
}

/**************************************************************************//**
 * @brief Displays wake up source information
 *****************************************************************************/
void displayWakeupInfo(void)
{
  // Get the cause of reset and clear the flags
  uint32_t volatile resetCause = RMU->RSTCAUSE;
  RMU_ResetCauseClear();

  enableTimer();

  if (resetCause & RMU_RSTCAUSE_EXTRST)
  {
    // A different reset was triggered. Reset counters.
    displayString(STR_WAKEUP_RESET);
    BURTC_WAKEUP_COUNT = 0;
    GPIO_WAKEUP_COUNT = 0;
  }
#if defined (RMU_RSTCAUSE_EM4RST)
  // Check if reset was caused by a wakeup from EM4
  else if (resetCause & RMU_RSTCAUSE_EM4RST)
  {
#if defined(CRYOTIMER_PRESENT)
    if (CRYOTIMER_IntGet())
    {
      CRYOTIMER_IntClear(CRYOTIMER_IF_PERIOD);
      displayString(STR_WAKEUP_TIMER);
    }
    else
#endif
      // The reset is a wakeup from EM4
      if (GPIO_EM4WUIF)
      {
        GPIO_IntClear(EM4_WU_PB_EN);
        // The wakeup was triggered by GPIO. Increase counter.
        GPIO_WAKEUP_COUNT++;
        displayString(STR_WAKEUP_GPIO);
        displayNumber(GPIO_WAKEUP_COUNT);
      }
      else
      {
        // The reset was triggered by BURTC. Increase counter.
        BURTC_WAKEUP_COUNT++;
        displayString(STR_WAKEUP_TIMER);
        displayNumber(BURTC_WAKEUP_COUNT);
      }
  }
#endif
}



/**************************************************************************//**
 * @brief  EM4 Wakeup function
 *****************************************************************************/
void em4Wakeup(void)
{
  // Enable SysTick interrupt, necessary for Delay()
  if (SysTick_Config(CMU_ClockFreqGet(cmuClock_CORE) / 1000)) while (1) ;

  displayWakeupInfo();

  delay(2000);
  displayOff();

  enableWakeupSource();
  EMU_EnterEM4();
  return;
}

