/******************************************************************************
 * @file main.c
 * @brief Main entry
 * @author Silicon Labs
 * @version  1.11

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#include <stdio.h>
#include "em_device.h"
#include "em_chip.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "em_gpio.h"
#include "em_rmu.h"
#include "bspconfig.h"
#include "config.h"


// Three energy mode examples, includes Prime, energy modes and EM4 wakeup
#define DEMO_ENERGY_MODES 0
#define DEMO_EM4_WAKEUP   1
#define DEMO_PRIME        2

#define NUMBER_OF_DEMOS   3

#define LCD_STR_LEN       7
static const char demoString[NUMBER_OF_DEMOS][LCD_STR_LEN] =
{
  "ENERGY",
  "EM4WU",
  "PRIME"
};

static uint32_t volatile eDemo = DEMO_ENERGY_MODES;
static uint32_t volatile eSelected = false;

// Three demos entry funtions
extern void energyModes(void);
extern void em4Wakeup(void);
extern void prime(void);

void (*runDemo[NUMBER_OF_DEMOS])(void) = {energyModes, em4Wakeup, prime};


/**************************************************************************//**
 * @brief  Enables all the clocks needed for this example
 *****************************************************************************/
void enableClocks(void)
{
  CMU_ClockEnable(cmuClock_CORELE, true);
  CMU_ClockEnable(cmuClock_GPIO, true);
}

/**************************************************************************//**
 * @brief  Configure two buttons and enable interrupt
 *****************************************************************************/
void configGPIO(void)
{
  // Configure PB0 as input
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeInputPull, 1);
  GPIO_IntConfig(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, false, true, true);

  // Configure PB1 as input.
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeInputPullFilter, 1);
  GPIO_IntConfig(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, false, true, true);

  // Enable interrupt.
  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);
}

/**************************************************************************//**
 * @brief Selecting three demos. The Prime, EM4 Wakeup and Energy Modes.
 *****************************************************************************/
void selectDemo(void)
{
  static uint32_t prevDemo = NUMBER_OF_DEMOS - 1;
#if defined(RMU_RSTCAUSE_EM4RST)
  uint32_t resetCause = RMU->RSTCAUSE;
  if ((resetCause & RMU_RSTCAUSE_EM4RST) && ((resetCause & RMU_RSTCAUSE_EXTRST) == 0))
  {
#if defined(_EMU_EM4CTRL_EM4IORETMODE_MASK)
    EMU_UnlatchPinRetention();
#endif
    prevDemo = DEMO_EM4_WAKEUP;
    eSelected = true;
  }
#endif

  while (eSelected == false)
  {
    if (prevDemo != eDemo)
    {
      prevDemo = eDemo;
      displayString(demoString[prevDemo]);
    }
  }
  (*runDemo[prevDemo])();
}

/**************************************************************************//**
 * @brief  Main function
 * Main is called from _program_start, see assembly startup file
 *****************************************************************************/
int main(void)
{
  CHIP_Init();
#if defined(_EMU_DCDCCTRL_MASK)
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif
  enableClocks();
  configGPIO();
  displayInit();
  selectDemo();
  return 0;
}


/**************************************************************************//**
 * @brief GPIO Interrupt handler .
 *  PB1 brows demo, PB0 to run the demo
 *****************************************************************************/
void GPIO_Unified_IRQ(void)
{
  // Get and clear all pending GPIO interrupts
  uint32_t interruptMask = GPIO_IntGet();
  GPIO_IntClear(interruptMask);

  // Act on interrupts
  if (interruptMask & (1 << BSP_GPIO_PB0_PIN))
  {
    eSelected = true;
  }

  if (interruptMask & (1 << BSP_GPIO_PB1_PIN))
  {
    eDemo = (eDemo + 1) % NUMBER_OF_DEMOS;
  }
}

void GPIO_ODD_IRQHandler(void)
{
  GPIO_Unified_IRQ();
}

void GPIO_EVEN_IRQHandler(void)
{
  GPIO_Unified_IRQ();
}

