/******************************************************************************
 * @file 4_energymodes_pg_solution.c
 * @brief Energy Modes example
 * @author Silicon Labs
 * @version  1.23

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

/* emlib includes */
#include "em_chip.h"
#include "em_device.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "em_rtcc.h"

#define LFRCO_FREQUENCY              32768
#define WAKEUP_INTERVAL_MS           5000
/* The -1 compensates for RTCC CNT is CC1_CCV+1 after a compare match event */ 
#define RTC_COUNT_BETWEEN_WAKEUP    (((LFRCO_FREQUENCY * WAKEUP_INTERVAL_MS) / 1000)-1)

RTCC_Init_TypeDef rtccInit;
RTCC_CCChConf_TypeDef rtccInitCompareChannel;

/******************************************************************************
 * @brief RTCC Interrupt Handler. Clears interrupt flag.
 *        The interrupt table is in assembly startup file startup_efm32.s
 *****************************************************************************/
void RTCC_IRQHandler(void)
{
  /* Clear interrupt source */
  RTCC_IntClear(RTCC_IFC_CC1);
}

/******************************************************************************
 * @brief  Main function
 *
 *****************************************************************************/
int main(void)
{ 
  /* Initialize chip */
  CHIP_Init();  

  /* Init DCDC regulator */
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);

  /* Starting LFRCO and waiting until it is stable */
  CMU_ClockSelectSet(cmuClock_LFE, cmuSelect_LFRCO);

  /* Enable the RTCC clock*/
  CMU_ClockEnable(cmuClock_RTCC, true);

  /* Enabling clock to the interface of the low energy modules */
  CMU_ClockEnable(cmuClock_HFLE, true);
  
  /* Set up RTCC init struct */
  rtccInit.enable = false;
  rtccInit.debugRun = false;
  rtccInit.precntWrapOnCCV0 = false;
  rtccInit.cntWrapOnCCV1 = true;
  rtccInit.presc = rtccCntPresc_1;
  rtccInit.prescMode = rtccCntTickPresc;
  rtccInit.enaOSCFailDetect = false;
  rtccInit.cntMode = rtccCntModeNormal;
  rtccInit.disLeapYearCorr = false;

  /* Input RTCC init struct in initialize function */
  RTCC_Init(&rtccInit);
  
  /* Set up RTCC channel output compare init structure */
  rtccInitCompareChannel.chMode = rtccCapComChModeCompare;
  rtccInitCompareChannel.compMatchOutAction = rtccCompMatchOutActionPulse;
  rtccInitCompareChannel.prsSel = rtccPRSCh0;
  rtccInitCompareChannel.inputEdgeSel = rtccInEdgeNone;
  rtccInitCompareChannel.compBase = rtccCompBaseCnt;
  rtccInitCompareChannel.compMask = 0;
  rtccInitCompareChannel.dayCompMode = rtccDayCompareModeMonth;

  /* Input RTCC channel output compare init struct in initialize function */
  RTCC_ChannelInit(1, &rtccInitCompareChannel);

  /* Set RTCC compare value */
  RTCC_ChannelCCVSet(1, RTC_COUNT_BETWEEN_WAKEUP);
  
  /* Enable RTCC interrupt from CC1 */
  RTCC_IntEnable(RTCC_IEN_CC1);
  
  /* Enable RTCC interrupt vector in NVIC */
  NVIC_EnableIRQ(RTCC_IRQn);
  
  /* Enable RTCC */
  RTCC_Enable(true);

   
  /* Loop between EM2 and EM1 */
  while(1){
    /* Enter EM2 and wait for RTCC interrupt */
    EMU_EnterEM2(false);
    
    /* Enter EM1 and wait for RTCC interrupt */
    EMU_EnterEM1();
  }
}
