/*****************************************************************************
 * @file main_wdog_seg_s0.c
 * @brief Watchdog Demo Application
 * @author Silicon Labs
 * @version  1.10

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "em_device.h"
#include "em_wdog.h"
#include "em_cmu.h"
#include "em_rmu.h"
#include "em_emu.h"
#include "em_rtc.h"
#include "em_chip.h"
#include "segmentlcd.h"
#include "ustimer.h"
#include "config.h"
#include "bspconfig.h"

/* GLOBAL VARIABLES */
/* Defining the watchdog initialization data */
WDOG_Init_TypeDef init =
{
  .enable     = true,               /* Start watchdog when init done */
  .debugRun   = false,              /* WDOG not counting during debug halt */
  .em2Run     = true,               /* WDOG counting when in EM2 */
  .em3Run     = true,               /* WDOG counting when in EM3 */
  .em4Block   = false,              /* EM4 can be entered */
  .swoscBlock = false,              /* Do not block disabling LFRCO/LFXO in CMU */
  .lock       = false,              /* Do not lock WDOG configuration (if locked, reset needed to unlock) */
  .clkSel     = wdogClkSelULFRCO,   /* Select 1kHZ WDOG oscillator */
  .perSel     = wdogPeriod_2k,      /* Set the watchdog period to 2049 clock periods (ie ~2 seconds) */
};

volatile int tMode;                 /* Selected test mode */
volatile int startTest;             /* Start test key pressed */
unsigned long resetCause;           /* Reset cause */      
unsigned int i;                     /* Iterration */

/* ISR */
/******************************************************************************
 * @brief  Main function
 *         Main is called from __iar_program_start, see assembly startup file
 *
 * The purpose of this example is to demonstrate some basic functionality of the
 * WDOG, and while doing so write feedback to the LCD to make it easier to follow.
 * After initialization and corresponding enabling of the WDOG, the program
 * enters a loop where feedback is written to the LCD, while the WDOG timer is
 * cleared timely. When this is done some time, the system intentionally enters
 * a stalemate, and stops clearing the WDOG. The WDOG then generates a system
 * reset. The cause of the last reset is detected during initialization, and
 * the system now enters a stalemate without enabling the WDOG. When running
 * debug, the debugger is not able to keep track when the WDOG resets the system.
 * This will cause undefined behavior of the debug session.
 *
 *****************************************************************************/
int main(void)
{
  /* Initialize chip */
  CHIP_Init();

  /* Store the cause of the last reset, and clear the reset cause register */
  resetCause = RMU_ResetCauseGet();
  RMU_ResetCauseClear();

  /* Enabling clock to the interface of the low energy modules (including the Watchdog) */
  CMU_ClockEnable(cmuClock_CORELE, true);

  /* Initialize LCD controller */
  lcdInit();

  /* Check if the watchdog triggered the last reset */
  if (resetCause & RMU_RSTCAUSE_WDOGRST)
  {
    /* Write feedback to lcd */
    SegmentLCD_Write("BITE");
    SegmentLCD_UnsignedHex(resetCause);
    /* Stay in this loop forever */
    while (1) ;
  }

  /* Initialize GPIO */
  gpioSetup();

  /* Handle test mode and wait start key pressing */
  selectState();

  /****************************
  Test mode select handling:
  1. If WDOG test, select ULFRCO as clock source, use timer as delay source (ustimer).
  2. Lock test, select LFXO as clock source, use RTC to wakeup.
  ****************************/
  switch (tMode)
  {
    case T_DOG:
      init.enable = true;
      init.clkSel = wdogClkSelULFRCO;
      init.perSel = wdogPeriod_4k;    /* Set the watchdog period to 4096 clock periods (ie ~4 seconds) */
      USTIMER_Init();
      break;
    case T_LCK:
      init.enable = false;
      init.clkSel = wdogClkSelLFXO;
      init.perSel = wdogPeriod_128k;  /* Set the watchdog period to 128k clock periods (ie ~4 seconds) */
      init.swoscBlock = true;
      init.em3Run = false; 
      break;
  }

  /* Initializing watchdog with choosen settings */
  WDOG_Init(&init);

  /* Test lock function */
  if (tMode == 1)
  {
    /* RTC setup */
    rtcSetup();

    /* Enabling watchdog, since it was not enabled during initialization */
    WDOG_Enable(true);

    /* Locking watchdog register (reset needed to unlock) */
    WDOG_Lock();

    /* Oscillator is locked, and can not be disabled */
    CMU_OscillatorEnable(cmuOsc_LFXO, false, false);

    /* Verifying that the LFXO is still running */
    while (!(CMU->STATUS & CMU_STATUS_LFXOENS)) ;

    /* Register is locked, can not be modified */
    while (WDOG->SYNCBUSY & WDOG_SYNCBUSY_CTRL) ;
    WDOG->CTRL |= WDOG_CTRL_EM3RUN;

    /* Verifying that the EM3RUN bit is not set */
    while (WDOG->CTRL & WDOG_CTRL_EM3RUN) ;
  }

  /* Do something for a while and make sure that the watchdog does not time out */
  SegmentLCD_Write("FEDDING");

  if (tMode == 0)
  {
    for (i = 0; i < 40; i++)
    {
    /* Processing takes place here */
      USTIMER_Delay(50000);
      GPIO_PinOutToggle(LED_PORT, LED_PIN);
      WDOG_Feed();
    }
  }
  else
  {
    for (i = 0; i < 40; i++)
    {
      /* Enter EM2 while the watchdog is still counting */
      EMU_EnterEM2(false);

      /* Processing takes place here */
      GPIO_PinOutToggle(LED_PORT, LED_PIN);
      WDOG_Feed();
    }
  }

  /* Stop feeding the watchdog, and it will trigger a reset */
  SegmentLCD_Write("WAIT");

  /* Enter loop, and wait for wdog reset */
  if (tMode == 0)
  {
    while (1) ;
  }
  else
  {
    while (1)
    {
      EMU_EnterEM2(false);
    }
  }
}
