/*****************************************************************************
 * @file wdog_lcd_setup.c
 * @brief Watchdog Demo Application
 * @author Silicon Labs
 * @version  1.10

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "stdio.h"
#include "em_cmu.h"
#include "em_gpio.h"
#include "config.h"
#include "bspconfig.h"

/* display driver includes */
#ifdef SEG_LCD_S0
#include "segmentlcd.h"
#else
#include "display.h"
#include "textdisplay.h"
#include "retargettextdisplay.h"
#include "displayconfigapp.h"
#endif

/* GLOBAL VARIABLES */
#if defined ( TFT_LCD_S0 ) || defined ( TFT_LCD_S1 )
DISPLAY_Device_t displayDevice;
#endif

extern volatile int tMode;            /* Selected test mode */
extern volatile int startTest;        /* Start test key pressed */
void GPIO_Unified_IRQ(void);

/* ISR */
/******************************************************************************
* @brief GPIO Interrupt handler (Pushbutton for test start).
Button 0 to start the test and button 1 to choose test mode
*****************************************************************************/
void GPIO_ODD_IRQHandler(void)
{
  GPIO_Unified_IRQ();
}

void GPIO_EVEN_IRQHandler(void)
{
  GPIO_Unified_IRQ();
}

void GPIO_Unified_IRQ(void)
{
  /* Get and clear all pending GPIO interrupts */
  uint32_t interruptMask = GPIO_IntGet();
  GPIO_IntClear(interruptMask);

  /* Act on interrupts */
  if (interruptMask & (1 << BSP_GPIO_PB0_PIN))
  {
    /* BTN0: Start test */
    startTest = true;
  }

  if (interruptMask & (1 << BSP_GPIO_PB1_PIN))
  {
    /* BTN1: cycle through tests */
    tMode = (tMode + 1) % T_END;
  }
}

/******************************************************************************
* @brief Setup GPIO interrupt to change test mode.
*****************************************************************************/
void gpioSetup(void)
{
  /* Enable GPIO in CMU */
  CMU_ClockEnable(cmuClock_GPIO, true);

  /* Configure PB0 as input */
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeInputPull, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, BSP_GPIO_PB0_PIN, false, true, true);

  /* Configure PB1 as input */
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeInputPull, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, BSP_GPIO_PB1_PIN, false, true, true);

  /* Enable interrupt */
  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);

  /* Configure LED GPIO */
  GPIO_PinModeSet(LED_PORT, LED_PIN, gpioModePushPull, 0);
}

#if defined SEG_LCD_S0
/******************************************************************************
 * @brief  Selects tMode using the buttons and LCD.
 *
 * @return uint32_t selected eMode.
 *****************************************************************************/
int selectState(void)
{
  startTest = false;
  while (!startTest)
  {
    switch (tMode)
    {
      case T_DOG:
        SegmentLCD_Write("MD WDOG");
        break;
      case T_LCK:
        SegmentLCD_Write("MD LOCK");
        break;
    }
  }
  return tMode;
}

/******************************************************************************
 * @brief  LCD initialization.
 *****************************************************************************/
void lcdInit(void)
{
  /* Initialize LCD controller without boost */
  SegmentLCD_Init(false);
}

#else
/******************************************************************************
 * @brief  Selects tMode using the buttons and LCD.
 *
 * @return uint32_t selected eMode.
 *****************************************************************************/
int selectState(void)
{
  startTest = false;

  /* Display information to guide user */
  printf("\f");
  printf("Push PB1 to\n" 
         "  cycle modes\n\n"
         "Push PB0 to \n"
         "  start test\n\n\n\n");

  /* Loop to select test mode before start key pressed */
  while (!startTest)
  {
    switch (tMode)
    {
  #if defined (T_END) && (T_END == 2)
      case T_DOG:
        printf("\rMD DOG      ");
        break;
      case T_LCK:
        printf("\rMD LOCK     ");
        break;
  #else
      case T_DOG:
        printf("\rMD DOG      ");
        break;
      case T_LCK:
        printf("\rMD LOCK     ");
        break;
      case T_INT:
        printf("\rMD Interrupt");
        break;
      case T_CLR:
        printf("\rMD Clear    ");
        break;
      case T_EVT:
        printf("\rMD Event    ");
        break;
      case T_WRN:
        printf("\rMD Warning  ");
        break;
      case T_WIN:
        printf("\rMD Window   ");
        break;
  #endif
    }
  }

  /* Start test key pressed */
  return tMode;
}

/******************************************************************************
 * @brief  LCD initialization.
 *****************************************************************************/
void lcdInit(void)
{
  /* LCD initialization */
  DISPLAY_Init();
  if (DISPLAY_DeviceGet(0, &displayDevice) != DISPLAY_EMSTATUS_OK)
  {
    /* Unable to get display handle */
    while(1) ;
  }

  /* Retarget stdio to a text display */
  if (RETARGET_TextDisplayInit() != TEXTDISPLAY_EMSTATUS_OK)
  {
    /* Text display initialization failed */
    while(1) ;
  }
}
#endif
