/*****************************************************************************
 * @file main_wdog_tft_s1.c
 * @brief Watchdog Demo Application
 * @author Silicon Labs
 * @version  1.10

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "stdio.h"
#include "em_device.h"
#include "em_chip.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "em_wdog.h"
#include "em_rmu.h"
#include "em_timer.h"
#include "em_letimer.h"
#include "config.h"
#include "bspconfig.h"

/* GLOBAL VARIABLES */
volatile int tMode;            /* Selected test mode */
volatile int startTest;        /* Start test key pressed */
unsigned int wdogPrsCh;        /* WDog PRS channel */
unsigned int period;           /* Wakeup interval */
unsigned long resetCause;      /* Reset cause */      

/******************************************************************************
 * @brief  Main function
 *         Main is called from __iar_program_start, see assembly startup file
 *
 * The purpose of this example is to demonstrate some basic functionality of the
 * WDOG, and while doing so write feedback to the LCD to make it easier to follow.
 * After initialization and corresponding enabling of the WDOG, the program
 * enters a loop where feedback is written to the LCD, while the WDOG timer is
 * cleared timely. When this is done some time, the system intentionally enters
 * a stalemate, and stops clearing the WDOG. The WDOG then generates a system
 * reset. The cause of the last reset is detected during initialization, and
 * the system now enters a stalemate without enabling the WDOG. When running
 * debug, the debugger is not able to keep track when the WDOG resets the system.
 * This will cause undefined behavior of the debug session.
 *
 *****************************************************************************/
int main(void)
{
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_STK_DEFAULT;

  /* Initialize chip */
  CHIP_Init();

  /* Initialize DC-DC */
  EMU_DCDCInit(&dcdcInit);

  /* Store the cause of the last reset, and clear the reset cause register */
  resetCause = RMU_ResetCauseGet();
  RMU_ResetCauseClear();

  /* Enabling clock to the interface of the low energy modules (including the Watchdog) */
  CMU_ClockEnable(cmuClock_CORELE, true);

  /* Initialize LCD controller */
  lcdInit();

  /* Check if the watchdog triggered the last reset */
  if (resetCause & RMU_RSTCAUSE_WDOGRST)
  {
    /* Write feedback to lcd */
    printf("\n\n\nBITE!\n");
    printf("\nSOURCE:");
    printf("0x%x\n", (unsigned int) resetCause);

    /* Stay in this loop forever */
    while (1)
    {
      EMU_EnterEM2(false);
    }
  }

  /* Initialize GPIO */
  gpioSetup();

  /* Loop test mode and wait start key pressing */
  selectState();

  /* Configure watchdog */
  wdogSetup(tMode);

  /* Run the test */
  modeTest(tMode);
}
