/*****************************************************************************
 * @file main_common.c
 * @brief Watchdog Demo Application
 * @author Silicon Labs
 * @version  1.10

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#include "config.h"
#include "em_cmu.h"

#if defined ( SEG_LCD_S0 )
#include "em_rtc.h"
#include "em_emu.h"
#elif defined ( TFT_LCD_S0 )
#include "em_timer.h"
#else
#include "em_prs.h"
#include "em_timer.h"
#include "em_letimer.h"
#endif

/* GLOBAL VARIABLES */
#if defined ( SEG_LCD_S0 )
uint32_t rtcCountBetweenWakeup;       /* Wake up interval */
#endif

#if defined ( TFT_LCD_S0 ) || defined ( TFT_LCD_S1 )
uint32_t timerCountBetweenWakeup;     /* Wake up interval */
#endif

#if defined ( TFT_LCD_S1 )
extern unsigned int period;           /* Wakeup interval */
extern unsigned int wdogPrsCh;        /* WDog PRS channel */
extern volatile int tMode;            /* Selected test mode */
extern volatile int startTest;        /* Start test key pressed */
#endif

#if defined ( SEG_LCD_S0 )
/* Set up RTC init struct for wake up function */
RTC_Init_TypeDef rtcInit =
{
  .debugRun = false,                 /* Disable updating RTC during debug halt */
  .comp0Top = true,                  /* Count until compare. to wrap around */
  .enable   = true,                  /* Start counting when init completed */
};

/******************************************************************************
 * @brief RTC Interrupt Handler. Clears interrupt flag.
 *        The interrupt table is in assembly startup file startup_efm32.s
 *
 * The purpose of the RTC interrupt is to wake the CPU from deep sleep mode EM2
 * at given intervals. This is a way to save energy, while ensuring that the
 * CPU often enough checks if there are any other instructions ready to executed.
 *
 *****************************************************************************/
void RTC_IRQHandler(void)
{
  /* Clear interrupt source */
  RTC_IntClear(RTC_IFC_COMP0);
}

/******************************************************************************
 * @brief  RTC initialization.
 *****************************************************************************/
void rtcSetup(void)
{
  CMU_OscillatorEnable(cmuOsc_LFRCO, true, true);

  /* Enable the RTC clock */
  CMU_ClockEnable(cmuClock_RTC, true);

  /* Enabling clock to the interface of the low energy modules */
  CMU_ClockEnable(cmuClock_CORELE, true);

  /* Input RTC init struct in initialize funciton */
  RTC_Init(&rtcInit);

  /* Set RTC compare value */
  rtcCountBetweenWakeup = ((SystemLFRCOClockGet() * RTC_WAKEUP_INTERVAL_MS) / 1000 - 1);
  RTC_CompareSet(0, rtcCountBetweenWakeup);

  /* Enable RTC interrupt from COMP0 */
  RTC_IntEnable(RTC_IF_COMP0);

  /* Enable RTC interrupt vector in NVIC */
  NVIC_EnableIRQ(RTC_IRQn);

  /* Enable RTC */
  RTC_Enable(true);

  /* Starting LFXO and waiting in EM2 until it is stable */
  CMU_OscillatorEnable(cmuOsc_LFXO, true, false);
  while (!(CMU->STATUS & CMU_STATUS_LFXORDY))
  {
    EMU_EnterEM2(false);
  }
}
#endif

#if defined ( TFT_LCD_S0 ) || defined ( TFT_LCD_S1 )
/* Defining the TIMER initialization data */
TIMER_Init_TypeDef timerInit =
{
  .enable     = true,                 /* Start TIMER when init done */
  .debugRun   = false,                /* Stop counter during debug halt */
  .prescale   = timerPrescale512,     /* 512 prescaling */
  .clkSel     = timerClkSelHFPerClk,  /* Select HFPER clock */ 
  .fallAction = timerInputActionNone, /* No action on falling input edge */
  .riseAction = timerInputActionNone, /* No action on rising input edge */
  .mode       = timerModeUp,          /* Up-counting */ 
  .dmaClrAct  = false,                /* Do not clear DMA requests when DMA channel is active */
  .quadModeX4 = false,                /* Select X2 quadrature decode mode (if used) */  
  .oneShot    = false,                /* Disable one shot */ 
  .sync       = false,                /* Not started/stopped/reloaded by other TIMERs */ 
};

/******************************************************************************
 * @brief TIMER1 Interrupt Handler. Clears interrupt flag.
 *        The interrupt table is in assembly startup file startup_efm32.s
 *
 * The purpose of the TIMER1 interrupt is to wake the CPU from sleep mode EM1
 * at given intervals. This is a way to save energy, while ensuring that the
 * CPU often enough checks if there are any other instructions ready to executed.
 *
 *****************************************************************************/
void TIMER1_IRQHandler(void)
{ 
  /* Clear flag for TIMER1 overflow interrupt */
  TIMER_IntClear(TIMER1, TIMER_IF_OF);
}

/******************************************************************************
 * @brief  TIMER1 initialization.
 *****************************************************************************/
void timerSetup(void)
{
  /* Enable TIMER1 clock */
  CMU_ClockEnable(cmuClock_TIMER1, true);  

  /* Set compare value */
  timerCountBetweenWakeup = ((SystemHFClockGet() / 512 * TIMER_WAKEUP_INTERVAL_MS) / 1000 - 1);
  TIMER_TopSet(TIMER1, timerCountBetweenWakeup);

  /* Enable TIMER1 interrupt from overflow */
  TIMER_IntEnable(TIMER1, TIMER_IF_OF);
  /* Enable TIMER1 interrupt vector in NVIC */
  NVIC_EnableIRQ(TIMER1_IRQn);
  /* Enable TIMER1 */
  TIMER_Init(TIMER1, &timerInit);
}
#endif

#if defined ( TFT_LCD_S1 )
/* Defining the LETIMER initialization data */
const LETIMER_Init_TypeDef letimerInit =
{
  .enable         = true,             /* Start counting when init completed */
  .debugRun       = false,            /* Counter shall not keep running during debug halt */
  .comp0Top       = true,             /* Load COMP0 register into CNT when counter underflows. COMP0 is used as TOP */
  .bufTop         = false,            /* Don't load COMP1 into COMP0 when REP0 reaches 0 */
  .out0Pol        = 0,                /* Idle value for output 0 */
  .out1Pol        = 0,                /* Idle value for output 1 */
  .ufoa0          = letimerUFOAToggle,/* Toggle output on output 0 */
  .ufoa1          = letimerUFOAToggle,/* Toggle output on output 1 */
  .repMode        = letimerRepeatFree /* Count until stopped */
};

/******************************************************************************
 * @brief LETIMER0 Interrupt Handler. Clears interrupt flag.
 *        The interrupt table is in assembly startup file startup_efm32.s
 *
 *****************************************************************************/
void LETIMER0_IRQHandler(void)
{
  /* Clear LETIMER0 underflow interrupt flag */
  LETIMER_IntClear(LETIMER0, LETIMER_IF_UF);
  LETIMER0->CNT = period;
}

/******************************************************************************
 * @brief  LETIMER0 initialization.
   Clock source is ULFRCO, prescale = 0, 1KHz/1 = 1kHz
 *****************************************************************************/
void letimerSetup(unsigned int tMode)
{
  if(tMode == T_EVT)
    period = 3000;                   /* ~ 3 seconds, ULFRCO as LETIMER clock source */
  else
    period = 1000;                   /* ~ 1 second, ULFRCO as LETIMER clock source */

  /* The clock source of LETIMER0 */
  CMU_ClockSelectSet(cmuClock_LFA, cmuSelect_ULFRCO);
  CMU_ClockEnable(cmuClock_LETIMER0, true);

  /* Time out value configuration */
  /* For CLEAR mode test, about 1 second (period = 1000) */
  /* For EVENT mode test, about 3 seconds (period = 3000) */
  LETIMER_CompareSet(LETIMER0, 0, period);

  /* Repetition values must be nonzero so that the outputs
     return switch between idle and active state */
  LETIMER_RepeatSet(LETIMER0, 0, 0x01);
  LETIMER_RepeatSet(LETIMER0, 1, 0x01);

  /* Initializing watchdog with choosen settings */
  LETIMER_Init(LETIMER0, &letimerInit);

  if (tMode == T_CLR)
  {
    /* Enable overflow interrupt */
    LETIMER_IntEnable(LETIMER0, LETIMER_IF_UF);
    NVIC_EnableIRQ(LETIMER0_IRQn);
  }
}

/******************************************************************************
 * @brief  PRS initialization.
 *****************************************************************************/
void prsSetup(void)
{
  CMU_ClockEnable(cmuClock_PRS, true);
  if (tMode == T_CLR)
  {
    PRS_SourceSignalSet(wdogPrsCh, PRS_CH_CTRL_SOURCESEL_WDOG, PRS_CH_CTRL_SIGSEL_LETIMER0CH0, prsEdgeOff);
  }
  else if (tMode == T_EVT)
  {
    PRS_SourceSignalSet(wdogPrsCh, PRS_CH_CTRL_SOURCESEL_WDOG, PRS_CH_CTRL_SIGSEL_LETIMER0CH1, prsEdgeOff);
  }
}
#endif
