/*****************************************************************************
 * @file wdog_mode_test.c
 * @brief Watchdog Demo Application
 * @author Silicon Labs
 * @version  1.10

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#include "stdio.h"
#include "config.h"
#include "em_emu.h"
#include "em_cmu.h"
#include "em_wdog.h"
#include "em_gpio.h"
#include "em_letimer.h"
#include "ustimer.h"

/* GLOBAL VARIABLES */
extern unsigned int period;           /* Wakeup interval */
extern unsigned int wdogPrsCh;        /* WDog PRS channel */
extern volatile int tMode;            /* Selected test mode */
extern volatile int startTest;        /* Start test key pressed */
unsigned int i;

/* Defining the watchdog initialization data */
WDOG_Init_TypeDef init =
{
  .enable     = true,                 /* Start watchdog when init done */
  .debugRun   = false,                /* WDOG not counting during debug halt */
  .em2Run     = true,                 /* WDOG counting when in EM2 */
  .em3Run     = true,                 /* WDOG counting when in EM3 */
  .em4Block   = false,                /* EM4 can be entered */
  .swoscBlock = false,                /* Do not block disabling LFRCO/LFXO in CMU */
  .lock       = false,                /* Do not lock WDOG configuration (if locked, reset needed to unlock) */
  .clkSel     = wdogClkSelULFRCO,     /* Select 1kHZ WDOG oscillator */
  .perSel     = wdogPeriod_8k,        /* Set the watchdog period to 2049 clock periods (ie ~2 seconds) */
};

/* ISR */
/******************************************************************************
 * @brief WDOG Interrupt Handler. Clears interrupt flag.
 *        The interrupt table is in assembly startup file startup_efm32.s
 *
 *****************************************************************************/
void WDOG0_IRQHandler(void)
{
  /* Clear flag for interrupt */
  switch (tMode)
  {
    case T_INT:
      WDOGn_IntClear(DEFAULT_WDOG, WDOG_IEN_TOUT);
      GPIO_PinOutToggle(LED_PORT, LED_PIN);
      break;

    case T_EVT:
      WDOGn_IntClear(DEFAULT_WDOG, WDOG_IEN_PEM0);
      LETIMER_IntClear(LETIMER0, LETIMER_IF_UF);
      LETIMER0->CNT = period;
      GPIO_PinOutToggle(LED_PORT, LED_PIN);
      break;

    case T_WRN:
      WDOGn_IntClear(DEFAULT_WDOG, WDOG_IEN_WARN);
      GPIO_PinOutToggle(LED_PORT, LED_PIN);
      WDOG_Feed();
      break;

    case T_WIN:
      WDOGn_IntClear(DEFAULT_WDOG, WDOG_IEN_WIN);
      GPIO_PinOutToggle(LED_PORT, LED_PIN);
      break;
  }
}

/******************************************************************************
 * @brief  Run watchdog different mode test.
 *****************************************************************************/
void modeTest(unsigned int tMode)
{
  switch (tMode)
  {
    case T_DOG:
      testDog();
      break;
    case T_LCK:
      testLock();
      break;
    case T_INT:
      testInterrupt();
      break;
    case T_CLR:
      testClear();
      break;
    case T_EVT:
      testEvent();
      break;
    case T_WRN:
      testWarning();
      break;
    case T_WIN:
      testWindow();
      break;
  }
}

/******************************************************************************
 * @brief  Watchdog parameter configuration.
 *****************************************************************************/
void wdogSetup(unsigned int tMode)
{
/****************************
  Test mode select handling:
  1. If WDOG test, select ULFRCO as clock source, use timer as delay source (ustimer).
  2. Lock test, select LFXO as clock source, use TIMER1 to wakeup.
  3. Interrupt test, select LFRCO as watchdog clock source.
  4. Other test modes, select ULFRCO as watchdog clock source.
  ****************************/
  switch (tMode)
  {
    case T_DOG:
      init.enable = true;
      init.clkSel = wdogClkSelULFRCO;   /* Select 1kHZ WDOG oscillator */
      init.perSel = wdogPeriod_4k;      /* Set the watchdog period to 4092 clock periods (ie ~4 seconds) */
      break;

    case T_LCK:
      init.enable = false;
      init.swoscBlock = true,           /* Do not block disabling LFRCO/LFXO in CMU */
      init.em3Run = false,              /* WDOG counting when in EM3 */
      init.clkSel = wdogClkSelLFXO;     /* Select 32kHZ WDOG oscillator */
      init.perSel = wdogPeriod_128k;    /* Set the watchdog period to 128k clock periods (ie ~4 seconds) */
      break;

    case T_INT:
      init.enable = true;
      init.clkSel = wdogClkSelLFRCO;    /* Select 32kHZ WDOG oscillator */
      init.perSel = wdogPeriod_32k;     /* Set the watchdog period to 32K clock periods (ie ~1 second) */
      init.resetDisable = true;         /* Disable watchdog reset the device when fired */
      break;

    case T_CLR:
      init.enable = true;
      init.clkSel = wdogClkSelULFRCO;   /* Select 1kHZ WDOG oscillator */
      init.perSel = wdogPeriod_4k;      /* Set the watchdog period to 4096 clock periods (ie ~4 seconds) */
      wdogPrsCh = 0;
      break;

    case T_EVT:
      init.enable = true;
      init.clkSel = wdogClkSelULFRCO;   /* Select 1kHZ WDOG oscillator */
      init.perSel = wdogPeriod_4k;      /* Set the watchdog period to 4096 clock periods (ie ~4 seconds) */
      wdogPrsCh = 1;
      break;

    case T_WRN:
      init.enable = true;
      init.clkSel = wdogClkSelULFRCO;   /* Select 1kHZ WDOG oscillator */
      init.perSel = wdogPeriod_4k;      /* Set the watchdog period to 4096 clock periods (ie ~4 seconds) */
      init.warnSel = wdogWarnTime25pct; /* Set warning to 25 percent (ie ~1 seconds) */
      break;

  case T_WIN:
      init.enable = true;
      init.clkSel = wdogClkSelULFRCO;   /* Select 1kHZ WDOG oscillator */
      init.perSel = wdogPeriod_4k;      /* Set the watchdog period to 4096 clock periods (ie ~4 seconds) */
      init.winSel = wdogIllegalWindowTime50_0pct; /* Set window to 75 percent (ie ~2 seconds) */
      break;
  }
}

/******************************************************************************
 * @brief  Watchdog reset test.

ThE test uses ULFRCO as watchdog clock source. It feeds watchdog every 50mS with usTimer
driver. You could see the LED blinking during feeding process.
 *****************************************************************************/
void testDog(void)
{
  /* Initializing watchdog with choosen settings */
  USTIMER_Init();
  WDOG_Init(&init);

  /* Do something for a while and make sure that the watchdog does not time out */
  printf("\n\nFeeding...");
  for (i = 0; i < 40; i++)
  {
    /* Processing takes place here */
    USTIMER_Delay(50000);
    GPIO_PinOutToggle(LED_PORT, LED_PIN);
    WDOG_Feed();
  }

  /* Stop feeding the watchdog, and it will trigger a reset */
  printf("\rWait...   \n");

  /* Enter loop, and wait for wdog reset */
  while (1) ;
}

/******************************************************************************
 * @brief  Watchdog Lock test.

The test uses LFXO as watchdog clock source. It shows the watchdog lock feature.
 *****************************************************************************/
void testLock(void)
{
  /* TIMER1 setup */
  timerSetup();

  /* Initializing watchdog with choosen settings */
  WDOG_Init(&init);

  /* Enabling watchdog, since it was not enabled during initialization */
  WDOG_Enable(true);

  /* Locking watchdog register (reset needed to unlock) */
  WDOG_Lock();

  /* Oscillator is locked, and can not be disabled */
  /* CMU_OscillatorEnable(cmuOsc_LFXO, false, false) could not be used here,
  because it check the CMU_OSCENCMD_LFXOEN status to go low */
  CMU->OSCENCMD = CMU_OSCENCMD_LFXODIS;

  /* Verifying that the LFXO is still running */
  while (!(CMU->STATUS & CMU_STATUS_LFXOENS)) ;

  /* Register is locked, can not be modified */
  while (DEFAULT_WDOG->SYNCBUSY & WDOG_SYNCBUSY_CTRL) ;
  DEFAULT_WDOG->CTRL |= WDOG_CTRL_EM3RUN;

  /* Verifying that the EM3RUN bit is not set */
  while (DEFAULT_WDOG->CTRL & WDOG_CTRL_EM3RUN) ;

  /* Do something for a while and make sure that the watchdog does not time out */
  printf("\n\nFeeding...");
  for (i = 0; i < 4; i++)
  {
    /* Enter EM1 while the watchdog is still counting */
    EMU_EnterEM1();

    /* Processing takes place here */
    GPIO_PinOutToggle(LED_PORT, LED_PIN);
    WDOG_Feed();
  }

  /* Stop feeding the watchdog, and it will trigger a reset */
  printf("\rWait...   \n");

  /* Enter loop, and wait for wdog reset */
  while (1)
  {
     EMU_EnterEM2(false);
  }
}

/******************************************************************************
 * @brief  Interrupt test.

The test uses LFRCO as watchdog clock source. It tests the watchdog interrupt feature.
When watchdog bites it generates an interrupt instead reset the MCU.
 *****************************************************************************/
void testInterrupt(void)
{
  CMU_OscillatorEnable(cmuOsc_LFRCO, true, true);

  /* Initializing watchdog with choosen settings */
  WDOGn_IntEnable(DEFAULT_WDOG, WDOG_IEN_TOUT);
  NVIC_EnableIRQ(WDOG0_IRQn);
  WDOG_Init(&init);

  /* Do something for a while and make sure that the watchdog does not time out */
  printf("\n\nFeeding...\n");

  for (i = 0; i < 4; i++)
  {
    /* Enter EM2 while the watchdog is still counting */
    EMU_EnterEM2(false);
    /* Wait WDOG interrupt to wakeup the MCU */
  }

  WDOG_Feed();
  WDOGn_IntDisable(DEFAULT_WDOG, WDOG_IEN_TOUT);
  NVIC_DisableIRQ(WDOG0_IRQn);
  WDOG_Enable(false);

  /* Stop feeding the watchdog, and it will trigger a reset */
  printf("\rWAIT...   \n");  
  WDOG0->CTRL &= ~WDOG_CTRL_WDOGRSTDIS;
  WDOG_Enable(true);
  WDOG_Feed();
  /* Enter loop, and wait for wdog reset */
  while (1) ;
}

/******************************************************************************
 * @brief  PRS clear test.

The test uses ULFRCO as watchdog clock source. It tests watchdog PRS clear feature.
The LETIMER0 was used to clear the watchdog regularly (~ 0.5 second).
You could see the LED blinking during clearing process.
 *****************************************************************************/
void testClear(void)
{
  /* LETIMER0 setup */
  letimerSetup(tMode);

  /* PRS setup, PRS channel 0 use LETIMER0 as source */
  prsSetup();

  /* Initializing watchdog with choosen settings */
  WDOG_Init(&init);

  /* PRS rising edge as clear source */
  /* PRS rising edge of channel 0 as WDOG PCH[0] PRS source */
  WDOG0->PCH[0].PRSCTRL = WDOG_PCH_PRSCTRL_PRSSEL_PRSCH0;
  while(DEFAULT_WDOG->SYNCBUSY & WDOG_SYNCBUSY_PCH0_PRSCTRL) ;

  /* Enable PRS clear for watchdog */
  DEFAULT_WDOG->CTRL |= WDOG_CTRL_CLRSRC;

  /* Do something for a while and make sure that the watchdog does not time out */
  printf("\n\nFeeding...");
  for (i = 0; i < 4; i++)
  {
    /* LETIMER0 will wakeup MCU and feed watchdog regularly */
    EMU_EnterEM2(false);

    /* Processing takes place here */
    GPIO_PinOutToggle(LED_PORT, LED_PIN);
  }
  DEFAULT_WDOG->CTRL &= ~WDOG_CTRL_CLRSRC;
  LETIMER_IntDisable(LETIMER0, LETIMER_IF_UF);
  NVIC_DisableIRQ(LETIMER0_IRQn);

  /* Stop feeding the watchdog, and it will trigger a reset */
  printf("\rWait...   \n");

  /* Enter loop, and wait for wdog reset */
  while (1) ;
}

/******************************************************************************
 * @brief  Watchdog PRS event test.

The test uses ULFRCO as watchdog clock source. It feeds watchdog every 1 second with usTimer
driver. The LETIMER0 was the event to be detected, it was configured to 3 seconds interval.
The WDOG interrupt will trigger if the event does not happen.
You could see the LED blinking during feeding process.
 *****************************************************************************/
void testEvent(void)
{
  /* usTimer driver initialization */
  USTIMER_Init();

  /* LETIMER 0 setup */
  letimerSetup(tMode);

  /* PRS setup, PRS channel 1 use LETIMER0 as source */
  prsSetup();

  /* Initializing watchdog with choosen settings */
  WDOG_Init(&init);

  /* PRS rising edge of channel 1 as WDOG PCH[0] PRS source */
  DEFAULT_WDOG->PCH[0].PRSCTRL |= WDOG_PCH_PRSCTRL_PRSSEL_PRSCH1;
  while(DEFAULT_WDOG->SYNCBUSY & WDOG_SYNCBUSY_PCH0_PRSCTRL) ;

  /* Enable watchdog warning interrupt */
  WDOGn_IntEnable(DEFAULT_WDOG, WDOG_IEN_PEM0);
  NVIC_EnableIRQ(WDOG0_IRQn);

  /* Do something for a while and make sure that the watchdog does not time out */
  printf("\n\nFeeding...");

  for (i = 0; i < 5; i++)
  {
    /* Feeding watchdog regularly before LETIMER0 timeout happen */
    USTIMER_Delay(1000000);  /* around 1 second */
    WDOG_Feed();
  }

  WDOGn_IntDisable (DEFAULT_WDOG, WDOG_IEN_PEM0);
  NVIC_DisableIRQ(WDOG0_IRQn);

  /* Stop feeding the watchdog, and it will trigger a reset */
  printf("\rWait...   \n");

  /* Enter loop, and wait for wdog reset */
  while (1) ;
}

/******************************************************************************
 * @brief  Watchdog warning test.

The test uses ULFRCO as watchdog clock source. It tests the watchdog warning feature.
Watchdog generates warning regularly (~1 seconds, 25% of watchdong interval). 
The watchdong was fed when warning was generated. 
You could see the LED blinking during feeding process.
 *****************************************************************************/
void testWarning(void)
{
  /* Initializing watchdog with choosen settings */
  WDOG_Init(&init);

  /* Enable watchdog warning interrupt */
  WDOGn_IntEnable(DEFAULT_WDOG, WDOG_IEN_WARN);
  NVIC_EnableIRQ(WDOG0_IRQn);

  /* Do something for a while and make sure that the watchdog does not time out */
  printf("\n\nFeeding...");
  for (i = 0; i < 4; i++)
  {
    /* Warning interrupt would happen and wakeup CPU */
    /* Feeding was executed in the warning ISR */
    EMU_EnterEM1();
  }

  /* Disable warning interrupt */
  WDOGn_IntDisable(DEFAULT_WDOG, WDOG_IEN_WARN);
  NVIC_DisableIRQ(WDOG0_IRQn);

  /* Stop feeding the watchdog, and it will trigger a reset */
  printf("\rWait...   \n");

  /* Enter loop, and wait for wdog reset */
  while (1) ;
}

/******************************************************************************
 * @brief  Watchdog window test.

The test uses ULFRCO as watchdog clock source. It tests the watchdog window feature.
Watchdog generates window interrupt when it was fed (1 second) before window period time elapsed
(2 seconds, 50% of watchdog interval period).
You could see the LED blinking during feeding process.
 *****************************************************************************/
void testWindow(void)
{
  /* Initializing watchdog with choosen settings */
  USTIMER_Init();

  /* Initializing watchdog with choosen settings */
  WDOG_Init(&init);

  /* Enable watchdog window interrupt */
  WDOGn_IntEnable(DEFAULT_WDOG, WDOG_IEN_WIN);
  NVIC_EnableIRQ(WDOG0_IRQn);

  /* Do something for a while and make sure that the watchdog does not time out */
  printf("\n\nFeeding...");
  for (i = 0; i < 5; i++)
  {
    /* Feed watchdog every 1 second, that was lower than window sectting 2 seconds */
    USTIMER_Delay(1000000);
    WDOG_Feed();
  }

  /* Disable window interrupt */
  WDOG_Feed();
  WDOGn_IntDisable(DEFAULT_WDOG, WDOG_IEN_WIN);
  NVIC_DisableIRQ(WDOG0_IRQn);

  /* Stop feeding the watchdog, and it will trigger a reset */
  printf("\rWait...   \n");

  /* Enter loop, and wait for wdog reset */
  while (1) ;
}
