/******************************************************************************
 * @file oscillator_startup.c
 * @brief Main entry
 * @author Silicon Labs
 * @version  1.00

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2018 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#include "em_device.h"
#include "em_chip.h"
#include "em_cmu.h"

#define HFXO_FREQ 38400000

/// global variables
bool Hfxo_Started;
bool Lfxo_Started;

/// Function prototypes
void startHFXO(void);
void startLFXO(void);

int main(void)
{
  // Chip errata
  CHIP_Init();

  startHFXO();
  startLFXO();

  // Check that both the HFXO and LFXO successfully started up
  Hfxo_Started = (bool) (HFXO0->STATUS & HFXO_STATUS_ENS);
  Lfxo_Started = (bool) (LFXO->STATUS & LFXO_STATUS_ENS);

  while (1) {};
}

/* startHFXO
 * @brief Showcases HFXO configuration and startup
 */
void startHFXO(void) {
  //em_lib defaults for HFXO, this should safely start most oscillators
  CMU_HFXOInit_TypeDef hfxoInit = CMU_HFXOINIT_DEFAULT;

  // Check if device has HFXO configuration information in DEVINFO page
  if (DEVINFO->MODULEINFO & DEVINFO_MODULEINFO_HFXOCALVAL) {
    // Use the DEVINFO page's CTUNE values
    hfxoInit.ctuneXoAna = (DEVINFO->MODXOCAL & DEVINFO_MODXOCAL_HFXOCTUNEXOANA_DEFAULT)
    		>> _DEVINFO_MODXOCAL_HFXOCTUNEXOANA_SHIFT;
    hfxoInit.ctuneXiAna = (DEVINFO->MODXOCAL & DEVINFO_MODXOCAL_HFXOCTUNEXIANA_DEFAULT)
    		>> _DEVINFO_MODXOCAL_HFXOCTUNEXIANA_SHIFT;
  }

  // Configure HFXO settings
  CMU_HFXOInit(&hfxoInit);

  // Select HFXO as the system clock
  CMU_ClockSelectSet(cmuClock_SYSCLK, cmuSelect_HFXO);

  // Set system HFXO frequency
  SystemHFXOClockSet(HFXO_FREQ);

  // Enable HFXO oscillator, and wait for it to be stable
  CMU_OscillatorEnable(cmuOsc_HFXO, true, true);
}

/* startLFXO
 * @brief Showcases LFXO configuration and startup
 */
void startLFXO(void){
  // em_lib defaults for LFXO, this should safely start most oscillators
  CMU_LFXOInit_TypeDef lfxoInit = CMU_LFXOINIT_DEFAULT;

  // Check if device has LFXO configuration information in DEVINFO page
  if (DEVINFO->MODULEINFO & DEVINFO_MODULEINFO_LFXOCALVAL) {
    // Use the DEVINFO page's CTUNE values
    lfxoInit.capTune = (DEVINFO->MODXOCAL & DEVINFO_MODXOCAL_LFXOCAPTUNE_DEFAULT)
    		>> _DEVINFO_MODXOCAL_LFXOCAPTUNE_SHIFT;
  }

  CMU_LFXOInit(&lfxoInit);

  // Enable HFXO oscillator, and wait for it to be stable
  CMU_OscillatorEnable(cmuOsc_LFXO, true, true);
}
